package com.bcxin.tenant.open.rest.apis.controllers;

import com.bcxin.tenant.open.infrastructures.EntityCollection;
import com.bcxin.tenant.open.infrastructures.criterias.CriteriaAbstract;
import com.bcxin.tenant.open.infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.open.jdks.SecurityStationRailMessageRpcProvider;
import com.bcxin.tenant.open.jdks.requests.PoliceIncidentsSearchRequest;
import com.bcxin.tenant.open.jdks.requests.SecurityStationRailMessageSearchRequest;
import com.bcxin.tenant.open.jdks.responses.AttendanceEmpReaderResponse;
import com.bcxin.tenant.open.jdks.responses.PoliceIncidentOverviewResponse;
import com.bcxin.tenant.open.jdks.responses.SecurityStationRailMessageOverviewResponse;
import com.bcxin.tenant.open.rest.apis.utils.ExcelExportUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.http.HttpServletResponse;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;


@Tag(name = "SecurityStationRailMessageController", description = "电子围栏告警信息查询")
@RestController
@RequestMapping("/security-station-rail-messages")
public class SecurityStationRailMessageController extends ControllerAbstract{
    private final SecurityStationRailMessageRpcProvider securityStationRailMessageRpcProvider;

    public SecurityStationRailMessageController(SecurityStationRailMessageRpcProvider securityStationRailMessageRpcProvider) {
        this.securityStationRailMessageRpcProvider = securityStationRailMessageRpcProvider;
    }

    @Operation(
            summary = "业务管理-搜索驻勤点底下的电子围栏告警信息", description = "成功, 则返回列表",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/search")
    public ResponseEntity<EntityCollection<SecurityStationRailMessageOverviewResponse>> post(
            @RequestBody SecurityStationRailMessageSearchRequest request) {
        EntityCollection<SecurityStationRailMessageOverviewResponse> responses =
                this.securityStationRailMessageRpcProvider.search(request);

        return this.ok(responses);
    }

    @Operation(
            summary = "业务管理-搜索驻勤点底下的电子围栏告警信息", description = "成功, 则返回列表",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/export")
    public void export(@RequestBody SecurityStationRailMessageSearchRequest request,
                       HttpServletResponse response) throws IOException {
        if(request.getEventStartDate()==null || request.getEventEndDate()==null) {
            throw new BadTenantException("日期范围不能为空");
        }

        String fileName = String.format("围栏告警-%s", new SimpleDateFormat("yyyy-MM-dd").format(new Date()));
        request.setForExport(true);
        request.setPageSize(1000);
        ExcelExportUtil.export(response, SecurityStationRailMessageOverviewResponse.class, fileName, (excelWriter, sheet) -> {
            int pageIndex = 1;
            EntityCollection<SecurityStationRailMessageOverviewResponse> data = null;
            do {
                request.setPageIndex(pageIndex);
                data = this.securityStationRailMessageRpcProvider.search(request);

                excelWriter.write(data.getData(), sheet);
                pageIndex++;
            } while (data != null && !data.endOfData());
        });
    }
}
