package com.bcxin.tenant.open.rest.apis.controllers;

import com.bcxin.tenant.open.infrastructures.EntityCollection;
import com.bcxin.tenant.open.infrastructures.TenantContext;
import com.bcxin.tenant.open.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.enums.DeskType;
import com.bcxin.tenant.open.infrastructures.utils.StationCodeUtils;
import com.bcxin.tenant.open.infrastructures.valueTypes.StationCodeValueType;
import com.bcxin.tenant.open.jdks.DynamicStationTypeWriterRpcProvider;
import com.bcxin.tenant.open.jdks.SecurityStationReaderRpcProvider;
import com.bcxin.tenant.open.jdks.SecurityStationWriterRpcProvider;
import com.bcxin.tenant.open.jdks.requests.SecurityStationReaderRequest;
import com.bcxin.tenant.open.jdks.requests.SecurityStationShortSearchRequest;
import com.bcxin.tenant.open.jdks.requests.StationStatisticsSearchRequest;
import com.bcxin.tenant.open.jdks.requests.StationTypeSearchRequest;
import com.bcxin.tenant.open.jdks.responses.SecurityStationDetailResponse;
import com.bcxin.tenant.open.jdks.responses.SecurityStationOverviewReaderResponse;
import com.bcxin.tenant.open.jdks.responses.SecurityStationShortResponse;
import com.bcxin.tenant.open.jdks.responses.StationStatisticsResponse;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.Collection;

@RestController
@RequestMapping("/security-stations")
public class SecurityStationController extends ControllerAbstract {
    private final SecurityStationReaderRpcProvider securityStationReaderRpcProvider;
    private final SecurityStationWriterRpcProvider securityStationWriterRpcProvider;
    private final JsonProvider jsonProvider;

    private final DynamicStationTypeWriterRpcProvider dynamicStationTypeWriterRpcProvider;

    public SecurityStationController(SecurityStationReaderRpcProvider securityStationReaderRpcProvider,
                                     SecurityStationWriterRpcProvider securityStationWriterRpcProvider,
                                     JsonProvider jsonProvider,
                                     DynamicStationTypeWriterRpcProvider dynamicStationTypeWriterRpcProvider) {
        this.securityStationReaderRpcProvider = securityStationReaderRpcProvider;
        this.securityStationWriterRpcProvider = securityStationWriterRpcProvider;
        this.jsonProvider = jsonProvider;
        this.dynamicStationTypeWriterRpcProvider = dynamicStationTypeWriterRpcProvider;
    }

    @Operation(summary = "获取驻勤点信息", description = "获取驻勤点信息",
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            },
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功驻勤点信息"),
                    @ApiResponse(responseCode = "404", description = "找不到该驻勤点信息")
            }
    )
    @GetMapping("/{id}")
    public ResponseEntity<SecurityStationDetailResponse> get(@PathVariable String id) {
        SecurityStationDetailResponse data =
                this.securityStationReaderRpcProvider.get(id);
        if (data == null) {
            return this.notFound();
        }

        return this.ok(data);
    }

    @Operation(summary = "高级搜索驻勤信息", description = "按页面显示驻勤信息; 只返回集合数据不返回总行数",
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            },
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回驻勤列表")
            }
    )
    @PostMapping("/search")
    public ResponseEntity<Collection<SecurityStationOverviewReaderResponse>> search(@RequestBody SecurityStationReaderRequest request) {
        Collection<SecurityStationOverviewReaderResponse> responses =
                this.securityStationReaderRpcProvider.search(request);

        return this.ok(responses);
    }

    @Operation(summary = "驻勤点基本搜索", description = "按页面显示驻勤信息; 分页返回驻勤点基本信息",
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            },
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回驻勤列表")
            }
    )
    @PostMapping("/basic-search")
    public ResponseEntity<EntityCollection<SecurityStationShortResponse>> search(@RequestBody SecurityStationShortSearchRequest request) {
        EntityCollection<SecurityStationShortResponse> responses =
                this.securityStationReaderRpcProvider.search(request);

        return this.ok(responses);
    }

    @Operation(summary = "驻勤点的统计信息", description = "根据当前的权限以及指定驻勤Id返回驻勤点类型的统计信息",
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            },
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回驻勤列表")
            }
    )
    @PostMapping("/statistics-search")
    public ResponseEntity<Collection<StationStatisticsResponse>> search(@RequestBody StationStatisticsSearchRequest request) {
        Collection<StationStatisticsResponse> responses = this.securityStationReaderRpcProvider.search(request);

        return this.ok(responses);
    }

    @Operation(summary = "获取全部驻勤点类型", description = "获取全部驻勤点类型; 查询驻勤点的时候直接使用code进行查询",
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            },
            responses = {
                    @ApiResponse(responseCode = "200", description = "获取全部驻勤点类型")
            }
    )
    @RequestMapping(value = "/station-types",method = {RequestMethod.GET,RequestMethod.POST})
    public ResponseEntity<Collection<StationCodeValueType>> postSearchTypes(@RequestBody(required = false) StationTypeSearchRequest request) {
        Collection<StationCodeValueType> valueTypes = new ArrayList<>();

        if (request!=null && !CollectionUtils.isEmpty(request.getDeskTypes()) &&
                request.getDeskTypes().contains(DeskType.TemporarySecurity)) {
            Collection<StationCodeValueType> dynamicValueTypes = this.dynamicStationTypeWriterRpcProvider.getAll();

            valueTypes.addAll(dynamicValueTypes);
        } else {
            var staticValueTypes = StationCodeUtils.getStationCodeValues();
            valueTypes.addAll(staticValueTypes);
        }

        return this.ok(valueTypes);
    }
}
