package com.bcxin.tenant.open.rest.apis.controllers;

import com.alibaba.nacos.shaded.com.google.common.collect.Lists;
import com.bcxin.tenant.open.infrastructures.enums.DeskType;
import com.bcxin.tenant.open.infrastructures.enums.TRTCActionType;
import com.bcxin.tenant.open.infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.open.infrastructures.exceptions.NoFoundTenantException;
import com.bcxin.tenant.open.infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.open.jdks.EmployeeReaderRpcProvider;
import com.bcxin.tenant.open.jdks.RoomRpcProvider;
import com.bcxin.tenant.open.jdks.SocketPublishRpcProvider;
import com.bcxin.tenant.open.jdks.requests.*;
import com.bcxin.tenant.open.jdks.requests.enums.BroadcastMessageType;
import com.bcxin.tenant.open.jdks.responses.*;
import com.bcxin.tenant.open.rest.apis.controllers.requests.CheckRoomRequest;
import com.bcxin.tenant.open.rest.apis.utils.JwtUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Optional;
import java.util.stream.Collectors;

@Tag(name = "RoomController", description = "音视频房间接口")
@RestController
@RequestMapping("/rooms")
public class RoomController extends ControllerAbstract {
    private static final Logger logger = LoggerFactory.getLogger(RoomController.class);
    private final RoomRpcProvider roomRpcProvider;
    private final SocketPublishRpcProvider socketPublishRpcProvider;
    private final EmployeeReaderRpcProvider employeeReaderRpcProvider;

    public RoomController(RoomRpcProvider roomRpcProvider,
                          SocketPublishRpcProvider socketPublishRpcProvider,
                          EmployeeReaderRpcProvider employeeReaderRpcProvider) {
        this.roomRpcProvider = roomRpcProvider;
        this.socketPublishRpcProvider = socketPublishRpcProvider;
        this.employeeReaderRpcProvider = employeeReaderRpcProvider;
    }

    @Operation(
            summary = "创建房间-发起音视频调度及联动值验证", description = "如果验证成功, 则返回房间号, 否则返回不可调度信息",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok."),
                    @ApiResponse(responseCode = "404", description = "找不到保安人员.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping
    public ResponseEntity<CreateRoomWriterRpcResponse> post(@RequestBody CreateRoomWriterRpcRequest request) {

        if(request.getDeskType()==null) {
            request.setDeskType(DeskType.Normal);

            logger.error(
                    "必填DeskType信息为空; 非预期情况, 联系前端人员进行调整:referenceType={};referenceNumber={}",
                    request.getReferenceType(), request.getReferenceNumber()
            );
        }

        CreateRoomWriterRpcResponse response = this.roomRpcProvider.create(request);
        String msg = "联动值充足, 可调度.";
        if (response.getId() == (-1l)) {
            String selectedEmployeeIds =
                    response.getRoomUsers().stream().filter(ii -> !ii.isDispatchable())
                            .map(ix -> ix.getEmployeeId()).collect(Collectors.joining(";"));
            msg = String.format("如下人员(%s)所在企业联动值不足, 无法进行调度", selectedEmployeeIds);
        }

        return this.ok(response, msg);
    }

    @Operation(
            summary = "邀请人员进入房间", description = "返回邀请成功的人员信息",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok."),
                    @ApiResponse(responseCode = "404", description = "找不到保安人员.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"
                    )
            }
    )
    @PostMapping("/{roomId}/users")
    public ResponseEntity<InviteRoomUsersWriterRpcResponse> inviteUsers(@PathVariable final long roomId, @RequestBody InviteRoomUsersWriterRpcRequest request) {
        InviteRoomUsersWriterRpcResponse response = this.roomRpcProvider.inviteUsers(roomId, request);
        String msg = "联动值充足, 可调度.";
        if (response.getRoomId() == (-1l)) {
            String selectedEmployeeIds =
                    response.getRoomUsers().stream().filter(ii -> !ii.isDispatchable())
                            .map(ix -> ix.getEmployeeId())
                            .collect(Collectors.joining(";"));
            msg = String.format("如下人员(%s)所在企业联动值不足, 无法进行调度", selectedEmployeeIds);
        }

        return this.ok(response, msg);
    }

    @Operation(
            summary = "从房间移除人员", description = "从房间移除人员",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok."),
                    @ApiResponse(responseCode = "404", description = "找不到保安人员.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @DeleteMapping("/{roomId}/users/{tencentUserId}")
    public ResponseEntity<InviteRoomUsersWriterRpcResponse> removeUsers(@PathVariable final long roomId, @PathVariable String tencentUserId) {
        this.roomRpcProvider.removeUser(roomId, tencentUserId);

        Collection<BroadcastTrtcMessageContentItem> items = Arrays.asList(BroadcastTrtcMessageContentItem.create(roomId, TRTCActionType.REMOVEUSER, tencentUserId));
        this.socketPublishRpcProvider.dispatch(BroadcastMessageRequest.create(BroadcastMessageType.TRTC,
                BroadcastTrtcMessageContent.create(roomId, TRTCActionType.REMOVEUSER, items, Lists.newArrayList(tencentUserId))));
        return this.ok();
    }

    @Operation(
            summary = "静音/解除静音/开启摄像头/关闭摄像头/发言请求被允许/发言请求被拒绝操作",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/{roomId}/action")
    public ResponseEntity avActionControl(@PathVariable final long roomId, @RequestBody TRTCActionWriterRpcRequest request) {

        //获取信息接收人
        Collection<String> specialTencentIds = roomRpcProvider.getTrtcSendToSessionSpecialIds(roomId, request);
        //发送的数据
        Collection<BroadcastTrtcMessageContentItem> items = specialTencentIds.stream()
                .map(ix -> BroadcastTrtcMessageContentItem.create(roomId, request.getAction(), ix)
                ).collect(Collectors.toList());
        if (specialTencentIds.isEmpty()) {
            throw new NoFoundTenantException("找不到接收消息的人员");
        }
        this.socketPublishRpcProvider.dispatch(
                BroadcastMessageRequest.create(BroadcastMessageType.TRTC,
                        BroadcastTrtcMessageContent.create(roomId, request.getAction(), items, specialTencentIds)));

        //无效的用户
        Collection<String> invalidTencentUserId = Arrays.stream(request.getTencentUserIds())
                .filter(ix -> !specialTencentIds.stream().anyMatch(ii -> ii.equals(ix))).collect(Collectors.toList());
        ResponseEntity response = null;
        if (invalidTencentUserId.isEmpty()) {
            response = this.ok();
        } else {
            response = this.status(HttpStatus.BAD_REQUEST, String.format("用户{%s}不在房间内！", invalidTencentUserId.stream().collect(Collectors.joining(","))));
        }
        return response;
    }

    @Operation(
            summary = "申请开麦/打开摄像头操作",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "accessToken")
            }
    )
    @PostMapping("/{roomId}/actions/{action}")
    public ResponseEntity enableAV(
            @PathVariable final long roomId, @PathVariable String action,
            @RequestHeader("accessToken") String accessToken) {
        String employeeId = JwtUtil.getUserIdFromToken(accessToken);
        if (employeeId == null) {
            throw new UnAuthorizedTenantException("未授权的用户");
        }
        Optional<TRTCActionType> actionTypeOptional = Arrays.stream(TRTCActionType.values())
                .filter(ix -> ix.getName().equalsIgnoreCase(action)).findFirst();
        if (!actionTypeOptional.isPresent()) {
            throw new BadTenantException("参数有误，无法识别操作类型!");
        }

        TRTCActionType selectedAction = actionTypeOptional.get();
        if (selectedAction == TRTCActionType.ANSWER || selectedAction == TRTCActionType.REFUSE) {
            Collection<String> empIds = new ArrayList<>();
            empIds.add(employeeId);
            this.roomRpcProvider.dispatch(RoomReplyRequest.create(roomId, selectedAction, empIds));
        } else {
            EmployeeDetailResponse response = employeeReaderRpcProvider.get(employeeId,DeskType.Normal);
            if (response == null) {
                throw new NoFoundTenantException("找不到当前用户信息");
            }
            Collection<String> specialTencentIds = roomRpcProvider.getTrtcSendToSessionSpecialIds(roomId,
                    TRTCActionWriterRpcRequest.create(actionTypeOptional.get(), new String[]{response.getTencentUserId()}));
            if (specialTencentIds.isEmpty()) {
                throw new NoFoundTenantException("找不到消息接收人员");

            }
            //发送websocket消息
            Collection<BroadcastTrtcMessageContentItem> items =
                    Lists.newArrayList(BroadcastTrtcMessageContentItem.create(roomId, actionTypeOptional.get(), response.getTencentUserId()));

            this.socketPublishRpcProvider.dispatch(BroadcastMessageRequest.create(BroadcastMessageType.TRTC,
                    BroadcastTrtcMessageContent.create(roomId, actionTypeOptional.get(), items, specialTencentIds)));
        }

        return this.ok();
    }

    @Operation(
            summary = "解散房间", description = "解散房间",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok."),
                    @ApiResponse(responseCode = "404", description = "找不到该房间.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/{roomId}/leave")
    public ResponseEntity<CreateRoomWriterRpcResponse> doLeave(@PathVariable long roomId) {
        this.roomRpcProvider.close(roomId);
        return this.ok();
    }

    @Operation(
            summary = "获取房间信息", description = "根据房间Id获取邀请调度的人员列表",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok."),
                    @ApiResponse(responseCode = "404", description = "找不到房间.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.DEFAULT, required = true, name = "id", description = "POST接口返回的房间号")
            }
    )
    @GetMapping("/{id}")
    public ResponseEntity<RoomResponse> get(@PathVariable Long id) {
        RoomResponse response = this.roomRpcProvider.get(id);
        if (response == null) {
            return this.notFound();
        }

        return this.ok(response);
    }


}
