package com.bcxin.tenant.open.rest.apis.controllers;

import com.bcxin.tenant.open.jdks.NotifyMessageRpcProvider;
import com.bcxin.tenant.open.jdks.requests.ConfirmMessageRequest;
import com.bcxin.tenant.open.jdks.responses.NotifyMessageResponse;
import com.bcxin.tenant.open.rest.apis.controllers.requests.CheckRoomRequest;
import com.bcxin.tenant.open.rest.apis.utils.JwtUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import java.util.Collection;

@Tag(name = "RoomController", description = "音视频房间接口")
@RestController
@RequestMapping("/messages")
public class NotifyMessageController extends ControllerAbstract {
    private final NotifyMessageRpcProvider notifyMessageRpcProvider;

    public NotifyMessageController(NotifyMessageRpcProvider notifyMessageRpcProvider) {
        this.notifyMessageRpcProvider = notifyMessageRpcProvider;
    }

    @Operation(
            summary = "doGetInstantMessages", description = "获取当前的即时消息: 待接听的调度+待确认的点名信息",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken/accessToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken/accessToken")
            }
    )
    @PostMapping("/do-get-instants")
    public ResponseEntity<Collection<NotifyMessageResponse>> doGetInstantMessages(
            @RequestHeader("accessToken") String accessToken,
            @RequestBody CheckRoomRequest request) {
        String employeeId = JwtUtil.getUserIdFromToken(accessToken);
        if (!StringUtils.hasLength(employeeId)) {
            return this.status(HttpStatus.UNAUTHORIZED, "未授权用户");
        }

        Collection<NotifyMessageResponse> responses =
                this.notifyMessageRpcProvider.getInstantMessages(request.getEmployeeIds());

        return this.ok(responses);
    }

    @Operation(
            summary = "只要APP打开接听电话的界面就调用该接口来确认已经收到消息", description = "只要APP打开接听电话的界面就调用该接口来确认已经收到消息",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回ok."),
                    @ApiResponse(responseCode = "404", description = "找不到房间.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken/accessToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken/accessToken"),
                    @Parameter(in = ParameterIn.DEFAULT, required = true, name = "id", description = "房间号")
            }
    )
    @PostMapping("/{reference}/do-confirm")
    public ResponseEntity doConfirm(
            @RequestHeader("accessToken") String accessToken,
            @RequestBody ConfirmMessageRequest request,
            @PathVariable String reference
    ) {
        String employeeId = JwtUtil.getUserIdFromToken(accessToken);
        if (!StringUtils.hasLength(employeeId)) {
            return this.status(HttpStatus.UNAUTHORIZED, "未授权用户");
        }

        this.notifyMessageRpcProvider.doConfirm(reference, request.getEmployeeIds());

        return this.ok();
    }
}
