package com.bcxin.tenant.open.rest.apis.controllers;

import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.infrastructures.enums.DispatchDataScopeType;
import com.bcxin.tenant.open.infrastructures.enums.DispatchDataType;
import com.bcxin.tenant.open.infrastructures.exceptions.ArgumentTenantException;
import com.bcxin.tenant.open.infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.open.infrastructures.exceptions.NotSupportTenantException;
import com.bcxin.tenant.open.jdks.*;
import com.bcxin.tenant.open.jdks.requests.FlushNonUsedRequest;
import com.bcxin.tenant.open.jdks.requests.HotCacheRequest;
import com.bcxin.tenant.open.jdks.requests.SyncParameterWrapperRequest;
import com.bcxin.tenant.open.jdks.requests.UpdateGeoEmployeeRequest;
import com.bcxin.tenant.open.rest.apis.controllers.requests.FlushDb2RedisRequest;
import io.swagger.v3.oas.annotations.Operation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicLong;


/**
 * 新增企业的时候; 刷新当前的保安员数量, 驻勤数量
 * 更新企业的时候不进行修改
 *
 * 新增人员的时候更新所在的驻勤点及监管归属、修改的时候不执行
 *
 * 新增驻勤人员的时候更新所在驻勤点
 * 更新驻勤点的时候更新人员所在的驻勤点
 */
@RestController
@RequestMapping("/flush")
public class FlushDbRedisController extends ControllerAbstract {
    private static final Logger logger = LoggerFactory.getLogger(FlushDbRedisController.class);

    private final CompanyWriterRpcProvider companyWriterRpcProvider;
    private final EmployeeWriterRpcProvider employeeWriterRpcProvider;

    private final SecurityStationWriterRpcProvider securityStationWriterRpcProvider;
    private final SecurityStationPersonWriterRpcProvider securityStationPersonWriterRpcProvider;
    private final ExamRoomRpcProvider examRoomRpcProvider;
    private final ExamSiteWriterRpcProvider examSiteRpcProvider;
    private final SystemExamInfoRpcProvider systemExamInfoRpcProvider;
    private final JsonProvider jsonProvider;
    private final RdSyncRpcWriterProvider syncRpcWriterProvider;

    private final CommunityUserWriterRpcProvider communityUserWriterRpcProvider;

    private final DispatchDataScopeRpcProvider dispatchDataScopeRpcProvider;

    private final ProjectWriterRpcProvider projectWriterRpcProvider;

    private final DeviceWriterRpcProvider deviceWriterRpcProvider;
    
    private final AtomicLong employeeCounter = new AtomicLong(0);
    private final HotCacheRpcProvider hotCacheRpcProvider;

    private final ThirdDeviceRpcProvider thirdDeviceRpcProvider;

    public FlushDbRedisController(CompanyWriterRpcProvider companyWriterRpcProvider,
                                  EmployeeWriterRpcProvider employeeWriterRpcProvider,
                                  SecurityStationWriterRpcProvider securityStationWriterRpcProvider,
                                  SecurityStationPersonWriterRpcProvider securityStationPersonWriterRpcProvider,
                                  JsonProvider jsonProvider, RdSyncRpcWriterProvider syncRpcWriterProvider,
                                  ExamRoomRpcProvider examRoomRpcProvider, ExamSiteWriterRpcProvider examSiteRpcProvider,
                                  SystemExamInfoRpcProvider systemExamInfoRpcProvider,
                                  CommunityUserWriterRpcProvider communityUserWriterRpcProvider,
                                  DispatchDataScopeRpcProvider dispatchDataScopeRpcProvider,
                                  ProjectWriterRpcProvider projectWriterRpcProvider,
                                  DeviceWriterRpcProvider deviceWriterRpcProvider,
                                  HotCacheRpcProvider hotCacheRpcProvider,
                                  ThirdDeviceRpcProvider thirdDeviceRpcProvider) {
        this.companyWriterRpcProvider = companyWriterRpcProvider;
        this.employeeWriterRpcProvider = employeeWriterRpcProvider;
        this.securityStationWriterRpcProvider = securityStationWriterRpcProvider;
        this.securityStationPersonWriterRpcProvider = securityStationPersonWriterRpcProvider;
        this.examRoomRpcProvider = examRoomRpcProvider;
        this.examSiteRpcProvider = examSiteRpcProvider;
        this.systemExamInfoRpcProvider = systemExamInfoRpcProvider;
        this.jsonProvider = jsonProvider;
        this.syncRpcWriterProvider = syncRpcWriterProvider;
        this.communityUserWriterRpcProvider = communityUserWriterRpcProvider;
        this.dispatchDataScopeRpcProvider = dispatchDataScopeRpcProvider;
        this.projectWriterRpcProvider = projectWriterRpcProvider;
        this.deviceWriterRpcProvider = deviceWriterRpcProvider;
        this.hotCacheRpcProvider = hotCacheRpcProvider;
        this.thirdDeviceRpcProvider = thirdDeviceRpcProvider;
    }

    @Operation(hidden = true, description = "对外隐藏该API")
    @PostMapping("/request")
    public ResponseEntity db2Redis(@RequestBody FlushDb2RedisRequest request) {

        if (CollectionUtils.isEmpty(request.getContents())) {
            return this.status(HttpStatus.FORBIDDEN, "不支持该操作, 禁用");
        }

        Collection<CompletableFuture<Integer>> futures = new ArrayList<>();


        String content = this.jsonProvider.getJson(request);
        request.getContents().forEach(rq -> {
            if (rq.getDataType()!= FlushDb2RedisRequest.DataType.DispatchDataScope &&
                    request.isReload() && CollectionUtils.isEmpty(rq.getIds())) {
                throw new BadTenantException("当reload设置为true的时候; 必须指定ids");
            }

            switch (rq.getDataType()) {
                case Company -> {
                    CompletableFuture<Integer> companyFuture =
                            CompletableFuture.supplyAsync(() -> {
                                if (request.isReload()) {
                                    this.syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(DispatchDataType.Company, rq.getIds()));
                                    if(!CollectionUtils.isEmpty(rq.getIds())) {
                                        this.hotCacheRpcProvider.refresh(HotCacheRequest.create(HotCacheRequest.CacheCategory.Organization,
                                                rq.getIds()
                                        ));
                                    }

                                    return rq.getIds().size();
                                } else {
                                    this.hotCacheRpcProvider.refresh(HotCacheRequest.create(HotCacheRequest.CacheCategory.Organization, rq.getIds()));
                                    this.hotCacheRpcProvider.refresh(HotCacheRequest.create(HotCacheRequest.CacheCategory.OrgPurse, rq.getIds()));
                                    return this.companyWriterRpcProvider.flush2Redis(rq.getIds(), DispatchDataType.Company);
                                }
                            });

                    futures.add(companyFuture);
                }
                case Station -> {
                    futures.add(CompletableFuture.supplyAsync(() -> {
                        if (request.isReload()) {
                            SyncParameterWrapperRequest wrapperRequest = SyncParameterWrapperRequest.create(DispatchDataType.Station, rq.getIds());
                            wrapperRequest.setAutoUpdateExpired(request.isAutoUpdatedExpired());
                            wrapperRequest.setAdditionalParameter(rq.getIds());
                            this.syncRpcWriterProvider.sync(wrapperRequest);
                            return rq.getIds().size();
                        } else {
                            this.hotCacheRpcProvider.refresh(HotCacheRequest.create(HotCacheRequest.CacheCategory.Station, rq.getIds()));
                            return this.securityStationWriterRpcProvider.flush2Redis(rq.getIds());
                        }
                    }));
                }
                case User -> {
                    futures.add(CompletableFuture.supplyAsync(() -> {
                        if (request.isReload()) {
                            this.syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(DispatchDataType.User, rq.getIds()));
                            return rq.getIds().size();
                        } else {
                            return this.employeeWriterRpcProvider.flush2RedisByTenantUserIds(rq.getIds());
                        }
                    }));
                }
                case Employee, Member -> {
                    try {
                        employeeCounter.addAndGet(rq.getIds().size());
                    } catch (Exception ex) {
                    }

                    futures.add(CompletableFuture.supplyAsync(() -> {
                        if (request.isReload()) {
                            this.syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(
                                    rq.getDataType() == FlushDb2RedisRequest.DataType.Member ? DispatchDataType.Member : DispatchDataType.Employee,
                                    rq.getIds()));
                            if (!CollectionUtils.isEmpty(rq.getIds())) {
                                this.hotCacheRpcProvider.refresh(HotCacheRequest.create(HotCacheRequest.CacheCategory.Employee,
                                        rq.getIds()
                                ));
                            }

                            return rq.getIds().size();
                        } else {
                            this.hotCacheRpcProvider.refresh(HotCacheRequest.create(HotCacheRequest.CacheCategory.Employee, rq.getIds()));
                            return this.employeeWriterRpcProvider.flush2Redis(rq.getIds(), request.isClearRails());
                        }
                    }));
                }
                case ExamRoom -> {
                    futures.add(CompletableFuture.supplyAsync(() -> {
                        if (request.isReload()) {
                            this.syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(DispatchDataType.ExamRoom,rq.getIds()));
                            return rq.getIds().size();
                        } else {
                            return this.examRoomRpcProvider.flush2Redis(rq.getIds());
                        }
                    }));
                }
                case ExamSite -> {
                    futures.add(CompletableFuture.supplyAsync(() -> {
                        if (request.isReload()) {
                            this.syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(DispatchDataType.ExamSite,rq.getIds()));
                            return rq.getIds().size();
                        } else {
                            return this.examSiteRpcProvider.flush2Redis(rq.getIds());
                        }
                    }));
                }
                case SystemExamInfo -> {
                    futures.add(CompletableFuture.supplyAsync(() -> {
                        if (request.isReload()) {
                            this.syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create( DispatchDataType.SystemExamInfo,rq.getIds()));
                            return rq.getIds().size();
                        } else {
                            return this.systemExamInfoRpcProvider.flush2Redis(rq.getIds());
                        }
                    }));
                }
                case StationPersonRelative -> {
                    futures.add(CompletableFuture.supplyAsync(() -> {
                        throw new NotSupportTenantException("暂时屏蔽, 不刷新保安员人员表: station-person-summary");
                        // return this.securityStationPersonWriterRpcProvider.flush2RedisByStationIds(rq.getStationPersonRequest());
                    }));
                }
                case RefreshCompanyDesks -> {
                    CompletableFuture<Integer> companyDeskFuture =
                            CompletableFuture.supplyAsync(() -> {
                                if (request.isReload()) {
                                    this.syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(DispatchDataType.RefreshCompanyDesks, rq.getIds()));
                                    return rq.getIds().size();
                                } else {
                                    throw new NotSupportTenantException("不支持该类型");
                                }
                            });

                    futures.add(companyDeskFuture);
                }
                case RefreshCompanyPoints -> {
                    CompletableFuture<Integer> companyPointFuture =
                            CompletableFuture.supplyAsync(() -> {
                                if (request.isReload()) {
                                    this.syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(DispatchDataType.RefreshCompanyPoints, rq.getIds()));

                                    /**
                                     * 刷新缓存
                                     */
                                    if (!CollectionUtils.isEmpty(rq.getIds())) {
                                        this.hotCacheRpcProvider.refresh(HotCacheRequest.create(HotCacheRequest.CacheCategory.Organization, rq.getIds()));
                                    }
                                    return rq.getIds().size();
                                } else {
                                    throw new NotSupportTenantException("不支持该类型");
                                }
                            });

                    futures.add(companyPointFuture);
                }
                case CommunityUser -> {
                    CompletableFuture<Integer> companyPointFuture =
                            CompletableFuture.supplyAsync(() -> {
                                if (request.isReload()) {
                                    SyncParameterWrapperRequest wrapperRequest = SyncParameterWrapperRequest.create(DispatchDataType.CommunityUser, rq.getIds());
                                    wrapperRequest.setAdditionalParameter(rq.getIds());
                                    wrapperRequest.setAutoUpdateExpired(request.isAutoUpdatedExpired());

                                    this.syncRpcWriterProvider.sync(wrapperRequest);
                                    return rq.getIds().size();
                                } else {
                                    return this.communityUserWriterRpcProvider.flush2Redis(rq.getIds());
                                }
                            });

                    futures.add(companyPointFuture);
                }
                case DispatchDataScope -> {
                    CompletableFuture<Integer> dataScopes =
                            CompletableFuture.supplyAsync(() -> {
                                if (request.isReload()) {
                                    if(request.isAutoUpdatedExpired()) {
                                        SyncParameterWrapperRequest wrapperRequest =
                                                SyncParameterWrapperRequest.create(
                                                        DispatchDataType.ProprietorStations,
                                                        rq.getIds()
                                                );
                                        wrapperRequest.setAdditionalParameter(rq.getIds());
                                        wrapperRequest.setAutoUpdateExpired(request.isAutoUpdatedExpired());

                                        this.syncRpcWriterProvider.sync(wrapperRequest);
                                    }else {
                                        if (CollectionUtils.isEmpty(rq.getIds())) {
                                            throw new ArgumentTenantException("参数不能为空");
                                        }

                                        this.dispatchDataScopeRpcProvider.flush2Redis(rq.getIds(), DispatchDataScopeType.Proprietor);
                                    }

                                    return rq.getIds().size();
                                } else {
                                    Collection<String> attIds = (Collection<String>) rq.getIds();
                                    if (CollectionUtils.isEmpty(attIds)) {
                                        throw new ArgumentTenantException("内保关联表取得的驻勤点Id无效");
                                    }

                                    this.dispatchDataScopeRpcProvider.flush2Redis(attIds,DispatchDataScopeType.Proprietor);

                                    return rq.getIds().size();
                                }
                            });
                    futures.add(dataScopes);
                }
                case SuperviseDeparts -> {
                    CompletableFuture<Integer> superDepartsFuture =
                            CompletableFuture.supplyAsync(() -> {
                                if (request.isReload()) {
                                    SyncParameterWrapperRequest wrapperRequest = SyncParameterWrapperRequest.create(DispatchDataType.SuperviseDeparts, rq.getIds());
                                    wrapperRequest.setAdditionalParameter(rq.getIds());
                                    wrapperRequest.setAutoUpdateExpired(request.isAutoUpdatedExpired());

                                    this.syncRpcWriterProvider.sync(wrapperRequest);
                                    return rq.getIds().size();
                                } else {
                                    this.dispatchDataScopeRpcProvider.flush2Redis(rq.getIds(),DispatchDataScopeType.Supervised);

                                    return rq.getIds().size();
                                }
                            });

                    futures.add(superDepartsFuture);
                }
                case  TemporaryProtectionProject-> {
                    futures.add(CompletableFuture.supplyAsync(() -> {
                        if (request.isReload()) {
                            this.syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(DispatchDataType.TemporaryProtectionProject, rq.getIds()));
                            return rq.getIds().size();
                        } else {
                            return this.projectWriterRpcProvider.flush2Redis(rq.getIds());
                        }
                    }));
                }
                case  TemporaryProtectionProjectGroup-> {
                    futures.add(CompletableFuture.supplyAsync(() -> {
                        if (request.isReload()) {
                            this.syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(DispatchDataType.TemporaryProtectionProjectGroup, rq.getIds()));
                            return rq.getIds().size();
                        } else {
                            return this.projectWriterRpcProvider.flush2Redis(rq.getIds());
                        }
                    }));
                }
                case  TemporaryProtectionProjectMember-> {
                    futures.add(CompletableFuture.supplyAsync(() -> {
                        if (request.isReload()) {
                            this.syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(DispatchDataType.TemporaryProtectionProjectMember, rq.getIds()));
                            return rq.getIds().size();
                        } else {
                            return this.projectWriterRpcProvider.flush2Redis(rq.getIds());
                        }
                    }));
                }

                case Device -> {
                    futures.add(CompletableFuture.supplyAsync(() -> {
                        if (request.isReload()) {
                            //this.syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(DispatchDataType.Device, rq.getIds()));
                            Collection<String> deviceNos = this.deviceWriterRpcProvider.getDeviceNosByIds(rq.getIds());
                            this.thirdDeviceRpcProvider.refresh(deviceNos);
                            return rq.getIds().size();
                        } else {
                            return this.deviceWriterRpcProvider.flush2Redis(rq.getIds());
                        }
                    }));
                }
                case TemporaryProtectionProjectPersonType -> {
                    futures.add(CompletableFuture.supplyAsync(() -> {
                        this.syncRpcWriterProvider.sync(SyncParameterWrapperRequest.create(DispatchDataType.TemporaryProtectionProjectPersonType, rq.getIds()));
                        return rq.getIds().size();
                    }));
                }
                case EventOrganizerJoinProject -> {
                    futures.add(CompletableFuture.supplyAsync(() -> {
                        SyncParameterWrapperRequest syncRq = SyncParameterWrapperRequest.create(DispatchDataType.EventOrganizerJoinProject, rq.getIds());
                        syncRq.setAdditionalParameter(rq.getIds());

                        this.syncRpcWriterProvider.sync(syncRq);
                        return rq.getIds().size();
                    }));
                }
                case EventOrganizerStationManager -> {
                    futures.add(CompletableFuture.supplyAsync(() -> {
                        SyncParameterWrapperRequest syncRq = SyncParameterWrapperRequest.create(DispatchDataType.EventOrganizerStationManager, rq.getIds());
                        syncRq.setAdditionalParameter(rq.getIds());

                        this.syncRpcWriterProvider.sync(syncRq);
                        return rq.getIds().size();
                    }));
                }
                case EventOrganizerLimitedResource->{
                    futures.add(CompletableFuture.supplyAsync(() -> {
                        SyncParameterWrapperRequest syncRq = SyncParameterWrapperRequest.create(DispatchDataType.EventOrganizerLimitedResource, rq.getIds());
                        syncRq.setAdditionalParameter(rq.getIds());

                        this.syncRpcWriterProvider.sync(syncRq);
                        return rq.getIds().size();
                    }));
                }
                case DispatchDataSource -> {
                    futures.add(CompletableFuture.supplyAsync(() -> {
                        SyncParameterWrapperRequest syncRq = SyncParameterWrapperRequest.create(DispatchDataType.DispatchDataSource, rq.getIds());
                        syncRq.setAdditionalParameter(rq.getIds());

                        this.syncRpcWriterProvider.sync(syncRq);
                        return rq.getIds().size();
                    }));
                }
                case Workstation -> {
                    futures.add(CompletableFuture.supplyAsync(() -> {

                        /*
                        int addParameterValue = (int) rq.getAdditionalParameter();
                        if(addParameterValue!=1 && addParameterValue!=2) {
                            throw new BadTenantException(String.format("不允许的数据:%s", addParameterValue));
                        }
                         */

                        SyncParameterWrapperRequest syncRq = SyncParameterWrapperRequest.create(DispatchDataType.Workstation, rq.getIds());
                        syncRq.setAdditionalParameter(rq.getAdditionalParameter());

                        this.syncRpcWriterProvider.sync(syncRq);
                        return rq.getIds().size();
                    }));
                }
                case WorkStationUsers -> {
                    futures.add(CompletableFuture.supplyAsync(() -> {
                        SyncParameterWrapperRequest syncRq =
                                SyncParameterWrapperRequest.create(DispatchDataType.WorkStationUsers, rq.getIds());

                        this.syncRpcWriterProvider.sync(syncRq);

                        this.employeeWriterRpcProvider.flush2Redis(rq.getIds(), false);
                        return rq.getIds().size();
                    }));
                }
                case WorkStationUsersBySecurityStation -> {
                    futures.add(CompletableFuture.supplyAsync(() -> {
                        SyncParameterWrapperRequest syncRq = SyncParameterWrapperRequest.create(DispatchDataType.WorkStationUsersBySecurityStation, rq.getIds());

                        this.syncRpcWriterProvider.sync(syncRq);

                        this.employeeWriterRpcProvider.flush2Redis(rq.getIds(), false);
                        return rq.getIds().size();
                    }));
                }
            }
        });

        Double affectedCount = futures.stream().mapToDouble(ix -> {
            try {
                int value = ix.get().intValue();
                return value;
            } catch (Exception ex) {
                ex.printStackTrace();
                throw new BadTenantException("执行异常" + ex.getMessage(), ex);
            }
        }).sum();

        logger.error("[当前人员刷新总数:{}]请求数据(类型={})-数据-:{}",
                employeeCounter.get(),
                content,
                affectedCount);
        return this.ok(affectedCount, "同步完毕OK-v2!");
    }

    @PutMapping
    public ResponseEntity putEmployeeData(@RequestBody UpdateGeoEmployeeRequest request) {
        this.employeeWriterRpcProvider.update(request);

        return this.ok();
    }

    @PostMapping("/clean-non-used-data")
    public ResponseEntity cleanNonUsedData(@RequestBody FlushNonUsedRequest rq) {
        int affectedCount = this.employeeWriterRpcProvider.clearNonUsedEmployees(rq.getIds());
        return this.ok(affectedCount);
    }

    @Operation(hidden = true, description = "对外隐藏该API")
    @PostMapping("/get-config")
    public ResponseEntity getByContentRequest(@RequestBody FlushDb2RedisRequest.ContentRequest contentRequest) {
        if(contentRequest.getDataType()==null) {
            throw new NotSupportTenantException("不支持该类型");
        }

        switch (contentRequest.getDataType()) {
            case DispatchDataScope -> {
                Collection<String> content = this.dispatchDataScopeRpcProvider.getJsonResult(contentRequest.getIds());

                return this.ok(content);
            }
            case CommunityUser -> {
                Collection<String> content =
                        this.communityUserWriterRpcProvider.getJsonResult(contentRequest.getIds());

                return this.ok(content);
            }
            default -> {
                throw new NotSupportTenantException("不支持该类型");
            }
        }
    }
}
