package com.bcxin.tenant.open.rest.apis.controllers;
import com.bcxin.tenant.open.infrastructures.EntityCollection;
import com.bcxin.tenant.open.infrastructures.enums.DeskType;
import com.bcxin.tenant.open.infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.open.infrastructures.utils.BusinessUtil;
import com.bcxin.tenant.open.jdks.DispatchDataSourceRpcProvider;
import com.bcxin.tenant.open.jdks.EmployeeReaderRpcProvider;
import com.bcxin.tenant.open.jdks.EmployeeWriterRpcProvider;
import com.bcxin.tenant.open.jdks.requests.EmployeeAdvanceRequest;
import com.bcxin.tenant.open.jdks.requests.EmployeeSearchRequest;
import com.bcxin.tenant.open.jdks.requests.EmployeeDutyStatusRequest;
import com.bcxin.tenant.open.jdks.requests.EmployeeTrackRequest;
import com.bcxin.tenant.open.jdks.responses.*;
import com.bcxin.tenant.open.rest.apis.caches.CacheKeyManage;
import com.bcxin.tenant.open.rest.apis.components.HotCacheProvider;
import com.bcxin.tenant.open.rest.apis.controllers.requests.EmployeeLocationSearchRequest;
import com.bcxin.tenant.open.rest.apis.utils.ExcelExportUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.ResponseEntity;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.util.Collection;
import java.util.Date;

@Tag(name = "EmployeeController", description = "保安人员查询")
@RestController
@RequestMapping("/employees")
public class EmployeeController extends ControllerAbstract {
    private static final Logger logger = LoggerFactory.getLogger(EmployeeController.class);
    private final EmployeeReaderRpcProvider employeeReaderRpcProvider;
    private final EmployeeWriterRpcProvider employeeWriterRpcProvider;

    private final HotCacheProvider hotCacheProvider;

    private final DispatchDataSourceRpcProvider dispatchDataSourceRpcProvider;

    public EmployeeController(EmployeeReaderRpcProvider employeeReaderRpcProvider,
                              EmployeeWriterRpcProvider employeeWriterRpcProvider,
                              HotCacheProvider hotCacheProvider,
                              DispatchDataSourceRpcProvider dispatchDataSourceRpcProvider) {
        this.employeeReaderRpcProvider = employeeReaderRpcProvider;
        this.employeeWriterRpcProvider = employeeWriterRpcProvider;
        this.hotCacheProvider = hotCacheProvider;
        this.dispatchDataSourceRpcProvider = dispatchDataSourceRpcProvider;
    }

    @Operation(
            summary = "获取保安员详情信息", description = "获取保安员详情信息",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回保安员信息."),
                    @ApiResponse(responseCode = "404", description = "找不到保安人员.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.DEFAULT, required = true, name = "id", description = "职员的id"),
                    @Parameter(in = ParameterIn.DEFAULT, required = true, name = "desk-type", description = "当前的调度台类型;Proprietor=内保调度台; Normal=正常的指挥调度台;TemporarySecurity=临保业务")
            }
    )
    @GetMapping("/{id}")
    public ResponseEntity<EmployeeDetailResponse> get(@PathVariable String id,@RequestParam(value = "desk-type",defaultValue = "Normal") String deskTypeLabel) {
        DeskType deskType = DeskType.Normal;
        if (StringUtils.hasLength(deskTypeLabel)) {
            if ("Proprietor".equalsIgnoreCase(deskTypeLabel)) {
                deskType = DeskType.Proprietor;
            } else if ("TemporarySecurity".equalsIgnoreCase(deskTypeLabel)) {
                deskType = DeskType.TemporarySecurity;
            }
        }

        EmployeeDetailResponse data =
                this.employeeReaderRpcProvider.get(id, deskType);

        if (data == null) {
            return this.notFound();
        }

        try {
            if (StringUtils.hasLength(data.getThirdParty())) {
                DispatchDataSourceResponse dispatchDataSourceResponse =
                        this.hotCacheProvider.get(CacheKeyManage.getDispatchDataSourceKey(data.getThirdParty()), DispatchDataSourceResponse.class);
                if (dispatchDataSourceResponse == null) {
                    dispatchDataSourceResponse = this.dispatchDataSourceRpcProvider.getById(data.getThirdParty());
                    if (dispatchDataSourceResponse != null) {
                        this.hotCacheProvider.put(CacheKeyManage.getDispatchDataSourceKey(data.getThirdParty()), 30 * 60, dispatchDataSourceResponse,false);
                    }
                }

                if (dispatchDataSourceResponse != null) {
                    data.assignThirdPartyName(dispatchDataSourceResponse.getName());
                }
            }
        }
        catch (Exception ex) {
            data.assignThirdPartyName(String.format("%s-%s", data.getThirdParty(), ex.getMessage()));

            logger.error("获取数据来源异常:thirdParty={};institutional={}",data.getThirdParty(),data.getInstitutional(),ex);
        }

        return this.ok(data);
    }

    @Operation(summary = "高级搜索保安人员", description = "按页面显示保安人员信息; 只返回集合数据不返回总行数",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回保安员列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/search")
    public ResponseEntity<Collection<EmployeeOverviewResponse>> post(@RequestBody EmployeeSearchRequest request) {
        Collection<EmployeeOverviewResponse> data =
                this.employeeReaderRpcProvider.search(request);

        return this.ok(data);
    }

    @Operation(summary = "获取指定人员指定时间内的坐标轨迹", description = "获取指定人员指定时间内的坐标轨迹",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回某人的坐标列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/{employeeId}/locations")
    public ResponseEntity<Collection<EmployeeTrackResponse>> post(@PathVariable String employeeId, @RequestBody EmployeeLocationSearchRequest request) {
        if (request.getBeginDate() == null) {
            throw new BadTenantException("搜索轨迹的开始时间不能为空");
        }

        if (request.getEndDate() == null) {
            throw new BadTenantException("搜索轨迹的结束时间不能为空");
        }

        if (request.getEndDate().before(request.getBeginDate())) {
            throw new BadTenantException("搜索轨迹的结束时间不能早于开始时间");
        }

        Duration duration =
                Duration.between(request.getBeginDate().toInstant(),
                        request.getEndDate().toInstant());
        if (duration.toDays() > 30) {
            throw new BadTenantException("只提供30天的轨迹查询功能");
        }

        Collection<EmployeeTrackResponse> data =
                this.employeeWriterRpcProvider
                        .getTrackLocations(new EmployeeTrackRequest(
                                employeeId,
                                request.getBeginDate(),
                                request.getEndDate())
                        );

        return this.ok(data);
    }

    @Operation(summary = "高级搜索保安人员", description = "高级搜索保安人员-业务管理系统搜索",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回保安员列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/advance-search")
    public ResponseEntity<EntityCollection<EmployeeAdvanceResponse>> post(@RequestBody EmployeeAdvanceRequest request) {
        EntityCollection<EmployeeAdvanceResponse> data =
                this.employeeReaderRpcProvider.search(request);

        return this.ok(data);
    }
    @Operation(summary = "导出-业务管理-未激活人员", description = "导出-根据过滤条件查询未激活人员列表 ",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "未激活人员列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/export")
    public void export(@RequestBody EmployeeAdvanceRequest request,
                       HttpServletResponse response) throws IOException {
        String fileName = String.format("未激活人员-%s", new SimpleDateFormat("yyyy-MM-dd").format(new Date()));
        request.setForExport(true);
        request.setPageSize(1000);
        ExcelExportUtil.export(response, EmployeeAdvanceResponse.class, fileName, (excelWriter, sheet) -> {
            int pageIndex = 1;
            EntityCollection<EmployeeAdvanceResponse> data = null;
            do {
                request.setPageIndex(pageIndex);
                data = this.employeeReaderRpcProvider.search(request);

                excelWriter.write(data.getData(), sheet);
                pageIndex++;
            } while (data != null && !data.endOfData());
        });
    }

    @Operation(summary = "查询驻勤点人员数量", description = "根据驻勤点id查询驻勤人员在离岗数量",
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "accessToken"),
                    @Parameter(in = ParameterIn.PATH, required = true, name = "stationId", description = "驻勤点id")
            },
            responses = @ApiResponse(responseCode = "200", description = "成功返回保安员数")
    )
    @GetMapping ("/duty-status/{stationId}/count")
    public ResponseEntity<EmployeeCountResponse> getCount(@PathVariable String stationId){
        EmployeeCountResponse response = this.employeeReaderRpcProvider.getEmployeeNumberByStationId(stationId);
        return this.ok(response);
    }

    @Operation(summary = "查询驻勤点在离职人员", description = "根据驻勤点id和在离职状态查询驻勤人员",
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "accessToken")
            },
            requestBody = @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = @ApiResponse(responseCode = "200", description = "成功返回保安员信息")
    )
    @PostMapping("/duty-status/list")
    public ResponseEntity<Collection<EmployeeDutyStatusResponse>> post(@RequestBody EmployeeDutyStatusRequest request){
        Collection<EmployeeDutyStatusResponse> response =
                this.employeeReaderRpcProvider.getEmployeesByStationIdAndDutyStatus(request);
        return this.ok(response);
    }
}