package com.bcxin.tenant.open.rest.apis.controllers;

import com.bcxin.tenant.open.infrastructures.EntityCollection;
import com.bcxin.tenant.open.infrastructures.enums.DispatchReasonType;
import com.bcxin.tenant.open.jdks.DispatchLogRpcProvider;
import com.bcxin.tenant.open.jdks.requests.DispatchLogDetailSearchRequest;
import com.bcxin.tenant.open.jdks.requests.DispatchLogSearchRequest;
import com.bcxin.tenant.open.jdks.requests.MyDispatchLogRecordRequest;
import com.bcxin.tenant.open.jdks.requests.SponsorDispatchLogSearchRequest;
import com.bcxin.tenant.open.jdks.responses.DispatchLogDetailSearchResponse;
import com.bcxin.tenant.open.jdks.responses.DispatchLogSearchResponse;
import com.bcxin.tenant.open.jdks.responses.MyDispatchLogRecordResponse;
import com.bcxin.tenant.open.jdks.responses.SponsorDispatchLogSearchResponse;
import com.bcxin.tenant.open.rest.apis.utils.ExcelExportUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.http.HttpServletResponse;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;

@Tag(name = "DispatchLogController", description = "通讯日志查询")
@RestController
@RequestMapping("/dispatch-logs")
public class DispatchLogController extends ControllerAbstract {
    private final DispatchLogRpcProvider dispatchLogRpcProvider;

    public DispatchLogController(DispatchLogRpcProvider dispatchLogRpcProvider) {
        this.dispatchLogRpcProvider = dispatchLogRpcProvider;
    }

    @Operation(summary = "搜索通讯日志", description = "业务管理-根据当前用户搜索通讯日志",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回通讯日志列表及分页信息")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping
    public ResponseEntity<EntityCollection<DispatchLogSearchResponse>> search(@RequestBody DispatchLogSearchRequest request) {
        EntityCollection<DispatchLogSearchResponse> data = this.dispatchLogRpcProvider.search(request);

        return this.ok(data);
    }

    @Operation(summary = "针对指定调度的企业详细信息", description = "针对指定调度的企业详细信息",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回通讯日志列表及分页信息")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.PATH, required = true, name = "logId",
                            description = "调度日志Id"),
                    @Parameter(in = ParameterIn.PATH, required = true, name = "organizationId",
                            description = "被调度的企业Id-来自/dispatch-logs中返回的calledOrganizationId")
            }
    )
    @PostMapping("/{logId}/detail-logs/{organizationId}")
    public ResponseEntity<EntityCollection<DispatchLogDetailSearchResponse>> searchDetail(@PathVariable String logId,
                                                                                          @PathVariable String organizationId) {
        DispatchLogDetailSearchRequest request = new DispatchLogDetailSearchRequest();
        request.setLogId(logId);
        request.setCalledOrganizationId(organizationId);

        EntityCollection<DispatchLogDetailSearchResponse> data = this.dispatchLogRpcProvider.search(request);

        return this.ok(data);
    }

    @Operation(summary = "根据类型通过Id查询调度记录", description = "根据类型通过Id查询调度记录（Sos界面中referenceType设置为: Sos）",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回通讯日志列表及分页信息")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.PATH, required = true, name = "logId",
                            description = "调度日志Id"),
                    @Parameter(in = ParameterIn.PATH, required = true, name = "organizationId",
                            description = "被调度的企业Id-来自/dispatch-logs中返回的calledOrganizationId")
            }
    )
    @GetMapping("/{referenceType}/{referenceNumber}")
    public ResponseEntity<EntityCollection<DispatchLogDetailSearchResponse>> getDetailByRoomId(@PathVariable DispatchReasonType referenceType,
                                                                                               @PathVariable String referenceNumber) {
        DispatchLogDetailSearchRequest request = new DispatchLogDetailSearchRequest();
        request.setReferenceType(referenceType);
        request.setReferenceNumber(referenceNumber);

        EntityCollection<DispatchLogDetailSearchResponse> data = this.dispatchLogRpcProvider.search(request);

        return this.ok(data);
    }

    @Operation(summary = "search", description = "查询当前用户的调度记录",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回Excel")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/search")
    public ResponseEntity<Collection<MyDispatchLogRecordResponse>> search(@RequestBody MyDispatchLogRecordRequest request) {
        Collection<MyDispatchLogRecordResponse> data = this.dispatchLogRpcProvider.search(request);

        return this.ok(data);
    }

    @Operation(summary = "导出通讯日志", description = "根据当前用户搜索通讯日志",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回Excel")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/export")
    public void export(@RequestBody DispatchLogSearchRequest request,
                       HttpServletResponse response) throws IOException {

        String fileName =
                String.format("调度记录-%s", new SimpleDateFormat("yyyy-MM-dd").format(new Date()));
        request.setForExport(true);
        request.setPageSize(1000);
        ExcelExportUtil.export(response, DispatchLogSearchResponse.class, fileName, (excelWriter, sheet) -> {
            int pageIndex = 1;
            EntityCollection<DispatchLogSearchResponse> data = null;
            do {
                request.setPageIndex(pageIndex);
                data = this.dispatchLogRpcProvider.search(request);

                excelWriter.write(data.getData(), sheet);
                pageIndex++;
            } while (data != null && !data.endOfData());
        });
    }


    @Operation(summary = "sponsor-search", description = "查询当前用户发起的调度信息",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回Excel")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.QUERY, required = true, name = "permission",
                            description = "固定传:advance 表示根据用户当前的权限而非依赖于调度台来实现权限管理; 不支持监管方")
            }
    )
    @PostMapping("/sponsor-search")
    public ResponseEntity<Collection<SponsorDispatchLogSearchResponse>> search(@RequestBody SponsorDispatchLogSearchRequest request) {
        Collection<SponsorDispatchLogSearchResponse> data = this.dispatchLogRpcProvider.search(request);

        return this.ok(data);
    }

    @Operation(summary = "/{id}/details", description = "查询当前用户发起的调度信息",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回Excel")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.QUERY, required = true, name = "permission",
                            description = "固定传:advance 表示根据用户当前的权限而非依赖于调度台来实现权限管理; 不支持监管方")
            }
    )
    @PostMapping("/{id}/details")
    public ResponseEntity<Collection<DispatchLogDetailSearchResponse>> search(@PathVariable String id) {
        Collection<DispatchLogDetailSearchResponse> data = this.dispatchLogRpcProvider.getById(id);

        return this.ok(data);
    }
}