package com.bcxin.tenant.open.rest.apis.controllers;

import com.bcxin.tenant.open.infrastructures.exceptions.BadTenantException;
import com.bcxin.tenant.open.jdks.DeviceReaderRpcProvider;
import com.bcxin.tenant.open.jdks.DeviceWriterRpcProvider;
import com.bcxin.tenant.open.jdks.requests.BatchDeviceLocationRequest;
import com.bcxin.tenant.open.jdks.requests.DeviceLocationSearchRequest;
import com.bcxin.tenant.open.jdks.requests.DeviceSearchRequest;
import com.bcxin.tenant.open.jdks.requests.DeviceTrackRequest;
import com.bcxin.tenant.open.jdks.responses.DeviceResponse;
import com.bcxin.tenant.open.jdks.responses.DeviceTrackResponse;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.time.Duration;
import java.util.Collection;

@Tag(name = "DeviceController", description = "智能设备信息查询")
@RestController
@RequestMapping("/devices")
public class DeviceController extends ControllerAbstract{

    private final DeviceReaderRpcProvider deviceReaderRpcProvider;
    private final DeviceWriterRpcProvider deviceWriterRpcProvider;

    public DeviceController(DeviceReaderRpcProvider deviceReaderRpcProvider,
                            DeviceWriterRpcProvider deviceWriterRpcProvider) {
        this.deviceReaderRpcProvider = deviceReaderRpcProvider;
        this.deviceWriterRpcProvider = deviceWriterRpcProvider;
    }

    @Operation(
            summary = "获取智能设备详情信息", description = "获取智能设备详情信息",
            responses = {
                    @ApiResponse(responseCode = "200", description = "成功返回保安员信息."),
                    @ApiResponse(responseCode = "404", description = "找不到智能设备.")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken"),
                    @Parameter(in = ParameterIn.DEFAULT, required = true, name = "id", description = "职员的id")
            }
    )
    @GetMapping("/{id}")
    public ResponseEntity<DeviceResponse> get(@PathVariable String id) {
        DeviceResponse data =
                this.deviceReaderRpcProvider.get(id);
        if (data == null) {
            return this.notFound();
        }

        return this.ok(data);
    }

    @Operation(summary = "搜索智能设备信息", description = "按页面显示智能设备信息; 只返回集合数据不返回总行数",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回保安员列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/search")
    public ResponseEntity<Collection<DeviceResponse>> post(@RequestBody DeviceSearchRequest request) {
        Collection<DeviceResponse> data =
                this.deviceReaderRpcProvider.search(request);

        return this.ok(data);
    }

    @Operation(summary = "获取指定设备指定时间内的坐标轨迹", description = "获取指定设备指定时间内的坐标轨迹",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "返回设备的坐标列表")
            },
            parameters = {
                    @Parameter(in = ParameterIn.HEADER, required = true, name = "dispatchToken",
                            description = "来自认证接口产生的调度系统的/identity/auto-login产生的dispatchToken")
            }
    )
    @PostMapping("/{id}/locations")
    public ResponseEntity<Collection<DeviceTrackResponse>> post(@PathVariable String id, @RequestBody DeviceLocationSearchRequest request) {
        if (request.getBeginDate() == null) {
            throw new BadTenantException("搜索轨迹的开始时间不能为空");
        }

        if (request.getEndDate() == null) {
            throw new BadTenantException("搜索轨迹的结束时间不能为空");
        }

        if (request.getEndDate().before(request.getBeginDate())) {
            throw new BadTenantException("搜索轨迹的结束时间不能早于开始时间");
        }

        Duration duration =
                Duration.between(request.getBeginDate().toInstant(),
                        request.getEndDate().toInstant());
        if (duration.toDays() > 30) {
            throw new BadTenantException("只提供30天的轨迹查询功能");
        }

        Collection<DeviceTrackResponse> data =
        this.deviceWriterRpcProvider.getTrackLocations(new DeviceTrackRequest(id, request.getBeginDate(), request.getEndDate()));

        return this.ok(data);
    }

    @Operation(summary = "同步设备位置信息", description = "同步设备位置信息",
            requestBody =
            @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    required = true),
            responses = {
                    @ApiResponse(responseCode = "200", description = "ok")
            }
    )
    @PostMapping("/sync-locations")
    public ResponseEntity post(@RequestBody BatchDeviceLocationRequest request) {
        this.deviceWriterRpcProvider.batch(request);
        return this.ok();
    }
}
