package com.bcxin.tenant.open.rest.apis.components;

import com.bcxin.tenant.open.infrastructures.components.JsonProvider;
import com.bcxin.tenant.open.rest.apis.caches.CacheKeyManage;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;

import java.time.Duration;
import java.util.Collection;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.stream.Collectors;

@Component
public class HotCacheProviderImpl implements HotCacheProvider {
    private static final Logger logger = LoggerFactory.getLogger(HotCacheProviderImpl.class);
    private static final String HEIGHT_WEIGHT_STATION_KEY=CacheKeyManage.getHotEmployeeStationId("");
    private static Cache<String, Object> _hotCache;

    static {
        _hotCache =
                Caffeine.newBuilder()
                        .maximumWeight(1000)
                        .softValues()
                        .weigher((String key, Object value) -> {
                            /**
                             * 优先确保小批量的驻勤信息
                             */
                            if (key.startsWith(HEIGHT_WEIGHT_STATION_KEY)) {
                                return 1000; // 高权重，确保不会被移除
                            }
                            return 1; // 默认权重
                        })
                        .removalListener((key, value, cause) -> {
                            try {
                                if (_hotCache != null) {
                                    logger.error("Caffeine缓存数据被自动移除:{}-{} 原因:{}; 当前的大小:{}", key, value, cause.toString(), _hotCache.estimatedSize());
                                }
                            }
                            catch (Exception ex) {
                                ex.printStackTrace();
                            }
                        }).expireAfterAccess(30, TimeUnit.MINUTES).build();
    }

    private final RedisTemplate redisTemplate;
    private final JsonProvider jsonProvider;

    public HotCacheProviderImpl(RedisTemplate redisTemplate, JsonProvider jsonProvider) {
        this.redisTemplate = redisTemplate;
        this.jsonProvider = jsonProvider;
    }

    /**
     * 这边的缓存时间要大于_hotCache的时间
     * @param key
     * @param supplier
     * @param cacheInSeconds
     * @param tClass
     * @return
     * @param <T>
     */
    @Override
    public <T> T get(String key, Supplier<T> supplier, int cacheInSeconds, Class<T> tClass, boolean enableLocalCache) {
        T data = this.get(key,tClass);
        try {
            if (data == null) {
                data = supplier.get();
                if (data != null) {
                    this.put(key, cacheInSeconds, data, enableLocalCache);
                }
            }

            return data;
        } catch (Exception ex) {
            logger.error("缓存获取异常v2:{}", getCacheKey(key), ex);
            return supplier.get();
        }

    }

    @Override
    public <T> T get(String key, Class<T> tClass) {
        T data = null;
        try {
            data = (T) _hotCache.getIfPresent(key);
        } catch (Exception ex) {
            //todo: 忽略本地缓存的问题
        }

        if (data == null) {
            try {
                String content = (String) this.redisTemplate.opsForValue().get(getCacheKey(key));
                if (StringUtils.hasLength(content)) {
                    data = this.jsonProvider.toObject(tClass, content);
                    _hotCache.put(key, data);
                }

                return data;
            } catch (Exception ex) {
                logger.error("缓存获取异常:{}", getCacheKey(key), ex);
            }
        }

        return data;
    }

    @Override
    public <T> void put(String key, int cacheInSeconds, T data, boolean enableLocalCache) {
        try {
            this.redisTemplate.opsForValue().set(getCacheKey(key), this.jsonProvider.getJson(data), Duration.ofSeconds(cacheInSeconds));
            if(enableLocalCache) {
                _hotCache.put(key, data);
            }
        } catch (Exception ex) {
            logger.error("存储缓存发生异常:{}", getCacheKey(key), ex);
        }
    }

    @Override
    public void delete(String key) {
        try {
            this.redisTemplate.delete(getCacheKey(key));
            _hotCache.invalidate(key);
        } catch (Exception ex) {
            logger.error("缓存删除异常:{}", getCacheKey(key), ex);
        }
    }

    @Override
    public void delete(Collection<String> keys) {
        if (CollectionUtils.isEmpty(keys)) {
            return;
        }

        Collection<String> cacheKeys = keys.stream().map(ii -> getCacheKey(ii)).collect(Collectors.toList());
        try {
            _hotCache.invalidateAll(keys);
        } catch (Exception ex) {
            logger.error("批量本地缓存删除异常:{}", cacheKeys, ex);
        }

        try {
            this.redisTemplate.delete(cacheKeys);
        } catch (Exception ex) {
            logger.error("批量缓存删除异常:{}", cacheKeys, ex);
        }
    }

    @Override
    public <T> T getFromLocal(String key, Supplier<T> supplier, int cacheInSeconds, Class<T> tClass) {
        T data = null;
        try {
            data = (T) _hotCache.getIfPresent(key);
        } catch (Exception ex) {
            //todo: 忽略本地缓存的问题
        }

        if (data == null) {
            data = supplier.get();
            _hotCache.put(key, data);
        }

        return data;
    }

    @Override
    public Map<String,Object> getAllLocalCaches() {
        return _hotCache.asMap();
    }

    private static String getCacheKey(String key) {
        return String.format("ld:l1:%s", key);
    }
}