package com.bcxin.flink.cdc.kafka.source.task.compnents;

import com.bcxin.event.core.JsonProvider;
import com.bcxin.event.core.JsonProviderImpl;
import com.bcxin.event.core.dtos.HttpDomainRegionDTO;
import com.bcxin.flink.cdc.kafka.source.task.JobContext;
import com.bcxin.flink.cdc.kafka.source.task.cdcs.CdcSourceMeta;
import com.bcxin.flink.cdc.kafka.source.task.proerpties.HttpRegionSinkProperty;
import com.bcxin.flink.streaming.cores.JdbcJobExecutorUtil;
import lombok.Data;
import lombok.Getter;
import org.apache.flink.api.connector.sink2.SinkWriter;
import org.apache.http.HttpStatus;
import org.apache.http.StatusLine;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.AbstractHttpEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.io.Serializable;
import java.nio.charset.StandardCharsets;
import java.sql.Connection;
import java.util.*;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 负责用于实现符合写入的功能
 * 业务逻辑:
 * 1, 计算数据并写入Redis
 * 2, 根据结算结果额外(原始数据 + 新kafka)产生数据写入kafka
 * @param
 */
@Getter
public class WebHttpSinkWriter implements SinkWriter<String> {
    private static final Logger logger = LoggerFactory.getLogger(WebHttpSinkWriter.class);
    private final List<WebContentParameterDTO> bulkRequests = new ArrayList<>();
    private transient CloseableHttpClient httpClient = null;
    private final int batchSize;
    private final int batchIntervalMs;
    private final JsonProvider jsonProvider;
    private final Collection<HttpRegionSinkProperty> httpRegionSinkProperties;
    private final String kafkaCdcTopicPrefix;
    private final Connection connection;

    public WebHttpSinkWriter(int batchSize, int batchIntervalMs,
                             Collection<HttpRegionSinkProperty> httpRegionSinkProperties,
                             String kafkaCdcTopicPrefix,
                             Connection connection) {
        this.httpRegionSinkProperties = httpRegionSinkProperties;
        this.kafkaCdcTopicPrefix = kafkaCdcTopicPrefix;
        if (batchSize < 10) {
            this.batchSize = 500;
        } else {
            this.batchSize = batchSize;
        }

        if (batchIntervalMs < 500) {
            this.batchIntervalMs = 5000;
        } else {
            this.batchIntervalMs = batchIntervalMs;
        }

        this.jsonProvider = new JsonProviderImpl();

        ScheduledExecutorService executorService =
                Executors.newSingleThreadScheduledExecutor();
        executorService.scheduleAtFixedRate(() -> {
            try {
                this.flush(true);
            } catch (Exception ex) {
                ex.printStackTrace();

                logger.error("执行数据推送发生异常", ex);
            }

        }, batchIntervalMs, batchIntervalMs, TimeUnit.MICROSECONDS);

        this.connection = connection;
        this.httpClient = HttpClients.createDefault();
    }


    @Override
    public void write(String element, Context context) throws IOException, InterruptedException {
        JsonProvider jsonProvider = new JsonProviderImpl();
        CdcSourceMeta schema = jsonProvider.toObject(CdcSourceMeta.class, element);
        if (schema == null || schema.getDbName() == null) {
            logger.error("当前无效数据:{}", element);
            throw new IllegalArgumentException(element);
        }
        String key = schema.getId();

        String topic =
                this.getKafkaCdcTopicPrefix().replace("[dbName]", schema.getDbName())
                        .replace("[tableName]", schema.getTableName());

        bulkRequests.add(WebContentParameterDTO.create(topic, key, element, schema.getDomainId()));
        if (batchSize > 0 && batchSize <= bulkRequests.size()) {
            flush(true);
        }
    }

    @Override
    public void flush(boolean endOfInput) throws IOException, InterruptedException {
        if (CollectionUtils.isEmpty(bulkRequests)) {
            return;
        }

        Collection<String> domainIds =
                this.bulkRequests.stream()
                .filter(ix -> StringUtils.hasLength(ix.getDomainId()))
                .map(ix -> ix.getDomainId())
                .distinct().collect(Collectors.toList());

        Collection<HttpDomainRegionDTO> domainRegionDTOS =
                JdbcJobExecutorUtil.getDomainRegionDTOs(domainIds, this.connection);

        for (HttpDomainRegionDTO drt : domainRegionDTOS) {
            if (!StringUtils.hasLength(drt.getDomainId())) {
                logger.error("当前组织（{}）的数据组织信息无效", drt.getDomainId());
                continue;
            }

            Collection<HttpRegionSinkProperty> regionSinkProperties =
                    this.getHttpRegionSinkProperties()
                            .stream().filter(ix -> StringUtils.hasLength(ix.getRegionIdPrefix()) &&
                                    drt.getRegionId().startsWith(ix.getRegionIdPrefix()))
                            .collect(Collectors.toList());
            if (CollectionUtils.isEmpty(regionSinkProperties)) {
                logger.error("可忽略: 当前组织（{}）的数据由于无配置区域信息; 因此, 无法推送到对应子站", drt.getDomainId());
                /**
                 * todo: 先推送到默认系统
                 */
                //continue;
                regionSinkProperties = Collections.singleton(this.getHttpRegionSinkProperties().stream().findFirst().get());
            }

            List<WebContentParameterDTO> selectedWebContents
                    = this.bulkRequests.stream().filter(ix ->
                            StringUtils.hasLength(ix.getDomainId()) &&
                                    ix.getDomainId().equalsIgnoreCase(drt.getDomainId()))
                    .collect(Collectors.toList());

            for (HttpRegionSinkProperty httpRegionSink : regionSinkProperties) {
                /**
                 * todo: 临时; 这些是无效数据
                 */
                if(StringUtils.hasLength(drt.getRegionId()) &&
                        !com.bcxin.event.core.utils.StringUtils.hasAlpha(drt.getRegionId())) {

                    for (WebContentParameterDTO wp : selectedWebContents) {
                        wp.setKey(String.format("%s#no_region#%s#%s", wp.getKey(), drt.getRegionId(), drt.getSuperviseDepartId()));
                    }
                }

                //String api = "http://49.4.21.141:4090/event-transfer/data-binlog-transfer";
                this.doExecute(httpRegionSink.getApi(), selectedWebContents);
            }
        }
    }

    @Override
    public void close() throws Exception {
        if (this.connection != null && !this.connection.isClosed()) {
            this.connection.close();
        }

        this.httpClient.close();
    }


    private void doExecute(String api, List<WebContentParameterDTO> parameters) throws IOException {
        String body = null;
        try {
            Map<String, List<WebContentParameterDTO>> requestItems = new HashMap<>();
            requestItems.put("items", parameters);

            body = this.jsonProvider.getJson(requestItems);


            HttpPost post = new HttpPost(api);
            AbstractHttpEntity httpEntity = new StringEntity(body, "UTF-8");
            httpEntity.setContentEncoding(StandardCharsets.UTF_8.toString());
            httpEntity.setContentType("application/json");
            post.setEntity(httpEntity);

            try (CloseableHttpResponse response = httpClient.execute(post)) {
                StatusLine statusLine = response.getStatusLine();

                if (statusLine.getStatusCode() != HttpStatus.SC_OK) {
                    String content = EntityUtils.toString(response.getEntity());
                    logger.error("{}: 执行结束(总数量={}):{} 失败-状态:{} 请求-{}; 响应:{}",
                            Thread.currentThread().getId(),
                            bulkRequests.size(),
                            api,
                            statusLine.getStatusCode(),
                            body,
                            content);
                } else {
                    logger.error("{}: 成功-执行结束(总数量={}):{} Ok-响应:{}",
                            Thread.currentThread().getId(),
                            bulkRequests.size(),
                            api, statusLine.getStatusCode());
                }
            }
        } catch (Exception ex) {
            logger.error("WebHttpSink调用(body={})发生异常", body, ex);
            //todo
        }
    }


    @Data
    public static class WebContentParameterDTO implements Serializable {
        private String topic;
        private String key;
        private String value;
        private String domainId;

        public static WebContentParameterDTO create(String topic, String key, String value,String domainId) {
            WebContentParameterDTO parameter = new WebContentParameterDTO();
            parameter.setKey(key);
            parameter.setTopic(topic);
            parameter.setValue(value);
            parameter.setDomainId(domainId);

            return parameter;
        }
    }
}
