package com.bcxin.flink.cdc.kafka.source.task;

import com.bcxin.event.core.JobParameterDTO;
import com.bcxin.event.core.JsonProvider;
import com.bcxin.event.core.JsonProviderImpl;
import com.bcxin.event.core.enums.JobType;
import com.bcxin.event.core.exceptions.BadEventException;
import com.bcxin.event.core.exceptions.IllegalArgumentEventException;
import com.bcxin.event.core.exceptions.NoFoundEventException;
import com.bcxin.flink.cdc.kafka.source.task.compnents.JdbcDbReaderComponent;
import com.bcxin.flink.cdc.kafka.source.task.jobs.DbStreamCdcHttpSinkJob;
import com.bcxin.flink.cdc.kafka.source.task.jobs.DbStreamCdcJobAbstract;
import com.bcxin.flink.cdc.kafka.source.task.jobs.DbStreamCdcKafkaJob;
import com.bcxin.flink.cdc.kafka.source.task.proerpties.*;
import com.bcxin.flink.streaming.cores.JdbcJobExecutorUtil;
import com.bcxin.flink.streaming.cores.SystemPropertyUtil;
import com.bcxin.flink.streaming.cores.TableConstant;
import com.bcxin.flink.streaming.cores.properties.CheckpointConfigProperty;
import org.apache.flink.api.java.utils.ParameterTool;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;

import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

public class JdbcBinLogCaptureApp {
    private static final Logger logger = LoggerFactory.getLogger(JdbcBinLogCaptureApp.class);
    private static final JsonProvider jsonProvider = new JsonProviderImpl();
    private static JdbcDbReaderComponent dbReaderComponent = null;
    /**
     * https://cloud.tencent.com/developer/article/2010848
     * @param args
     * @throws Exception
     */
    public static void main(String[] args) throws Exception {
        logger.info("接收参数准备执行CDC的BINLOG读取:{}", Arrays.stream(args).collect(Collectors.joining(";")));
        ParameterTool parameterTool = ParameterTool.fromArgs(args);
        buildJobEnvironment(parameterTool);

        DbStreamCdcJobAbstract job = null;
        if (isHttpSink(args)) {
            job = new DbStreamCdcHttpSinkJob();
        } else {
            job = new DbStreamCdcKafkaJob(dbReaderComponent);
        }

        job.execute();
    }

    private static void buildJobEnvironment(ParameterTool parameterTool) throws IOException {
        String[] args = new String[]{
                parameterTool.get("env"),
                parameterTool.get("jobId"),
                parameterTool.get("mode")
        };
        Properties properties = loadProperties(args);
        CheckpointConfigProperty configProperty =
                CheckpointConfigPropertyBuilder.build(properties,parameterTool);
        dbReaderComponent = JdbcDbReaderComponent.build(properties);

        if (isHttpSink(args)) {
            Collection<HttpRegionSinkProperty> regionSinkProperties = new ArrayList<>();
            Enumeration<Object> regionKeys = properties.keys();
            while (regionKeys.hasMoreElements()) {
                String name = (String) regionKeys.nextElement();
                String value = properties.getProperty(name);

                HttpRegionSinkProperty sinkProperty = HttpRegionSinkProperty.getRegionConfigProperty(name, value);
                if (sinkProperty != null) {
                    regionSinkProperties.add(sinkProperty);
                }
            }

            if (CollectionUtils.isEmpty(regionSinkProperties)) {
                throw new BadEventException("无效Http Region配置信息");
            }

            JobContext.initHttpSink(
                    properties.getProperty(Constants.JOB_NAME),
                    regionSinkProperties,
                    CdcDatabaseSourceProperty.create(properties),
                    configProperty);
        } else {
            /**
             * kafka property
             */
            KafkaConfigProperty kafkaConfigProperty = KafkaConfigProperty.create(
                    properties.getProperty(Constants.KAFKA_BOOTSTRAP_SERVER),
                    properties.getProperty(Constants.KAFKA_CDC_TOPIC_PREFIX),
                    properties.getProperty(Constants.KAFKA_CONSUMER_GROUP_ID)
            );

            JobContext.initKafkaSink(
                    properties.getProperty(Constants.JOB_NAME),
                    CdcDatabaseSourceProperty.create(properties),
                    configProperty,
                    kafkaConfigProperty);
        }
    }

    private static Properties loadProperties(String[] args) throws IOException {
        if (args == null || args.length < 2) {
            throw new BadEventException("无提供有效配置参数(环境以及jobId), 系统无法启动");
        }
        String env = args[0];
        Properties properties = SystemPropertyUtil.loadEnvConf(env);

        int jobId = Integer.parseInt(args[1]);
        JobParameterDTO jobParameterDTO = JdbcJobExecutorUtil.getJobParameter(
                jobId
        );

        if (jobParameterDTO == null) {
            throw new NoFoundEventException("参数无效; 无法加载对应的job参数");
        }

        if (jobParameterDTO.getJobType() != JobType.CDC_BINLOG) {
            throw new IllegalArgumentEventException(String.format("该jobType(%s)不支持BINLOG CDC处理", jobParameterDTO.getJobType()));
        }

        String corePropertyConf = jobParameterDTO.getParam1(jsonProvider);
        String databasePropertyConf = jobParameterDTO.getParam2(jsonProvider);

        Map<String, String> corePropertyConfMap = null;
        try {
            corePropertyConfMap = jsonProvider.toObject(Map.class, corePropertyConf);
        } catch (Exception ex) {
            throw new BadEventException(String.format("转化异常:%s", corePropertyConf), ex);
        }

        Map<String, String> databasePropertyConfMap = null;
        try {
            databasePropertyConfMap = jsonProvider.toObject(Map.class, databasePropertyConf);
        } catch (Exception ex) {
            throw new BadEventException(String.format("转化异常:%s", databasePropertyConf), ex);
        }

        properties.setProperty(TableConstant.PARAM_JOB_ID_NAME, String.valueOf(jobId));
        if (corePropertyConfMap != null) {
            for (String key : corePropertyConfMap.keySet()) {
                properties.setProperty(key, corePropertyConfMap.get(key));
            }
        }

        if (databasePropertyConfMap != null) {
            for (String key : databasePropertyConfMap.keySet()) {
                properties.setProperty(key, databasePropertyConfMap.get(key));
            }
        }

        properties.setProperty(TableConstant.PARAM_JOB_ID, String.valueOf(jobId));

        if (!isHttpSink(args) && args.length > 2 && args[2]!=null) {

            properties.setProperty(TableConstant.PARAM_BINLOG_SKIP_REDIS_CALCULATE, args[2]);

            if (!TableConstant.PARAM_BINLOG_SKIP_REDIS_CALCULATE_VALUE.equalsIgnoreCase(args[2])) {
                properties.setProperty(TableConstant.PARAM_BINLOG_STARTUP_OPTION, args[2]);
            }
        }

        return properties;
    }

    private static boolean isHttpSink(String[] args) {
        return args.length > 2 && "http-sink".equalsIgnoreCase(args[2]);
    }
}
