package com.bcxin.risk.sys;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.map.MapUtil;
import com.bcxin.risk.cache.redis.RedisUtil;
import com.bcxin.risk.common.dao.RegionDao;
import com.bcxin.risk.common.domain.Region;
import com.bcxin.risk.constant.RedisKeyConst;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 城市地区工具类
 */
@Service
@Transactional
public class SysRegionUtil {

    @Resource
    private RegionDao regionDao;

    @Resource
    private RedisUtil redisUtil;

    private static List<Region> CACHE_KEY_REGION_PROVINCE_LIST = Lists.newArrayList();

    private static List<Region> CACHE_KEY_REGION_CITY_LIST = Lists.newArrayList();

    private static List<Region> CACHE_KEY_REGION_AREA_LIST = Lists.newArrayList();

    private static Map<String,Region> CACHE_KEY_REGION_PROVINCE_MAP = Maps.newHashMap();

    private static Map<String,Collection<Region>> CACHE_KEY_REGION_UNIT_MAP = Maps.newHashMap();

    /**
     * 所有区域的map
     * @return
     */
    private void setRegionMap(){
        List<Region> rList = regionDao.selectRegionList();
        Map<String,Object> regionMap = Maps.newLinkedHashMap();
        for( Region region:rList){
            regionMap.put(region.getOid()+"",region);
        }
        redisUtil.addAllHashValue(RedisKeyConst.REDIS_KEY_REGION, regionMap, RedisKeyConst.TIMEOUT_ONE_WEEK);
    }


    /**
     * 省份
     * @return
     */
    public Collection<Region> provinceMap(){
        if (CollectionUtil.isNotEmpty(CACHE_KEY_REGION_PROVINCE_LIST)) {
            return CACHE_KEY_REGION_PROVINCE_LIST;
        }
        List<Region> rList = regionDao.selectProvinceRegionList();
        for (Region region :rList) {
            CACHE_KEY_REGION_PROVINCE_MAP.put(region.getOid()+"",region);
        }
        return rList;
    }

    /**
     * 市的map
     * @return
     */
    public Map<String,Collection<Region>> cityMap(){
        if (CollectionUtil.isNotEmpty(CACHE_KEY_REGION_CITY_LIST)) {
            Multimap<String, Region> cMap = ArrayListMultimap.create();
            for (Region region : CACHE_KEY_REGION_CITY_LIST){
                cMap.put(region.getParent_id()+"",region);
            }
            return cMap.asMap();
        }
        Multimap<String, Region> cMap = ArrayListMultimap.create();
        CACHE_KEY_REGION_CITY_LIST = regionDao.selectCityRegionList();
        for (Region region : CACHE_KEY_REGION_CITY_LIST){
            cMap.put(region.getParent_id()+"",region);
        }
        return cMap.asMap();
    }

    /**
     * 市的map
     * @return
     */
    public Collection<Region> cityMap(String parentId){
        return cityMap().get(parentId);
    }


    public Map<String,Collection<Region>> areaMap(){
        if (CollectionUtil.isNotEmpty(CACHE_KEY_REGION_AREA_LIST) ) {
            Multimap<String, Region> cMap = ArrayListMultimap.create();
            for (Region region : CACHE_KEY_REGION_AREA_LIST){
                cMap.put(region.getParent_id()+"",region);
            }
            return cMap.asMap();
        }
        Multimap<String, Region> cMap = ArrayListMultimap.create();
        CACHE_KEY_REGION_AREA_LIST = regionDao.selectAreaRegionList();
        for (Region region : CACHE_KEY_REGION_AREA_LIST){
            cMap.put(region.getParent_id()+"",region);
        }
        return cMap.asMap();
    }


    /**
     * 区的map
     * @return
     */
    public Collection<Region> areaMap(String parentId){
        return areaMap().get(parentId);
    }

    /**
     * 许可单位的map
     * @return
     */
    public Collection<Region> unitMap(String parentId){
        if (MapUtil.isNotEmpty(CACHE_KEY_REGION_UNIT_MAP)) {
            return CACHE_KEY_REGION_UNIT_MAP.get(parentId);
        }
        Multimap<String, Region> uMap = ArrayListMultimap.create();
        List<Region> unitList = regionDao.selectUnitRegionList();
        for (Region region : unitList){
            uMap.put(region.getParent_id(),region);
        }
        CACHE_KEY_REGION_UNIT_MAP = uMap.asMap();
        return CACHE_KEY_REGION_UNIT_MAP.get(parentId);
    }

    /**
     * 通过ID返回地区
     * @param id
     * @return
     */
    public Region getRegionById(String id){
        Object obj = redisUtil.getHashValue(RedisKeyConst.REDIS_KEY_REGION, id);
        if ( obj == null ) {
            setRegionMap();
            obj = redisUtil.getHashValue(RedisKeyConst.REDIS_KEY_REGION, id);
        }
        return (Region)obj;
    }

    /**
     * 通过ID返回省地区
     * @param id
     * @return
     */
    public Region getProvinceRegionById(String id){
        return CACHE_KEY_REGION_PROVINCE_MAP.get(id);
    }

}
