import { mapMutations, mapActions } from "vuex";
import mqttService from "@/services/mqttService";
import store from "@/store/index";

export default {
  data() {
    return {
      client: null,
      localStream: null,
      remoteStreamList: [],
      isJoining: false,
      isJoined: false,
      isPublishing: false,
      isPublished: false,
      isPlayingLocalStream: false,
    };
  },
  methods: {
    ...mapActions(["setYardman"]),
    ...mapMutations(["SET_MEMBER_LIST", "SET_YARDMAN_AUDIO_MUTED", "SET_YARDMAN_VIDEO_MUTED"]),
    addSuccessLog(message) {
      this.$store.commit("showMessage", { message });
    },
    // 初始化客户端
    async initClient(account) {
      console.log('mixin initClient-----', account)
      try {
        // 使用mqttService初始化客户端
        this.client = await mqttService.initClient(account);
        this.handleClientEvents();
      } catch (e) {
        this.$message.error(e);
      }
    },
    handleClientEvents() {
      // 设置组件特定的事件监听
      this.client.on("init", () => {
        this.addSuccessLog(`客户端已创建`);
      });
      
      this.client.on("error", (error) => {
        console.error(error);
        this.$store.commit("showMessage", {
          message: error,
          type: "error",
        });
      });
      
      this.client.on("message", ({ topic, response }) => {
        if (topic.indexOf("target/pc/qos0/msg") > -1) {
          // 返回上下线信息
          console.log("返回上下线信息", response);
        } else if (topic.indexOf("target/media/signal") > -1) {
          console.log("返回语音视频信令", response);
          const { from, data } = response;
          console.log("是谁发的信息：", from);
          if (from && data) {
            const { action } = data;
            if (action === 3) {
              // 更新成员列表
              const memberList = this.memberList.map((member) => {
                if (member.dispatchUid.toString() === from.toString()) member.handUp = true;
                return member;
              });
              this.SET_MEMBER_LIST(memberList); // 更新被调度人数组
            } 
          }
          // 延迟拉取视频
          setTimeout(() => {
            this.client.pullOnVideo(response);
          }, 1000);
        }
      });
    },
    async initLocalStream({ audio = true, video = true }) {
      this.client.publishLocalVideo({
        isAudio: audio,
        isVideo: video,
      });
      // this.client.sendPullVideoCall(account);
    },
    openLocalVideo({ audio = true, video = true }) {
      this.client.openLocalVideo({ isAudio: audio, isVideo: video });
    },
    closeLocalVideo() {
      this.client.closeLocalVideo();
    },
    async leave(isCallVideo, toId) {
      this.client.leave(isCallVideo, toId);
    },
    // 本地禁麦
    muteAudio() {
      if (!this.client.pushVideoObj) return;
      this.client.pushVideoObj.getLocalAudioEnabledTrack();
      this.SET_YARDMAN_AUDIO_MUTED(true);
      this.addSuccessLog("已禁用音频");
    },
    // 本地取消禁麦
    unmuteAudio() {
      if (!this.client.pushVideoObj) return;
      this.client.pushVideoObj.getLocalAudioTrack();
      this.SET_YARDMAN_AUDIO_MUTED(false);
      this.addSuccessLog("已启用音频");
    },
    muteVideo() {
      if (!this.client.pushVideoObj) return;
      this.client.pushVideoObj.getLocalCameraEnabledTrack();
      this.SET_YARDMAN_VIDEO_MUTED(true);
      this.addSuccessLog("已禁用视频");
    },
    unmuteVideo() {
      if (!this.client.pushVideoObj) return;
      this.client.pushVideoObj.getLocalCameraTrack();
      this.SET_YARDMAN_VIDEO_MUTED(false);
      this.addSuccessLog("已启用视频");
    },
    // 邀请成员入会
    inviteRemote(inviteList, options) {
      if (!inviteList || inviteList.length <= 0) return false;
      inviteList.forEach((item) => {
        this.client.invite({ ...options, toId: item.dispatchUid });
        this.setMemberConnectState(item.dispatchUid, "CONNECTING");
      });
    },
    // 同意远端发言
    handUpRemote(dispatchUid) {
      this.client.publish({ action: 5, toId: dispatchUid });
      const memberList = this.memberList.map((member) => {
        if (member.dispatchUid.toString() === dispatchUid.toString()) {
          member.handUp = false;
          member.isMutedAudio = false;
        }
        return member;
      });
      this.SET_MEMBER_LIST(memberList); // 更新被调度人数组
    },
    kickOutRemote(id) {
      if (id === "" || id === undefined || id === null) return false;
      this.client.kickOut({ toId: id });
      this.setMemberConnectState(id, "DISCONNECTED");
    },
    muteRemoteAudio(id) {
      console.log("muteRemoteAudio", id);
      if (id === "" || id === undefined || id === null) return false;
      this.client.toCancelMic({ toId: id });
      const memberList = this.memberList.map((member) => {
        if (member.dispatchUid.toString() === id.toString()) member.isMutedAudio = true;
        return member;
      });
      this.SET_MEMBER_LIST(memberList); // 更新被调度人数组
    },
    unmuteRemoteAudio(id) {
      console.log("unmuteRemoteAudio", id);
      if (id === "" || id === undefined || id === null) return false;
      this.client.toSendMic({ toId: id });
      const memberList = this.memberList.map((member) => {
        if (member.dispatchUid.toString() === id.toString()) member.isMutedAudio = false;
        return member;
      });
      this.SET_MEMBER_LIST(memberList); // 更新被调度人数组
    },
    // 更新被调度人连接状态
    setMemberConnectState(id, state) {
      const memberList = this.memberList.map((member) => {
        if (member.dispatchUid.toString() === id.toString()) member.state = state;
        return member;
      });
      this.SET_MEMBER_LIST(memberList); // 更新被调度人数组
    },
    // 监听到远端加入
    pullVideoFromServer(id) {
      this.micUserId = id;
      // 拉取远端流，播放远端音视频
      this.client.pullRemoteVideo(id);
      // 更新被调度人连接状态
      this.setMemberConnectState(id, "CONNECTED");
    },
    // 监听到远端离开
    pullVideoLeave(id) {
      console.log("监听到远端离开", id);
      this.client.deleteRemote(id);
      // 更新被调度人连接状态
      this.setMemberConnectState(id, "DISCONNECTED");
    },
  },
};