import { getOrgId, getDeviceListByOrgId, getDeviceChannelList } from '@/api/video-controller';
import { getStationOrExamDevices } from '@/api/security-station-controller'

/**
 * 设备通道服务类 - 用于获取组织下的设备通道列表
 */
class DeviceChannelService {
  constructor() {
    this.userId = process.env.VUE_APP_HUAWEI_USER_ID;
    this.cache = {
      orgIdMap: new Map(), // 组织名称 -> 组织ID的映射缓存
      deviceListMap: new Map(), // 组织ID -> 设备列表的映射缓存
    };
    this.cacheExpiration = 5 * 60 * 1000; // 缓存过期时间：5分钟
  }

  /**
   * 根据设备组名称获取通道列表
   * @param {string} deviceOrgName - 设备组名称
   * @param {boolean} useCache - 是否使用缓存，默认为true
   * @param {string} protocol - 通道协议，默认为"GB28181"
   * @returns {Promise<Array>} - 通道列表
   */
  async getDevicesByOrgName(deviceOrgName, useCache = true) {
    try {
      // 1. 获取组织ID
      const orgId = await this.getOrgIdByName(deviceOrgName, useCache);
      if (!orgId) {
        console.error(`未找到名称为 "${deviceOrgName}" 的设备组`);
        return [];
      }

      // 2. 获取组织下的设备列表
      const res = await this.getDevicesByOrgId(orgId, useCache);
      console.log('获取设备列表', res)
      if (!res || res.length === 0) {
        console.warn(`设备组 "${deviceOrgName}" (ID: ${orgId}) 下没有设备`);
        return [];
      }

      const deviceList = res.map(device => {
        return {
          device_id: device.device_id,
          channel_id: device.channels[0].channel_id,
        }
      });

      return deviceList;
    } catch (error) {
      console.error('获取设备列表失败:', error);
      throw error;
    }
  }

  /**
   * 根据设备组名称获取组织ID
   * @param {string} deviceOrgName - 设备组名称
   * @param {boolean} useCache - 是否使用缓存
   * @returns {Promise<string|null>} - 组织ID
   */
  async getOrgIdByName(deviceOrgName, useCache = true) {
    // 检查缓存
    const cacheKey = deviceOrgName;
    const cachedData = this.cache.orgIdMap.get(cacheKey);
    if (useCache && cachedData && Date.now() - cachedData.timestamp < this.cacheExpiration) {
      return cachedData.data;
    }
    try {
      const response = await getOrgId(deviceOrgName);
      
      if (response && response.result && response.result.length > 0) {
        // 查找匹配的组织
        const org = this.findOrgByName(response.result[0], deviceOrgName);
        console.log('获取组织ID',response.result[0],deviceOrgName, org)
        if (org) {
          // 更新缓存
          this.cache.orgIdMap.set(cacheKey, {
            data: org.device_org_id,
            timestamp: Date.now()
          });
          
          return org.device_org_id;
        }
      }
      
      return null;
    } catch (error) {
      console.error('获取组织ID失败:', error);
      throw error;
    }
  }

  /**
   * 递归查找组织树中匹配名称的组织
   * @param {Object} orgNode - 组织节点
   * @param {string} name - 要查找的组织名称
   * @returns {Object|null} - 找到的组织对象或null
   */
  findOrgByName(orgNode, name) {
    if (!orgNode) return null;
    
    // 检查当前节点
    if (orgNode.device_org_name === name) {
      return orgNode;
    }
    
    // 检查子节点
    if (orgNode.children && orgNode.children.length > 0) {
      for (const child of orgNode.children) {
        const result = this.findOrgByName(child, name);
        if (result) return result;
      }
    }
    
    return null;
  }

  /**
   * 根据组织ID获取设备列表
   * @param {string} orgId - 组织ID
   * @param {boolean} useCache - 是否使用缓存
   * @returns {Promise<Array>} - 设备列表
   */
  async getDevicesByOrgId(orgId, useCache = true) {
    // 检查缓存
    const cacheKey = orgId;
    const cachedData = this.cache.deviceListMap.get(cacheKey);
    if (useCache && cachedData && Date.now() - cachedData.timestamp < this.cacheExpiration) {
      return cachedData.data;
    }

    try {
      const response = await getDeviceListByOrgId(orgId);
      if (response && response.devices) {
        // 更新缓存
        this.cache.deviceListMap.set(cacheKey, {
          data: response.devices,
          timestamp: Date.now()
        });
        
        return response.devices;
      }
      
      return [];
    } catch (error) {
      console.error(`获取组织(ID: ${orgId})的设备列表失败:`, error);
      throw error;
    }
  }

    async getDevicesByStationId(StationId) {
 
    try {
      const response = await getStationOrExamDevices(StationId);
      if (response) {
        return response;
      }
      
      return [];
    } catch (error) {
      console.error(`获取站点(ID: ${StationId})的设备列表失败:`, error);
      throw error;
    }
  }

  /**
   * 清除所有缓存
   */
  clearCache() {
    this.cache.orgIdMap.clear();
    this.cache.deviceListMap.clear();
  }

  /**
   * 清除特定组织的缓存
   * @param {string} deviceOrgName - 设备组名称
   */
  clearOrgCache(deviceOrgName) {
    this.cache.orgIdMap.delete(deviceOrgName);
    
    // 由于组织ID到设备列表的映射需要先获取组织ID，所以这里不清除设备列表缓存
    // 如果需要完全清除与该组织相关的所有缓存，可以调用clearCache()
  }
}

// 导出单例实例
export default new DeviceChannelService();