import { getConfig } from "@/utils";
import axios from "axios";
import { getCurrentSetting } from "@/api/tenant-user-controller";
import store from "@/store/index";
import router from "@/router";
import { Loading } from "element-ui";
import setup from "@/utils/setup";
import generateRoutes from "@/router/dynamicRouters";
import { getAdminAccount,doCreateAdminAccount } from "@/api/gwsd";
import mqttService from "@/services/mqttService";
// ! 初始一个loading当遮罩层
const loadingInstance = Loading.service({
  fullscreen: true,
  spinner: "none",
  background: "rgba(0, 0, 0, 0.8)",
});

store.commit("SET_APP_LOADING_MASK", loadingInstance);

export const getAccessToken = () => {
  return new Promise((resolve) => {
    if (process.env.VUE_APP_ENV === "development") {
      getConfig("accessToken").then((json) => {
        console.log(json);
        const accessToken = json.find((x) => x.name === "苗兵").value;
        localStorage.setItem("accessToken", accessToken.toString());

        resolve(accessToken);
      });
    } else {
      resolve(localStorage.getItem("accessToken"));
    }
  });
};

/**
 * * 自动登录获取dispatchToken
 */
export const permission = async () => {
  const accessToken = await getAccessToken();
  return new Promise((resolve, reject) => {
    axios
      .post(
        `${process.env.VUE_APP_API_ROOT}/identity/Desk/auto-login`,
        {},
        {
          headers: {
            "Content-Type": "multipart/form-data",
            // "skip-ip-validation":1,
            accessToken,
          },
        }
      )
      .then(async (res) => {
        const { status, data } = res;
        if (status === 200) {
          const {
            data: { dispatchToken },
          } = data;
          localStorage.setItem("dispatchToken", dispatchToken);
          // 获取配置
          try {
            const setting = await getCurrentSetting();
            store.commit("SET_APP_SETTING", setting);
          } catch (e) {
            console.error(e);
          }
          resolve(true);
        } else if (status === 401 || status === 403) {
          localStorage.setItem("dispatchToken", "");
          localStorage.setItem("userInfo", "");
          store.commit("showMessage", {
            type: "error",
            message: "未授权无效调度台用户",
            duration: 0,
          });
          store.commit("SET_SYS_USER_INFO", null);
          reject("未授权无效调度台用户");
        }
      })
      .catch((error) => {
        localStorage.setItem("dispatchToken", "");
        localStorage.setItem("userInfo", "");
        store.commit("showMessage", {
          type: "error",
          message: "未授权无效调度台用户",
          duration: 0,
        });
        store.commit("SET_SYS_USER_INFO", null);
        reject("未授权无效调度台用户");
        throw error;
      });
  });
};

/**
 * 删除所有PTT调度组群
 * 避免没有关闭弹框直接关闭页面或者直接刷新页面没有退出组群下次无法创建组群的情况
 * @returns {Promise<void>}
 */
export const clearAllPTTGroups = async () => {
  await store.dispatch("clearGroup");
};

router.beforeEach(async (to, from, next) => {
  if (to.path.toLowerCase().indexOf("voip") > -1) {
    loadingInstance.close();
  }
  await permission();
  console.log("v2.调用permission之后的响应信息:", store.getters.sysUserInfo);
  // * 判断当前用户是否已拉取完user_info信息
  if (store.getters.sysUserInfo === null || Object.keys(store.getters.sysUserInfo).length <= 0) {
    
    // * 获取用户信息、启动调度客户端
    setup()
      .then(async (response) => {
        console.log("current user info ===", response);
        let tracking = "开始获取用户信息";
        try {
          let pttAccount = null;
          for(var aIndex=0;aIndex<2;aIndex++){
            // 获取ptt账号信息（账号、密码、用户名等）
            pttAccount = await getAdminAccount({
              subAccount: response.dispatchNo,
            });
            if(!pttAccount.data){
              let ptAt = await doCreateAdminAccount({
                subAccount: response.dispatchNo,
                adminName: process.env.VUE_APP_PTT_ADMIN_NAME,
                fgrp: 37,
              });
              console.log("v2.调用新增调度台账号之后的响应信息：",ptAt);
              if(ptAt.data){
                pttAccount = ptAt.data[0];
                tracking +="调用新增的API之后存在的用户信息:"+response.dispatchNo;
                pttAccount = await getAdminAccount({
                  subAccount: response.dispatchNo,
                });
                break;
              }
            }else{
              tracking +="用户已经存在:"+response.dispatchNo+"执行次数:"+(aIndex+1);
              break;
            }
          }
          // 初始化MQTT客户端
          await mqttService.initClient(pttAccount.data);
          store.commit("SET_PTT_ACCOUNT", pttAccount.data);

          console.log("v2.获取调度台账号信息"+tracking,pttAccount);

          // 清除所有的PTT调度组群
          await clearAllPTTGroups();
        } catch (e) {
          console.error("getAdminAccount 获取子组织账号接口异常:", e);
        }
        // * 储存用户信息
        localStorage.setItem("userInfo", JSON.stringify(response));
        store.commit("SET_SYS_USER_INFO", response);
        // * 根据用户角色判断调度台类型并储存
        const platformInfRole = [
          {
            value: "1",
            label: "指挥调度台",
            role: [
              "Company",
              "Urban",
              "SubUrb",
              "PoliceStation",
              "PopulationPoliceStation",
              "PopulationUrban",
              "PopulationSubUrb",
            ],
          },
          {
            value: "2",
            label: "内保调度台",
            role: ["Proprietor", "PoliceStation", "PopulationPoliceStation"],
          },
          {
            value: "3",
            label: "临保调度台",
            role: ["Company", "Urban", "SubUrb", "PoliceStation", "PopulationPoliceStation"],
            // disabled: true,
          },
          {
            value: "4",
            label: "赛演调度台",
            role: ["Company", "Urban", "SubUrb", "PoliceStation", "PopulationPoliceStation"],
            // disabled: true,
          },
        ].filter((x) => x.role.includes(response.role));
        let defaultPlatform = platformInfRole[0].value;
        const userInfo = localStorage.getItem("userInfo") && JSON.parse(localStorage.getItem("userInfo"));
        if (userInfo.orgInstitutional === "0201") {
          defaultPlatform = 3;
        }
        const storePlatform = localStorage.getItem("platform");
        if (platformInfRole.filter((x) => x.value === storePlatform).length <= 0) {
          localStorage.setItem("platform", defaultPlatform);
        }
        localStorage.setItem("userInfo", JSON.stringify(response));
        // * 根据调度台类型动态生成路由
        generateRoutes(localStorage.getItem("platform"), response).forEach((route) => {
          router.addRoute(route); // 生成该用户的新路由json操作完毕之后,调用vue-router的动态新增路由方法,将新路由添加
        });
        // !! 判断跳转页面是否有权访问
        if (router.getRoutes() && router.getRoutes().length > 0) {
          const [{ path }] = router.getRoutes(); // 第一个路由
          if (router.getRoutes().filter((x) => x.path === to.path).length > 0 || to.path === "/") {
            next({ ...to, replace: true });
          } else {
            next({ path, replace: true }); // 跳转到有权限的第一个路由
          }
        }
        loadingInstance.close();
      })
      .catch((e) => {
        console.log("err", e);
        loadingInstance.close();
        return false;
      });
  } else {
    // * 有用户信息直接进入
    next();
    loadingInstance.close();
  }
});