import { Position } from 'geojson';
import { AllGeoJSON } from '@turf/helpers';

/**
 * Converts a WGS84 GeoJSON object into Mercator (EPSG:900913) projection
 *
 * @function
 * @param {GeoJSON|Position} geojson WGS84 GeoJSON object
 * @param {Object} [options] Optional parameters
 * @param {boolean} [options.mutate=false] allows GeoJSON input to be mutated (significant performance increase if true)
 * @returns {GeoJSON} Projected GeoJSON
 * @example
 * var pt = turf.point([-71,41]);
 * var converted = turf.toMercator(pt);
 *
 * //addToMap
 * var addToMap = [pt, converted];
 */
declare function toMercator<G = AllGeoJSON | Position>(geojson: G, options?: {
    mutate?: boolean;
}): G;
/**
 * Converts a Mercator (EPSG:900913) GeoJSON object into WGS84 projection
 *
 * @function
 * @param {GeoJSON|Position} geojson Mercator GeoJSON object
 * @param {Object} [options] Optional parameters
 * @param {boolean} [options.mutate=false] allows GeoJSON input to be mutated (significant performance increase if true)
 * @returns {GeoJSON} Projected GeoJSON
 * @example
 * var pt = turf.point([-7903683.846322424, 5012341.663847514]);
 * var converted = turf.toWgs84(pt);
 *
 * //addToMap
 * var addToMap = [pt, converted];
 */
declare function toWgs84<G = AllGeoJSON | Position>(geojson: G, options?: {
    mutate?: boolean;
}): G;

export { toMercator, toWgs84 };
