package com.bcxin.platform.service.attend.impl;

import cn.hutool.core.util.StrUtil;
import com.bcxin.platform.common.constant.Constants;
import com.bcxin.platform.common.constant.DictConst;
import com.bcxin.platform.common.core.text.Convert;
import com.bcxin.platform.common.exception.BusinessException;
import com.bcxin.platform.common.utils.DateUtil;
import com.bcxin.platform.common.utils.DateUtils;
import com.bcxin.platform.common.utils.IdWorker;
import com.bcxin.platform.common.utils.bean.BeanUtils;
import com.bcxin.platform.common.utils.validator.ValidatorUtils;
import com.bcxin.platform.common.utils.validator.group.AddGroup;
import com.bcxin.platform.common.utils.validator.group.FactorGroup;
import com.bcxin.platform.common.utils.validator.group.OneGroup;
import com.bcxin.platform.common.utils.validator.group.TwoGroup;
import com.bcxin.platform.domain.attend.AttendShift;
import com.bcxin.platform.domain.company.PerOrgRelation;
import com.bcxin.platform.dto.app.AppAttendShiftDto;
import com.bcxin.platform.dto.attend.AttendShiftDto;
import com.bcxin.platform.mapper.attend.AttendSchedulDetailMapper;
import com.bcxin.platform.mapper.attend.AttendShiftMapper;
import com.bcxin.platform.mapper.company.PerOrgRelationMapper;
import com.bcxin.platform.service.attend.AttendShiftService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 考勤班次Service业务层处理
 * 
 * @author lin
 * @date 2021-08-10
 */
@Service
@Transactional("transactionManager")
public class AttendShiftServiceImpl implements AttendShiftService{

    @Autowired
    private AttendShiftMapper attendShiftMapper;

    @Autowired
    private AttendSchedulDetailMapper attendSchedulMapper;

    @Autowired
    private PerOrgRelationMapper perOrgRelationMapper;

    @Resource
    private IdWorker idWorker;

    /**
     * 查询考勤班次
     * 
     * @param id 考勤班次ID
     * @return 考勤班次
     */
    @Override
    public AttendShift findById(Long id){
        return attendShiftMapper.findById(id);
    }

    /**
     * 查询考勤班次列表
     * 
     * @param searchDto 考勤班次
     * @return 考勤班次
     */
    @Override
    public List<AttendShiftDto> selectList(AttendShiftDto searchDto){
        //加上数据权限
        if(!searchDto.isDomainAdmin() && (searchDto.getAdminDepartIds()== null || searchDto.getAdminDepartIds().size() == 0)){
            return new ArrayList<>();
        }
        return attendShiftMapper.selectList(searchDto);
    }

    @Override
    public int saveBatch(List<AttendShift> shiftList){
        for (AttendShift attendShift : shiftList) {
            update(attendShift);
        }
        return shiftList.size();
    }

    @Override
    public List<AppAttendShiftDto> findByAttendId(AttendShiftDto attendShift) {
        List<AppAttendShiftDto> list = attendShiftMapper.findByAttendId(attendShift);
        for (int i = 0; i < list.size(); i++) {
            list.get(i).setLetterTag(getLetterTag(i));
        }
        return list;
    }

    private static String getLetterTag(int index){
        String[] letters = {"A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","W","X","Y","Z"};
        return letters[index%letters.length]+(index/letters.length > 0 ? index/letters.length:"");
    }

    /**
     * 数据验证
     * @param attendShift
     */
    @Override
    public void validateData(AttendShift attendShift){
        attendShift.setCrossDay(Constants.N);
        ValidatorUtils.validateEntity(attendShift, AddGroup.class);
        if(DictConst.SHIFTTYPE_JSBC.equals(attendShift.getShiftType())) {
            ValidatorUtils.validateEntity(attendShift, OneGroup.class);

            Date stratTime = DateUtil.convertStringToDate(DateUtil.getCurrentDate() + " " + attendShift.getStartTime());
            Date endTime = DateUtil.convertStringToDate( DateUtil.getCurrentDate() + " " + attendShift.getEndTime());

            //跨天排班
            if(stratTime.getTime()>endTime.getTime()){
                endTime = DateUtils.addDays(endTime,1);
                attendShift.setCrossDay(Constants.Y);
            }

            Integer hourDiff = DateUtil.dateDiffForMinutes(stratTime,endTime);

            Integer restHourDiff = 0;
            if(Constants.Y.equals(attendShift.getNeedRest())){
                ValidatorUtils.validateEntity(attendShift, FactorGroup.class);

                Date restStratTime = DateUtil.convertStringToDate(DateUtil.getCurrentDate() + " " + attendShift.getRestStart());
                Date restEndTime = DateUtil.convertStringToDate( DateUtil.getCurrentDate() + " " + attendShift.getRestEnd());

                //跨天排班
                if(restStratTime.getTime()>restEndTime.getTime()){
                    restEndTime = DateUtils.addDays(restEndTime,1);
                }

                if(restStratTime.getTime()<stratTime.getTime() || restStratTime.getTime()>endTime.getTime()){
                    throw new BusinessException("休息开始时间设置错误！");
                }

                if(restEndTime.getTime()<stratTime.getTime() || restEndTime.getTime()>endTime.getTime()){
                    throw new BusinessException("休息结束时间设置错误！");
                }

                restHourDiff = DateUtil.dateDiffForMinutes(restStratTime,restEndTime);
            }

            if((hourDiff-restHourDiff)<30){
                throw new BusinessException("时间间隔太短，应该不少于（含）30分钟，请重新设置下班时间！");
            }

            //上班时长=总时长-休息时长
            attendShift.setShiftHour(BigDecimal.valueOf((hourDiff-restHourDiff)/60.0).setScale(2,BigDecimal.ROUND_HALF_UP).doubleValue());

            //if(attendShift.getShiftHour() > 12){
            //    throw new BusinessException("班次时长不合规（不超过12小时），请重新设置！");
            //}

            attendShift.setRestHour(BigDecimal.valueOf(restHourDiff/60.0).setScale(2,BigDecimal.ROUND_HALF_UP).doubleValue());
        }else {
            attendShift.setNeedRest(Constants.N);
            ValidatorUtils.validateEntity(attendShift, TwoGroup.class);

            AttendShiftDto searchDto = new AttendShiftDto();
            searchDto.setStartTime(attendShift.getStartTime());
            searchDto.setShiftType(attendShift.getShiftType());
            searchDto.setComId(attendShift.getComId());
            searchDto.setDomainAdmin(true);
            searchDto.setAttendId(attendShift.getAttendId());
            List<AttendShiftDto> attendShiftDtos = attendShiftMapper.selectList(searchDto);
            if(attendShiftDtos.size() > 0) {
                List<AttendShiftDto> checkList = attendShiftDtos.stream().filter(shiftDto -> !shiftDto.getId().equals(attendShift.getId())).collect(Collectors.toList());
                if(checkList.size() > 0){
                    throw new BusinessException("上班时间与班次【"+checkList.get(0).getShiftName()+"】的上班时间冲突！");
                }
            }
        }


        AttendShift checkShift = attendShiftMapper.findByNameAndAttend(attendShift);
        if(checkShift != null && !checkShift.getId().equals(attendShift.getId())){
            throw new BusinessException("班次名称已存在！");
        }

        //if(attendShiftDtos.size() > 0){
        //    List<AttendShiftDto> checkList = attendShiftDtos.stream().filter(shiftDto -> !shiftDto.getId().equals(attendShift.getId())).collect(Collectors.toList());
        //    checkList = checkList.stream().filter(shiftDto -> !shiftDto.getEndTime().equals(attendShift.getStartTime())).collect(Collectors.toList());
        //    if(checkList.size() > 0){
        //        throw new BusinessException("班次时间存在冲突！");
        //    }
        //}
    }

    /**
     * 修改考勤班次
     *
     * @param attendShift 考勤班次
     * @return 结果
     */
    @Override
    public int update(AttendShift attendShift){
        this.validateData(attendShift);
        if(attendShift.getId() == null){
            attendShift.setCreateTime(DateUtils.getNowDate());
            attendShift.setId(idWorker.nextId());
            attendShift.setIsDelete(Constants.N);

//            if(StrUtil.isNotEmpty(attendShift.getCreateBy())){
//                PerOrgRelation perOrgRelation = perOrgRelationMapper.findByPerId(Long.parseLong(attendShift.getCreateBy()));
//                attendShift.setDeptId(perOrgRelation.getOrgId());
//            }
        }else{
            AttendShift dbAttendShift = attendShiftMapper.findById(attendShift.getId());

            if(Constants.N.equals(attendShift.getActive())
                    || !attendShift.getStartTime().equals(dbAttendShift.getStartTime())
                    || (StrUtil.isEmpty(attendShift.getEndTime()) && (StrUtil.isNotEmpty(attendShift.getEndTime())))
                    || (StrUtil.isNotEmpty(attendShift.getEndTime()) && !attendShift.getEndTime().equals(dbAttendShift.getEndTime()))){
                //判断班次是否在使用中，如果在使用中，则不能删除。
                List<AttendShift> attendShifts = attendShiftMapper.findUsedList(Convert.toStrArray(attendShift.getId().toString()));
                if(attendShifts.size() > 0) {
                    throw new BusinessException("【" + attendShift.getShiftName() + "】该班次在使用中，无法作废或者修改上下班时间！");
                }
            }
            BeanUtils.copyPropertiesIgnore(attendShift,dbAttendShift,true);
            BeanUtils.copyPropertiesIgnore(dbAttendShift,attendShift,false);
        }

        if(DictConst.SHIFTTYPE_JCBC.equals(attendShift.getShiftType())) {
            attendShift.setShiftHour(null);
            attendShift.setEndTime(null);
            attendShift.setNeedRest(Constants.N);
        }

        if(Constants.N.equals(attendShift.getNeedRest())){
            attendShift.setRestStart(null);
            attendShift.setRestEnd(null);
            attendShift.setRestHour(0.0D);
        }
        attendShift.setUpdateTime(DateUtils.getNowDate());
        return attendShiftMapper.save(attendShift);
    }

    /**
     * 删除考勤班次对象
     * 
     * @param ids 需要删除的数据ID
     * @return 结果
     */
    @Override
    public int deleteByIds(String ids){
        List<AttendShift> attendShifts = attendShiftMapper.findByArrIds(Convert.toStrArray(ids));
        for (AttendShift attendShift : attendShifts) {
            if(Constants.Y.equals(attendShift.getActive())){
                throw new BusinessException("【" + attendShift.getShiftName() + "】该班次未作废，不能删除！");
            }
        }
        // 判断班次是否在使用中，如果在使用中，则不能删除。
        attendShifts = attendShiftMapper.findUsedList(Convert.toStrArray(ids));
        for (AttendShift attendShift : attendShifts) {
            throw new BusinessException("【" + attendShift.getShiftName() + "】该班次在使用中，不能删除！");
        }

        return attendShiftMapper.deleteByIds(Convert.toStrArray(ids));
    }

}
