package com.bcxin.tenant.bcx.domains.services.impls;

import com.bcxin.tenant.bcx.domains.entities.MetaModuleEntity;
import com.bcxin.tenant.bcx.domains.repositories.MetaModuleRepository;
import com.bcxin.tenant.bcx.domains.repositories.criterias.MetaModuleSearchCriteria;
import com.bcxin.tenant.bcx.domains.services.MetaModuleService;
import com.bcxin.tenant.bcx.domains.services.commands.DeleteDataCommand;
import com.bcxin.tenant.bcx.domains.services.commands.modules.BatchLogicalDeleteModuleCommand;
import com.bcxin.tenant.bcx.domains.services.commands.modules.CreateMetaModuleCommand;
import com.bcxin.tenant.bcx.domains.services.commands.modules.LogicalDeleteModuleCommand;
import com.bcxin.tenant.bcx.domains.services.commands.modules.UpdateMetaModuleCommand;
import com.bcxin.tenant.bcx.infrastructures.EntityCollection;
import com.bcxin.tenant.bcx.infrastructures.TenantContext;
import com.bcxin.tenant.bcx.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.bcx.infrastructures.components.IdWorker;
import com.bcxin.tenant.bcx.infrastructures.UnitWork;
import com.bcxin.tenant.bcx.infrastructures.exceptions.NoFoundTenantException;
import com.bcxin.tenant.bcx.infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.tenant.bcx.infrastructures.utils.StringUtil;
import org.springframework.stereotype.Service;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;

/**
 * 模块元数据服务实现
 */
@Service
public class MetaModuleServiceImpl implements MetaModuleService {
    private final MetaModuleRepository metaModuleRepository;
    private final UnitWork unitWork;
    private final IdWorker idWorker;

    public MetaModuleServiceImpl(MetaModuleRepository metaModuleRepository, UnitWork unitWork, IdWorker idWorker) {
        this.metaModuleRepository = metaModuleRepository;
        this.unitWork = unitWork;
        this.idWorker = idWorker;
    }

    @Override
    public void dispatch(CreateMetaModuleCommand command) {
        command.validate();

        if(StringUtil.isEmpty(command.getName())){
            throw new IllegalArgumentException("模块名称不能为空");
        }

        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if(userModel==null) {
            throw new UnAuthorizedTenantException();
        }

        this.unitWork.executeTran(() -> {
            String id = String.valueOf(this.idWorker.getNextId());
            // 字段值由AOP切面自动填充
            MetaModuleEntity entity = MetaModuleEntity.create(
                    id,
                    command.getProjectId(),
                    command.getName(),
                    userModel.getUserType(),
                    userModel.getOrganizationId(),
                    userModel.getId()
            );
            Timestamp now = new Timestamp(System.currentTimeMillis());
            entity.setCreatedTime(now);
            entity.setLastUpdatedTime(now);
            this.metaModuleRepository.insert(entity);
        });
    }

    @Override
    public void dispatch(UpdateMetaModuleCommand command) {
        command.validate();
        
        this.unitWork.executeTran(() -> {
            // 根据ID查询现有实体（包括已删除的）
            MetaModuleEntity entity = getByIdIncludeDeleted(command.getId());
            if (entity == null) {
                throw new NoFoundTenantException();
            }

            // 检查是否已删除
            if (entity.isDeleted()) {
                throw new NoFoundTenantException();
            }

            // 更新模型字段
            TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
            entity.change(command.getName(), userModel);
            this.metaModuleRepository.update(entity);
        });
    }

    @Override
    public void dispatch(DeleteDataCommand command) {
        command.validate();
        this.unitWork.executeTran(() -> {
            // 根据ID查询现有实体（包括已删除的）
            MetaModuleEntity entity = getByIdIncludeDeleted(command.getId());
            if (entity == null) {
                throw new NoFoundTenantException();
            }

            // 检查是否已删除
            if (entity.isDeleted()) {
                throw new NoFoundTenantException();
            }

            this.metaModuleRepository.delete(command.getId());
        });
    }

    @Override
    public void dispatch(LogicalDeleteModuleCommand command) {
        command.validate();

        this.unitWork.executeTran(() -> {
            // 根据ID查询现有实体（包括已删除的）
            MetaModuleEntity entity = getByIdIncludeDeleted(command.getId());
            if (entity == null) {
                throw new NoFoundTenantException();
            }

            // 检查模块是否已删除
            if (entity.isDeleted()) {
                throw new NoFoundTenantException();
            }

            // 标记为已删除
            TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
            entity.changeMarkAsDeleted(userModel);

            // 保存更新
            this.metaModuleRepository.update(entity);
        });
    }

    @Override
    public void dispatch(BatchLogicalDeleteModuleCommand command) {
        command.validate();

        this.unitWork.executeTran(() -> {
            List<String> ids = command.getIds();
            if (ids == null || ids.isEmpty()) {
                return;
            }
            
            // 1. 批量查询所有entities（一次查询）
            List<MetaModuleEntity> entities = this.metaModuleRepository.getByIds(ids);
            
            // 2. 过滤出未删除的validIds
            List<String> validIds = new ArrayList<>();
            
            for (MetaModuleEntity entity : entities) {
                if (entity != null && !entity.isDeleted()) {
                    validIds.add(entity.getId());
                }
            }
            
            // 如果没有有效的ID需要删除，直接返回
            if (validIds.isEmpty()) {
                return;
            }
            
            // 3. 批量更新所有entities为已删除（一次更新）
            Timestamp now = new Timestamp(System.currentTimeMillis());
            this.metaModuleRepository.batchMarkAsDeleted(validIds, now);
        });
    }

    @Override
    public EntityCollection<MetaModuleEntity> search(MetaModuleSearchCriteria criteria) {
        return this.metaModuleRepository.search(criteria);
    }

    @Override
    public MetaModuleEntity getById(String id) {
        MetaModuleEntity entity = this.metaModuleRepository.getById(id);
        
        // 如果记录已被逻辑删除，返回null
        if (entity != null && entity.isDeleted()) {
            return null;
        }
        
        return entity;
    }

    /**
     * 获取模块实体（包括已删除的记录）
     * 仅供内部使用
     */
    private MetaModuleEntity getByIdIncludeDeleted(String id) {
        return this.metaModuleRepository.getById(id);
    }
}
