package com.zbkj.front.controller.bcx;

import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.StrUtil;
import com.zbkj.common.exception.CrmebException;
import com.zbkj.common.model.bcx.BcxPayOffline;
import com.zbkj.common.model.merchant.MerchantInfo;
import com.zbkj.common.model.order.Order;
import com.zbkj.common.request.bcx.BcxPayOfflineRequest;
import com.zbkj.common.response.bcx.BcxPayOfflineResponse;
import com.zbkj.common.result.CommonResult;
import com.zbkj.common.vo.BcxPayOfflineInfoVo;
import com.zbkj.service.service.MerchantInfoService;
import com.zbkj.service.service.OrderService;
import com.zbkj.service.service.UserService;
import com.zbkj.service.service.bcx.BcxPayOfflineService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.transaction.support.TransactionTemplate;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;

import static com.zbkj.common.constants.PayConstants.PAY_TYPE_OFFLINE;

@Slf4j
@Api(tags = "用户订单线下付款 控制器")
@RestController
@RequestMapping("/api/front/order/offline")
public class BcxPayOfflineController {
    @Resource
    private MerchantInfoService merchantInfoService;
    @Resource
    private OrderService orderService;
    @Resource
    private UserService userService;
    @Resource
    private BcxPayOfflineService bcxPayOfflineService;
    @Resource
    private TransactionTemplate transactionTemplate;

    @ApiOperation("获取订单线下银行付款信息")
    @RequestMapping(value = "/bank/{orderNo}", method = RequestMethod.GET)
    public CommonResult<BcxPayOfflineInfoVo> queryById(@PathVariable String orderNo) {
        Integer uid = userService.getUserIdException();
        Order order = orderService.getByOrderNo(orderNo);
        if (!uid.equals(order.getUid())) {
            throw new CrmebException("非该用户订单禁止查询");
        }
        MerchantInfo merchantInfo = merchantInfoService.getByMerId(order.getMerId());
        if (ObjectUtil.isNull(merchantInfo)) {
            throw new CrmebException("商户信息不存在");
        }
        if (StrUtil.isBlank(merchantInfo.getBankUserName()) || StrUtil.isBlank(merchantInfo.getBankName()) || StrUtil.isBlank(merchantInfo.getBankCard())) {
            throw new CrmebException("商户未配置线下支付信息，请联系客服或管理员");
        }
        BcxPayOfflineInfoVo response = new BcxPayOfflineInfoVo();
        BeanUtils.copyProperties(merchantInfo, response);
        response.setOrderNo(orderNo);
        response.setPayPrice(order.getPayPrice());
        BcxPayOffline offline = bcxPayOfflineService.queryByOrderNo(orderNo);
        if (offline != null) {
            response.setStatus(offline.getStatus());
            response.setUserMsg(offline.getUserMsg());
        }
        return CommonResult.success(response);
    }

    @ApiOperation("订单线下付款提交")
    @RequestMapping(value = "/confirm", method = RequestMethod.POST)
    public CommonResult<BcxPayOfflineResponse> payOfflineConfirm(@RequestBody BcxPayOfflineRequest request) {
        Integer uid = userService.getUserIdException();
        Order order = orderService.getByOrderNo(request.getOrderNo());
        if (!uid.equals(order.getUid())) {
            throw new CrmebException("非该用户订单禁止操作");
        }
        if (order.getPaid()) {
            throw new CrmebException("该订单已支付，请勿重复提交");
        }
        order.setPayType(PAY_TYPE_OFFLINE);
        order.setPayChannel(PAY_TYPE_OFFLINE);

        BcxPayOffline offline = bcxPayOfflineService.queryByOrderNo(request.getOrderNo());
        if (offline != null) {
            if (offline.getStatus() > 0) {
                throw new CrmebException("该订单已提交，请勿重复提交");
            }
        } else {
            offline = new BcxPayOffline();
        }
        BeanUtils.copyProperties(request, offline);
        offline.setStatus(1);
        BcxPayOffline finalOffline = offline;
        transactionTemplate.execute(status -> {
            orderService.updateById(order);
            bcxPayOfflineService.saveOrUpdate(finalOffline);
            return true;
        });
        BcxPayOfflineResponse response = new BcxPayOfflineResponse();
        BeanUtils.copyProperties(offline, response);
        response.setOrderTime(order.getCreateTime());
        response.setPayPrice(order.getPayPrice());
        response.setGainIntegral(order.getGainIntegral());
        response.setPayType(order.getPayType());
        return CommonResult.success(response);
    }
}
