package com.bcxin.platform.service.grant;

import cn.hutool.core.text.StrSpliter;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.bcxin.platform.common.constant.DictConst;
import com.bcxin.platform.common.exception.V5BusinessException;
import com.bcxin.platform.common.utils.DateUtil;
import com.bcxin.platform.common.utils.Result;
import com.bcxin.platform.common.utils.StringUtils;
import com.bcxin.platform.domain.company.PerBaseInfo;
import com.bcxin.platform.domain.grant.ConfigBankOutletsWorkDay;
import com.bcxin.platform.domain.grant.PerBankOutletsAppointment;
import com.bcxin.platform.domain.grant.PerJointlyCard;
import com.bcxin.platform.dto.grant.ConfigBankOutletsDTO;
import com.bcxin.platform.dto.grant.ConfigBankOutletsWorkDayTimePeriodDto;
import com.bcxin.platform.dto.grant.PerJointlyCardDTO;
import com.bcxin.platform.dto.message.MessageDomainDTO;
import com.bcxin.platform.mapper.company.PerBaseInfoMapper;
import com.bcxin.platform.mapper.grant.ConfigBankOutletsWorkDayMapper;
import com.bcxin.platform.mapper.grant.ConfigBankOutletsWorkDayTimePeriodMapper;
import com.bcxin.platform.mapper.grant.PerBankOutletsAppointmentMapper;
import com.bcxin.platform.mapper.grant.PerJointlyCardMapper;
import com.bcxin.platform.service.common.CommonService;
import com.bcxin.platform.util.BH_AESUtil;
import com.bcxin.platform.util.DateUtils;
import com.bcxin.platform.util.ObjectUtils;
import com.bcxin.platform.util.PageInfoUtils;
import com.bcxin.platform.util.constants.CommonConst;
import com.bcxin.platform.util.constants.Constants;
import com.bcxin.platform.util.constants.DictMessageTypeConst;
import com.bcxin.platform.util.constants.MsgConst;
import com.bcxin.platform.util.excel.ExcelUtil;
import com.bcxin.platform.util.http.HuaWeiSmsContent;
import com.bcxin.platform.util.map.GaodeMapUtil;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.util.StringUtil;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.poi.hssf.usermodel.*;
import org.apache.poi.hssf.util.HSSFColor;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.IndexedColors;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.*;

/**
 * 个人联名卡服务
 *
 * @author llc
 * @date 2020-07-21
 */
@Service
@Transactional("transactionManager")
public class PerJointlyCardServiceImpl implements PerJointlyCardService {

    @Resource
    public PerJointlyCardMapper perJointlyCardMapper;

    @Resource
    public ConfigBankOutletsWorkDayMapper configBankOutletsWorkDayMapper;


    @Resource
    public ConfigBankOutletsWorkDayTimePeriodMapper configBankOutletsWorkDayTimePeriodMapper;


    @Resource
    public PerBankOutletsAppointmentMapper perBankOutletsAppointmentMapper;


    @Resource
    public CommonService commonService;

    @Resource
    public PerBaseInfoMapper perBaseInfoMapper;


    /**
     * 获取企业人员开通联名卡服务情况
     *
     * @param dto
     * @return
     * @author llc
     * @date 2020-07-21
     */
    @Override
    public Result getPerOpenJointlyInfo(PerJointlyCardDTO dto) throws V5BusinessException {
        if (dto.getComId() == null) {
            throw new V5BusinessException(Result.ERROR, "企业ID不能为空");
        }
        /*** 未开通人数 ***/
        List<String> perIdList = perJointlyCardMapper.getNoOpenJointlyPerIdList(dto.getComId());

        /*** 企业已开通联名卡人数 ***/
        Integer openJointlyPerCount = perJointlyCardMapper.getOpenJointlyPerCount(dto.getComId());
        Map<String, Object> map = new HashMap<>();
        map.put("openJointlyPerCount", openJointlyPerCount);
        map.put("noOpenJointlyPerCount", perIdList == null ? 0 : perIdList.size());
        return Result.success(Result.SUCCESS_QUERY_MSG, map);
    }

    /**
     * 获取企业未开通联名卡人员列表
     *
     * @param dto
     * @author llc
     * @date 2020-07-21
     */
    @Override
    public Result getNoOpenJointlyPerList(PerJointlyCardDTO dto) throws V5BusinessException {
        if (dto.getPageNumber() == null) {
            return Result.fail("第几页不能为空");
        }
        if (dto.getPageSize() == null) {
            return Result.fail("每页多少条不能为空");
        }
        if (dto.getDepartId() != null && dto.getDepartId() == -1L) {
            dto.setDepartId(dto.getComId());
        }
        /*** 多部门选择 ***/
        if (StringUtils.isNotEmpty(dto.getDepartIds())) {
            /*** 因为选择企业传的是-1，需要替换成企业ID ***/
            dto.getDepartIds().replaceAll("-1", dto.getComId().toString());
            List<String> departIdStrList = StrSpliter.split(dto.getDepartIds(), CommonConst.COMMA, true, true);
            List<Long> departIdList = new ArrayList<>();
            for (String departIdStr : departIdStrList) {
                departIdList.add(Long.parseLong(departIdStr));
            }
            dto.setDepartIdList(departIdList);
        }
        PageHelper.startPage(dto.getPageNumber(), dto.getPageSize());
        /*** 获取未开通人员列表 ***/
        List<Map<String, String>> list = perJointlyCardMapper.getNoOpenJointlyPerList(dto);
        PageInfoUtils pageInfo = new PageInfoUtils(list);
        return Result.success(Result.SUCCESS_QUERY_MSG, pageInfo);
    }

    /**
     * 导出企业未开通联名卡人员
     *
     * @param dto
     * @author llc
     * @date 2020-09-27
     */
    @Override
    public Result exportNoOpenJointlyPer(PerJointlyCardDTO dto, HttpServletResponse response)
            throws V5BusinessException {
        OutputStream os = null;
        HSSFWorkbook wb = new HSSFWorkbook();
        try {
            String fileName = "_" + "开通人员" + DateUtils.getDate("yyyyMMddHHmmss") + ".xls";
            response.setContentType("application/vnd.ms-excel");
            response.setHeader("Content-Disposition",
                    "attachment;fileName=" + new String(fileName.getBytes("gb2312"), "ISO8859-1"));
            os = response.getOutputStream();
            HSSFSheet sheet1 = wb.createSheet("未开通人员");

            if (dto.getDepartId() != null && dto.getDepartId() == -1L) {
                dto.setDepartId(dto.getComId());
            }
            /*** 多部门选择 ***/
            if (StringUtils.isNotEmpty(dto.getDepartIds())) {
                /*** 因为选择企业传的是-1，需要替换成企业ID ***/
                dto.getDepartIds().replaceAll("-1", dto.getComId().toString());
                List<String> departIdStrList = StrSpliter.split(dto.getDepartIds(), CommonConst.COMMA, true, true);
                List<Long> departIdList = new ArrayList<>();
                for (String departIdStr : departIdStrList) {
                    departIdList.add(Long.parseLong(departIdStr));
                }
                dto.setDepartIdList(departIdList);
            }
            /*** 获取未开通人员列表 ***/
            List<Map<String, String>> list = perJointlyCardMapper.getNoOpenJointlyPerList(dto);

            /*** 填充文本样式 ***/
            CellStyle stringStyle = ExcelUtil.getStringStyle(wb);

            /**** 生成sheet1的内容 ***/
            HSSFFont titleFont = wb.createFont();
            HSSFCellStyle titleStyle = wb.createCellStyle();
            titleFont.setFontHeightInPoints((short) 11);
            titleFont.setColor(HSSFColor.BLUE.index);
            titleFont.setFontName("黑体");
            titleStyle.setAlignment(HSSFCellStyle.ALIGN_CENTER);// 水平居中
            titleStyle.setVerticalAlignment(HSSFCellStyle.VERTICAL_CENTER);// 垂直居中
            titleStyle.setFont(titleFont);
            titleStyle.setWrapText(true);
            HSSFDataFormat format = wb.createDataFormat();
            titleStyle.setDataFormat(format.getFormat("@"));

            /*** 填充第一行数据 ***/
            HSSFRow rowFirst = sheet1.createRow(0);
            String[] firstArr = {"序号", "姓名", "身份证号", "手机号", "性别","部门", "人员类别", "认证状态", "资格证号", "发证日期", "是否采集证件照"};
            int column;
            for (int i = 0; i < firstArr.length; i++) {
                HSSFCell cell = rowFirst.createCell(i); // 获取第二行的每个单元格
                if (i == 0) {
                    column = 1500;
                } else if (i == 2) {
                    column = 5000;
                } else if (i == 5) {
                    column = 10000;
                } else {
                    column = 4000;
                }
                sheet1.setColumnWidth(i, column); // 设置每列的列宽
                sheet1.setDefaultColumnStyle(i, stringStyle); // 设置单元格格式 --文本格式
                cell.setCellStyle(titleStyle); // 文本格式
            }

            /*** 填充第一行数据 ***/
            for (int i = 0; i < firstArr.length; i++) {
                HSSFCell cell = rowFirst.createCell(i); // 获取第一行的每个单元格
                cell.setCellValue(firstArr[i]);
                cell.setCellStyle(titleStyle);
            }

            /*** 字体颜色样式 ***/
            CellStyle greenStyle = ExcelUtil.getColorStyle(wb, IndexedColors.SEA_GREEN.index);// 青绿色样式
            CellStyle redStyle = ExcelUtil.getColorStyle(wb, IndexedColors.RED.index); // 红色样式
            HSSFRow row = null;
            int rowIndex = 1;
            for (Map<String, String> result : list) {
                row = sheet1.createRow(rowIndex);
                HSSFCell cell0 = row.createCell(0);
                cell0.setCellValue(rowIndex);
                HSSFCell cell1 = row.createCell(1);
                cell1.setCellValue(result.get("name")); // 姓名
                HSSFCell cell2 = row.createCell(2);
                cell2.setCellValue(result.get("idCardNo")); // 身份证
                HSSFCell cell3 = row.createCell(3);
                cell3.setCellValue(result.get("mobilePhone")); // 手机号
                HSSFCell cell4 = row.createCell(4);
                cell4.setCellValue(result.get("sex")); // 性别
                HSSFCell cell5 = row.createCell(5);
                cell5.setCellValue(result.get("departName")); // 部门名称
                HSSFCell cell6 = row.createCell(6);
                cell6.setCellValue(result.get("perType")); // 人员类别
                HSSFCell cell7 = row.createCell(7);
                cell7.setCellValue(result.get("authStatus")); // 认证状态
                if (ObjectUtil.equal(result.get("authStatus"), "已认证")) {
                    cell7.setCellStyle(greenStyle);
                }
                HSSFCell cell8 = row.createCell(8);
                cell8.setCellValue(result.get("cerNo")); // 资格证号
                HSSFCell cell9 = row.createCell(9);
                cell9.setCellValue(result.get("licenceDate")); // 发证日期
                HSSFCell cell10 = row.createCell(10);
                cell10.setCellValue(result.get("isExistsCerPhoto")); // 是否存在证件照
                rowIndex++;
            }
            wb.write(os);
            os.close();
            wb.close();
        } catch (Exception e) {
            throw new V5BusinessException(Result.ERROR, e.getMessage());
        } finally {
            try {
                os.close();
                wb.close();
            } catch (IOException e1) {
            }
        }
        return Result.success("导出未开通人员文件失败");
    }

    /**
     * 获取企业已开通联名卡人员列表
     *
     * @param dto
     * @author llc
     * @date 2020-07-21
     */
    @Override
    public Result getOpenJointlyPerList(PerJointlyCardDTO dto) throws V5BusinessException {
        if (dto.getPageNumber() == null) {
            return Result.fail("第几页不能为空");
        }
        if (dto.getPageSize() == null) {
            return Result.fail("每页多少条不能为空");
        }
        if (dto.getDepartId() != null && dto.getDepartId() == -1L) {
            dto.setDepartId(dto.getComId());
        }

        /*** 多部门选择 ***/
        if (StringUtils.isNotEmpty(dto.getDepartIds())) {
            /*** 因为选择企业传的是-1，需要替换成企业ID ***/
            dto.getDepartIds().replaceAll("-1", dto.getComId().toString());
            List<String> departIdStrList = StrSpliter.split(dto.getDepartIds(), CommonConst.COMMA, true, true);
            List<Long> departIdList = new ArrayList<>();
            for (String departIdStr : departIdStrList) {
                departIdList.add(Long.parseLong(departIdStr));
            }
            dto.setDepartIdList(departIdList);
        }
        PageHelper.startPage(dto.getPageNumber(), dto.getPageSize());
        /*** 获取已开通人员列表 ***/
        List<Map<String, String>> list = perJointlyCardMapper.getOpenJointlyPerList(dto);
        PageInfoUtils pageInfo = new PageInfoUtils(list);
        return Result.success(Result.SUCCESS_QUERY_MSG, pageInfo);
    }

    /**
     * 导出企业已开通联名卡人员
     *
     * @param dto
     * @author llc
     * @date 2020-07-22
     */
    @Override
    public Result exportOpenJointlyPer(PerJointlyCardDTO dto, HttpServletResponse response)
            throws V5BusinessException {
        OutputStream os = null;
        HSSFWorkbook wb = new HSSFWorkbook();
        try {
            String fileName = "_" + "开通人员" + DateUtils.getDate("yyyyMMddHHmmss") + ".xls";
            response.setContentType("application/vnd.ms-excel");
            response.setHeader("Content-Disposition",
                    "attachment;fileName=" + new String(fileName.getBytes("gb2312"), "ISO8859-1"));
            os = response.getOutputStream();
            HSSFSheet sheet1 = wb.createSheet("开通人员");

            if (dto.getDepartId() != null && dto.getDepartId() == -1L) {
                dto.setDepartId(dto.getComId());
            }
            /*** 多部门选择 ***/
            if (StringUtils.isNotEmpty(dto.getDepartIds())) {
                /*** 因为选择企业传的是-1，需要替换成企业ID ***/
                dto.getDepartIds().replaceAll("-1", dto.getComId().toString());
                List<String> departIdStrList = StrSpliter.split(dto.getDepartIds(), CommonConst.COMMA, true, true);
                List<Long> departIdList = new ArrayList<>();
                for (String departIdStr : departIdStrList) {
                    departIdList.add(Long.parseLong(departIdStr));
                }
                dto.setDepartIdList(departIdList);
            }
            /*** 获取已开通人员列表 ***/
            List<Map<String, String>> list = perJointlyCardMapper.getOpenJointlyPerList(dto);

            /*** 填充文本样式 ***/
            CellStyle stringStyle = ExcelUtil.getStringStyle(wb);

            /**** 生成sheet1的内容 ***/
            HSSFFont titleFont = wb.createFont();
            HSSFCellStyle titleStyle = wb.createCellStyle();
            titleFont.setFontHeightInPoints((short) 11);
            titleFont.setColor(HSSFColor.BLUE.index);
            titleFont.setFontName("黑体");
            titleStyle.setAlignment(HSSFCellStyle.ALIGN_CENTER);// 水平居中
            titleStyle.setVerticalAlignment(HSSFCellStyle.VERTICAL_CENTER);// 垂直居中
            titleStyle.setFont(titleFont);
            titleStyle.setWrapText(true);
            HSSFDataFormat format = wb.createDataFormat();
            titleStyle.setDataFormat(format.getFormat("@"));

            /*** 填充第一行数据 ***/
            HSSFRow rowFirst = sheet1.createRow(0);
            String[] firstArr = {"序号", "姓名", "身份证号", "手机号", "部门", "银行卡号", "账户状态", "账户类型", "开户行", "开户行行号", "激活时间", "注销时间", "注销原因"};
            int column;
            for (int i = 0; i < firstArr.length; i++) {
                HSSFCell cell = rowFirst.createCell(i); // 获取第二行的每个单元格
                if (i == 0) {
                    column = 1500;
                } else if (i == 2 || i == 5 || i == 9 || i == 10 || i == 11) {
                    column = 5000;
                } else if (i == 4 || i == 8 || i == 12) {
                    column = 10000;
                } else {
                    column = 4000;
                }
                sheet1.setColumnWidth(i, column); // 设置每列的列宽
                sheet1.setDefaultColumnStyle(i, stringStyle); // 设置单元格格式 --文本格式
                cell.setCellStyle(titleStyle); // 文本格式
            }

            /*** 填充第一行数据 ***/
            for (int i = 0; i < firstArr.length; i++) {
                HSSFCell cell = rowFirst.createCell(i); // 获取第一行的每个单元格
                cell.setCellValue(firstArr[i]);
                cell.setCellStyle(titleStyle);
            }

            /*** 字体颜色样式 ***/
            CellStyle greenStyle = ExcelUtil.getColorStyle(wb, IndexedColors.SEA_GREEN.index);// 青绿色样式
            CellStyle redStyle = ExcelUtil.getColorStyle(wb, IndexedColors.RED.index); // 红色样式
            HSSFRow row = null;
            int rowIndex = 1;
            for (Map<String, String> result : list) {
                row = sheet1.createRow(rowIndex);
                HSSFCell cell0 = row.createCell(0);
                cell0.setCellValue(rowIndex);
                HSSFCell cell1 = row.createCell(1);
                cell1.setCellValue(result.get("name")); // 姓名
                HSSFCell cell2 = row.createCell(2);
                cell2.setCellValue(result.get("idCardNo")); // 身份证
                HSSFCell cell3 = row.createCell(3);
                cell3.setCellValue(result.get("mobilePhone")); // 手机号
                HSSFCell cell4 = row.createCell(4);
                cell4.setCellValue(result.get("departName")); // 部门名称
                HSSFCell cell5 = row.createCell(5);
                cell5.setCellValue(result.get("bankCardNo")); // 银行卡号
                HSSFCell cell6 = row.createCell(6);
                cell6.setCellValue(result.get("bankAccountStatus")); // 账户状态
                if (ObjectUtil.equal(result.get("bankAccountStatus"), "正常")) {
                    cell6.setCellStyle(greenStyle);
                } else if (ObjectUtil.equal(result.get("bankAccountStatus"), "注销")) {
                    cell6.setCellStyle(redStyle);
                }
                HSSFCell cell7 = row.createCell(7);
                cell7.setCellValue(result.get("bankAccountType")); // 银行账户类型
                HSSFCell cell8 = row.createCell(8);
                cell8.setCellValue(result.get("bankOutlets")); // 开户行
                HSSFCell cell9 = row.createCell(9);
                cell9.setCellValue(result.get("bankNo")); // 开户行行号
                HSSFCell cell10 = row.createCell(10);
                cell10.setCellValue(result.get("activeTime")); // 激活时间
                HSSFCell cell11 = row.createCell(11);
                cell11.setCellValue(result.get("cancelTime"));// 注销时间
                HSSFCell cell12 = row.createCell(12);
                cell12.setCellValue(result.get("cancelReason"));// 注销原因
                rowIndex++;
            }
            wb.write(os);
            os.close();
            wb.close();
        } catch (Exception e) {
            throw new V5BusinessException(Result.ERROR, e.getMessage());
        } finally {
            try {
                os.close();
                wb.close();
            } catch (IOException e1) {
            }
        }
        return Result.success("导出开通人员文件失败");
    }


    /**
     * 批量提醒开通
     *
     * @param dto
     * @return
     * @author llc
     * @date 2020-07-21
     */
    @Override
    public Result batchNotice(PerJointlyCardDTO dto) throws V5BusinessException {
        if (StringUtil.isEmpty(dto.getIsCheckAll())) {
            return Result.fail("选中标识不能为空");
        } else {
            if (ObjectUtils.equals(dto.getIsCheckAll(), CommonConst.N) && StringUtil.isEmpty(dto.getPerIds())) {
                return Result.fail("必须选择发送人员");
            }
        }
        List<String> perIdList = Lists.newArrayList();
        if (dto.getIsCheckAll().equals(CommonConst.Y)) {
            perIdList = perJointlyCardMapper.getNoOpenJointlyPerIdList(dto.getComId());
            if (perIdList == null) {
                return Result.fail("所有人均已开通，无须提醒");
            }
        } else {
            perIdList = StrSpliter.split(dto.getPerIds(), CommonConst.COMMA, true, true);
        }
        if (perIdList.size() == 0) {
        }
        String ids = Joiner.on(",").skipNulls().join(perIdList);
        /*** 微服务调用获取推送消息 ***/
        MessageDomainDTO messageDomainDTO = new MessageDomainDTO();
        messageDomainDTO.setIds(ids);
        messageDomainDTO.setTitle(MsgConst.TYPE_010302_OPEN_JOINTLY_CARD_REMINDER_TITLE);
        messageDomainDTO.setContent(MsgConst.TYPE_010302_OPEN_JOINTLY_CARD_REMINDER_CONTENT);
        messageDomainDTO.setMessageType(DictMessageTypeConst.MESSAGETYPE_010302);
        messageDomainDTO.setPlatId(10L);
        messageDomainDTO.setWay("3");
        //messageFeignService.sendMessage(messageDomainDTO);
        return Result.success(Result.SUCCESS_MSG);
    }

    /**
     * 批量同步联名卡
     *
     * @param list
     * @return
     * @author llc
     * @date 2020-07-21
     */
    @Override
    public Result batchSyncPerJointly(List<Map<String, String>> list) throws V5BusinessException {

        List<PerJointlyCard> perJointlyCardList = new ArrayList<>();
        Date now = new Date();
        for (Map<String, String> map : list) {
            PerJointlyCard perJointlyCard = new PerJointlyCard();
            perJointlyCard.setPerId(Long.parseLong(map.get("perId")));
            perJointlyCard.setCreateTime(now);
            perJointlyCard.setUpdateTime(now);
            perJointlyCard.setIsActive(map.get("isActive"));
            perJointlyCard.setActiveTime(map.get("activeTime"));
            perJointlyCard.setOpenCardTime(map.get("openCardTime"));
            perJointlyCard.setBankCode(map.get("bankCode"));
            perJointlyCard.setBankOutlets(map.get("bankOutlets"));
            perJointlyCard.setBankAccountType(map.get("bankAccountType"));
            perJointlyCard.setBankNo(map.get("bankNo"));
            perJointlyCard.setBankCardNo(map.get("bankCardNo"));
            perJointlyCard.setBankAccountStatus(map.get("bankAccountStatus"));
            perJointlyCard.setCancelTime(map.get("cancelTime"));
            perJointlyCard.setCancelReason(map.get("cancelReason"));
            perJointlyCardList.add(perJointlyCard);
        }
        if (perJointlyCardList.size() > 0) {
            perJointlyCardMapper.batchUpdatePerJointlyCard(perJointlyCardList);
        }

        return Result.success(Result.SUCCESS_QUERY_MSG);
    }

    /**
     * 获取企业人员开通联名卡服务情况
     *
     * @param perId
     * @return
     * @author llc
     * @date 2020-07-29
     */
    @Override
    public Result getPerJointlyByPerId(Long perId) throws V5BusinessException {
        if (perId == null) {
            return Result.fail("人员ID不能为空");
        }
        PerJointlyCard perJointlyCard = perJointlyCardMapper.getPerJointlyCard(perId);
        return Result.success(Result.SUCCESS_QUERY_MSG, perJointlyCard);
    }

    /**
     * 获取保安员联名卡信息
     *
     * @param dto
     * @return
     * @author llc
     * @date 2020-07-30
     */
    @Override
    public Result getPerJointlyCard(PerJointlyCardDTO dto) throws V5BusinessException {
        PerJointlyCard perJointlyCard = perJointlyCardMapper.getPerJointlyCard(dto.getPerId());
        if (perJointlyCard != null && ObjectUtil.equal(perJointlyCard.getBankAccountStatus(), CommonConst.N)) {
            return Result.success(Result.SUCCESS_QUERY_MSG);
        }
        return Result.success(Result.SUCCESS_QUERY_MSG, perJointlyCard);
    }


    /**
     * 获取银行网点列表
     *
     * @param dto
     * @return
     * @author llc
     * @date 2020-07-31
     */
    @Override
    public Result getBankOutletsList(PerJointlyCardDTO dto) throws V5BusinessException {

        List<ConfigBankOutletsDTO> bankList = commonService.getConfigBankOutlets();

        /*** 人员经纬度有正常上传才排序 ***/
        if (StringUtil.isNotEmpty(dto.getPerLongitude()) && StringUtil.isNotEmpty(dto.getPerLatitude())
                && !ObjectUtil.equal(dto.getPerLongitude(), dto.getPerLatitude())) {
            /*** 如果存在银行信息 ***/
            if (bankList != null && bankList.size() > 0) {
                StringBuilder lonLats = new StringBuilder();
                for (ConfigBankOutletsDTO bankDto : bankList) {
                    lonLats.append(bankDto.getAddressLongitude()).append(CommonConst.COMMA).
                            append(bankDto.getAddressLatitude()).append(CommonConst.VIRGULE);
                }
                String lonLatsStr = lonLats.toString();
                lonLatsStr = lonLatsStr.substring(0, lonLatsStr.length() - 1);
                String endLonLat = dto.getPerLongitude() + CommonConst.COMMA + dto.getPerLatitude();
                /*** 调用高德地图api获取位置列表 ***/
                String resultStr = GaodeMapUtil.getDistance(lonLatsStr, endLonLat);
                /*** 如果正常返回数据则进行排序***/
                if (StringUtil.isNotEmpty(resultStr) && !ObjectUtil.equal(resultStr, Result.ERROR)) {
                    /***  银行网点的数组 ***/
                    List<Map> list = JSON.parseArray(resultStr, Map.class);

                    for (int j = 0; j < bankList.size(); j++) {
                        for (int i = 0; i < list.size(); i++) {
                            if (i == j) {
                                bankList.get(j).setDistance(Long.parseLong(list.get(i).get("distance").toString()));
                                break;
                            }
                        }
                    }
                }

                /*** 进行网点的距离排序 距离从小到大***/
                bankList.sort((x, y) -> Long.compare(x.getDistance(), y.getDistance()));//这方法需要jdk1.8以上
            }
        }

        return Result.success(Result.SUCCESS_QUERY_MSG, bankList);
    }


    /**
     * 获取银行网点工作日列表
     *
     * @param configBankOutletsWorkDay
     * @return
     * @author llc
     * @date 2020-08-21
     */
    @Override
    public Result getBankOutletsWorkDayList(ConfigBankOutletsWorkDay configBankOutletsWorkDay) throws V5BusinessException {
        if (configBankOutletsWorkDay.getBankOutletsId() == null) {
            return Result.fail("银行网点ID不能为空");
        }

        /*** 预约的时间段从第二天开始计算往后推迟2周 ***/

        String startDate = DateUtil.getDateStrAfterToday(1);

        /*** 如果预约开始时间早于 2020-08-24，则预约开始时间为2020-08-24 ***/
        if (DateUtils.dateCompare(startDate, "2020-08-24")) {
            startDate = "2020-08-24";
        }
        String endDate = DateUtil.getDateStrAfterToday(7);

        List<Map<String, Object>> list = configBankOutletsWorkDayMapper.getBankOutletsWorkDayList(configBankOutletsWorkDay.getBankOutletsId(), startDate, endDate);

        return Result.success(Result.SUCCESS_QUERY_MSG, list);
    }


    /**
     * 获取银行网点工作日时间段列表
     *
     * @param dto
     * @return
     * @author llc
     * @date 2020-08-21
     */
    @Override
    public Result getBankOutletsWorkDayTimePeriodList(ConfigBankOutletsWorkDayTimePeriodDto dto) throws V5BusinessException {
        if (dto.getBankOutletsWorkDayId() == null) {
            return Result.fail("银行网点工作日ID不能为空");
        }
        if (StringUtil.isEmpty(dto.getAppointmentDate())) {
            return Result.fail("预约日期不能为空");
        }

        List<Map<String, Object>> list = configBankOutletsWorkDayTimePeriodMapper.getBankOutletsWorkDayTimePeriodMList(dto.getBankOutletsWorkDayId(), dto.getAppointmentDate());

        return Result.success(Result.SUCCESS_QUERY_MSG, list);
    }

    /**
     * 预约银行网点
     *
     * @param perBankOutletsAppointment
     * @return
     * @author llc
     * @date 2020-08-21 16:05
     */
    @Override
    public Result reserveBankOutlets(PerBankOutletsAppointment perBankOutletsAppointment) throws V5BusinessException {
        if (perBankOutletsAppointment.getBankOutletsWorkDayTimePeriodId() == null) {
            return Result.fail("请选择银行网点工作日时间段");
        }
        if (perBankOutletsAppointment.getAppointmentDate() == null) {
            return Result.fail("预约日期不能为空");
        }

        String currentTime = DateUtils.formatDate(perBankOutletsAppointment.getCreateTime(), "HH:mm");

        /*** 预约时间不能超过 16:30 **/
        if (currentTime.compareTo("16:30") > 0) {
            return Result.fail("当前已超过今日预约时间，请明日再预约！");
        }

        /*** 校验预约人数是否已满 ***/
        // TODO: 2020/8/21 0021 redis加锁校验

        String isCanAppointment = perBankOutletsAppointmentMapper.checkIsCanAppointment(perBankOutletsAppointment);
        if (StringUtil.isNotEmpty(isCanAppointment) && isCanAppointment.equals(CommonConst.N)) {
            return Result.fail("对不起，预约人数已满，请选择其他网点或者其他时间段预约！");
        }

        PerBankOutletsAppointment appointment = perBankOutletsAppointmentMapper.getPerBankOutletsAppointment(perBankOutletsAppointment.getPerId());
        if (appointment != null) {
            return Result.fail("对不起，您已有预约信息，无需再预约！");
        }
        Map<String, String> map = configBankOutletsWorkDayTimePeriodMapper.getBankOutletsappointmentInfo(perBankOutletsAppointment.getBankOutletsWorkDayTimePeriodId());

        PerBaseInfo perBaseInfo = perBaseInfoMapper.selectPerBaseInfoById(perBankOutletsAppointment.getPerId());
        /*** 如果存在手机号码则发送预约短信 ***/
        if (StringUtil.isNotEmpty(perBaseInfo.getMobilePhone())) {
            HuaWeiSmsContent huaWeiSmsContent = new HuaWeiSmsContent();
            huaWeiSmsContent.setPlatform(Constants.ENVI);
            huaWeiSmsContent.setSmsCode(DictConst.SMSCODE_B13);
            huaWeiSmsContent.setMobile(perBaseInfo.getMobilePhone());
            huaWeiSmsContent.setParams(JSON.toJSONString(new String[]{map.get("bankOutlets"),
                    DateUtils.formatDate(perBankOutletsAppointment.getAppointmentDate()) + " " + map.get("startTime") + "~" + map.get("endTime"),
                    map.get("bankAddress")}));
            commonService.sendSMS(huaWeiSmsContent);
        }
        perBankOutletsAppointmentMapper.insertPerBankOutletsAppointment(perBankOutletsAppointment);

        /*** 发送站内信 ***/
        String tilte = MsgConst.TYPE_010401_RESERVE_BANK_TITLE;// 标题
        String content = MsgConst.TYPE_010401_RESERVE_BANK_CONTENT; // 通知内容
        content = content.replace("{bankOutlets}", map.get("bankOutlets"))
                .replace("{timeArea}", DateUtils.formatDate(perBankOutletsAppointment.getAppointmentDate()) + " " + map.get("startTime") + "~" + map.get("endTime"))
                .replace("{bankAddress}", map.get("bankAddress"));
        commonService.sendMessageToApp(tilte, content, DictMessageTypeConst.MESSAGETYPE_010401, perBankOutletsAppointment.getPerId().toString(), null, null);

        return Result.success(Result.SUCCESS_MSG);
    }

    /**
     * 查询个人预约记录
     *
     * @param perBankOutletsAppointment
     * @return
     * @author llc
     * @date 2020-08-21
     */
    @Override
    public Result getPerBankOutletsAppointment(PerBankOutletsAppointment perBankOutletsAppointment) throws V5BusinessException {

        Map<String, String> map = perBankOutletsAppointmentMapper.getPerBankOutletsAppointmentByPerId(perBankOutletsAppointment.getPerId());
        if(map == null){
            return Result.success(Result.SUCCESS_QUERY_MSG);
        }
        //预约时间
        Date eTime = DateUtil.convertStringToDate(map.get("appointmentDate")+" "+map.get("endTime"));
        //当前时间大于预约时间
        if(StringUtil.isNotEmpty(map.get("appointmentDate"))
                &&StringUtil.isNotEmpty(map.get("endTime"))
                &&System.currentTimeMillis()>eTime.getTime()){
            //并且联名卡是注销状态，就不返回预约信息，以便前端可以重新走预约业务
            PerJointlyCard perJointlyCard = perJointlyCardMapper.getPerJointlyCard(perBankOutletsAppointment.getPerId());
            if (perJointlyCard != null && ObjectUtil.equal(perJointlyCard.getBankAccountStatus(), CommonConst.N)) {
                return Result.success(Result.SUCCESS_QUERY_MSG);
            }
        }
        return Result.success(Result.SUCCESS_QUERY_MSG, map);
    }


    /**
     * 取消银行网点预约
     *
     * @param perBankOutletsAppointment
     * @return
     * @author llc
     * @date 2020-08-21
     */
    @Override
    public Result cancelReserveBankOutlets(PerBankOutletsAppointment perBankOutletsAppointment) throws V5BusinessException {
        perBankOutletsAppointmentMapper.deletePerBankOutletsAppointment(perBankOutletsAppointment);
        return Result.success(Result.SUCCESS_MSG);
    }

    /**
     * <b> 获取联名卡申请、查询地址 </b>
     *
     * @author ZXF
     * @create 2020/09/04 0004 10:42
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result getJointlyCardApplyUrl(Long perId) throws UnsupportedEncodingException {
        Map<String,String> map = perBaseInfoMapper.findPerDecByperIdForBHAPI(perId);
        if (map == null) {
            return Result.fail("未找到人员或企业");
        }
        if (StringUtil.isEmpty(map.get("EntAddress"))||StringUtil.isEmpty(map.get("ZipCode"))) {
            return Result.fail("地址或邮编未完善");
        }
        String applyUrl = commonService.getSystemConfig(CommonConst.BOHAI_JOINTLYCARD_APPLY_URL);
        String queryUrl = commonService.getSystemConfig(CommonConst.BOHAI_JOINTLYCARD_QUERY_URL);
        String aesKey = commonService.getSystemConfig(CommonConst.BOHAI_JOINTLYCARD_APPLY_AESKEY);
        String aesStr = BH_AESUtil.AES_Encrypt(aesKey, JSON.toJSONString(map));
        Map paramMap = Maps.newHashMap();
        paramMap.put("applyUrl", applyUrl + URLEncoder.encode(aesStr, "UTF-8"));
        paramMap.put("queryUrl", queryUrl);
        return Result.success(Result.SUCCESS_MSG, paramMap);
    }

    /**
     * <b> 获取联名卡申请条件 </b>
     *
     * @author ZXF
     * @create 2020/09/09 0009 13:49
     * @version
     * @注意事项 </b>
     */
    @Override
    public Result getJointlyCardApplyConditions(Long perId, Long comId) throws UnsupportedEncodingException {
        int isZZ = 1;//是否在职
        int isBGD = 1;//注册完整地址到门牌号
        int isCZZP = 0;//是否存在照片

        String address = null;
        PerBaseInfo perBaseInfo = perBaseInfoMapper.selectPerBaseInfoById(perId);
        if (StringUtils.isNotEmpty(perBaseInfo.getIdCardNo())) {
        }
        Map<String, String> map = perBaseInfoMapper.findPerDecByperIdForBHAPI(perId);
        if (map == null) {
            isZZ = 0;
            isBGD = 0;
        } else {
            address = map.get("EntAddress");
            if (StringUtils.isEmpty(map.get("EntName")) || StringUtils.isEmpty(map.get("EntAddress")) || StringUtils.isEmpty(map.get("EntPhone"))||StringUtil.isEmpty(map.get("ZipCode"))) {
                isBGD = 0;
            }
        }
        Map paramMap = Maps.newHashMap();
        paramMap.put("isZZ", isZZ);
        paramMap.put("isBGD", isBGD);
        paramMap.put("isCZZP", isCZZP);
        paramMap.put("comAddress", address);
        return Result.success(Result.SUCCESS_MSG, paramMap);
    }


}