package com.bcxin.runtime.domain.syncs.services.impls;

import com.bcxin.runtime.domain.constants.FieldNames;
import com.bcxin.runtime.domain.snapshoots.JdbcMapSnapshot;
import com.bcxin.runtime.domain.snapshoots.TableMapSnapshot;
import com.bcxin.runtime.domain.syncs.commands.CreateDataSyncCommand;
import com.bcxin.runtime.domain.syncs.commands.CreateSyncFileCommand;
import com.bcxin.runtime.domain.syncs.commands.results.CreateDataSyncCommandResult;
import com.bcxin.runtime.domain.syncs.components.DbSyncExecutor;
import com.bcxin.runtime.domain.syncs.dtos.FileItemRequestDto;
import com.bcxin.runtime.domain.syncs.dtos.FtpConfigInfoDto;
import com.bcxin.runtime.domain.syncs.entities.DataSyncMapEntity;
import com.bcxin.runtime.domain.syncs.entities.DataSyncLogEntity;
import com.bcxin.runtime.domain.syncs.entities.FileSyncQueueEntity;
import com.bcxin.runtime.domain.syncs.enums.ProcessedStatus;
import com.bcxin.runtime.domain.syncs.ftp.FtpUploadTask;
import com.bcxin.runtime.domain.syncs.repositories.DataSyncLogRepository;
import com.bcxin.runtime.domain.syncs.repositories.DataSyncMapRepository;
import com.bcxin.runtime.domain.syncs.repositories.FileSyncQueueRepository;
import com.bcxin.runtime.domain.syncs.services.DataSyncService;
import com.bcxin.saas.core.AppConfigProperty;
import com.bcxin.saas.core.components.HttpRequestProvider;
import com.bcxin.saas.core.components.JsonProvider;
import com.bcxin.saas.core.exceptions.SaasBadException;
import com.bcxin.saas.core.exceptions.SaasNofoundException;
import com.bcxin.saas.core.utils.ExceptionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.*;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class DataSyncServiceImpl implements DataSyncService {
    private static Logger logger = LoggerFactory.getLogger(DataSyncServiceImpl.class);
    private static ExecutorService newFixedThreadPool = Executors.newFixedThreadPool(15);
    private final DataSyncMapRepository dataMapSyncRepository;
    private final DataSyncLogRepository dataSyncLogRepository;
    private final JsonProvider jsonProvider;
    private final DbSyncExecutor dbSyncExecutor;
    private final HttpRequestProvider httpRequestProvider;
    private final AppConfigProperty appConfigProperty;
    private final FileSyncQueueRepository fileSyncQueueRepository;

    public DataSyncServiceImpl(DataSyncMapRepository dataMapSyncRepository,
                               DataSyncLogRepository dataSyncLogRepository,
                               JsonProvider jsonProvider,
                               DbSyncExecutor dbSyncExecutor,
                               HttpRequestProvider httpRequestProvider,
                               AppConfigProperty appConfigProperty,
                               FileSyncQueueRepository fileSyncQueueRepository) {
        this.dataMapSyncRepository = dataMapSyncRepository;
        this.dataSyncLogRepository = dataSyncLogRepository;
        this.jsonProvider = jsonProvider;
        this.dbSyncExecutor = dbSyncExecutor;
        this.httpRequestProvider = httpRequestProvider;
        this.appConfigProperty = appConfigProperty;
        this.fileSyncQueueRepository = fileSyncQueueRepository;
    }

    @Override
    public CreateDataSyncCommandResult create(CreateDataSyncCommand command, FtpConfigInfoDto ftpConfigInfoDto) {
        if (command.getDataSets() == null || command.getDataSets().size() == 0) {
            throw new SaasBadException("DataSets数据集不能为空!");
        }
        if (!StringUtils.isEmpty(command.getFilePaths()) && command.getUrlFilePathMappings() == null) {
            throw new SaasBadException(
                    String.format("系统版本不一致(filePath=%s;urlFilePathMappings=NULL), 导致文件无法正常下载到目标系统,但是数据",
                            command.getFilePaths()));
        }

        Collection<String> mapKeys =
                command.getDataSets().stream()
                        .map(ii -> ii.getMapKey()).collect(Collectors.toList());

        Collection<DataSyncMapEntity> dataMapSyncEntities =
                this.dataMapSyncRepository.getAllByMapKeys(mapKeys);

        Collection<DataSyncLogEntity> dataSyncLogEntities = new ArrayList<>();
        AtomicReference<DataSyncMapEntity> randomSyncMapEntity = new AtomicReference<>();
        command.getDataSets().forEach(ds -> {
            ProcessedStatus status = ProcessedStatus.Done;
            String result = "完成";
            Collection<DataSyncMapEntity> dataSyncMapEntities = dataMapSyncEntities.stream()
                    .filter(ii -> ii.getMapKey().equals(ds.getMapKey())).collect(Collectors.toList());
            if (dataSyncMapEntities.size() == 0) {
                dataSyncMapEntities = getRegexDataSyncMapEntities(ds);
                if (dataSyncMapEntities.size() == 0) {
                    throw new SaasNofoundException(String.format("找不到%s数据映射配置, 无法执行同步操作", ds.getMapKey()));
                }
            }
            for (DataSyncMapEntity dataMapSync : dataSyncMapEntities) {
                try {
                    dataMapSync = doJdbcProcess(dataMapSync, ds);
                    status = ProcessedStatus.Done;
                } catch (Exception ex) {
                    status = ProcessedStatus.Error;
                    result = ExceptionUtils.getStackMessage(ex);
                    throw ex;
                } finally {
                    randomSyncMapEntity.set(dataMapSync);
                    String jsonData = this.jsonProvider.getJson(command.getFilePaths());
                    if (StringUtils.isEmpty(jsonData)) {
                        jsonData = ds.getId();
                    }

                    dataSyncLogEntities.add(
                            DataSyncLogEntity.create(dataMapSync, ds.getId(),
                                    jsonData,
                                    status, result));
                }
            }
        });

        if (randomSyncMapEntity == null || randomSyncMapEntity.get() == null) {
            throw new SaasBadException("randomSyncMapEntity无效!");
        }

        /**
         * 当有文件需要下载的时候，添加一条待执行的日志信息,
         * 后台服务将扫描并执行下载文件的操作
         */
        boolean savedFileQueued = saveSyncLogAndFiles(command, dataSyncLogEntities, ftpConfigInfoDto);

        Collection<CreateDataSyncCommandResult.DataSetResult> results = dataSyncLogEntities.stream()
                .map(ii -> CreateDataSyncCommandResult.DataSetResult
                        .create(ii.getSyncQueueId(), ii.getStatus() == ProcessedStatus.Done, ii.getResult()))
                .collect(Collectors.toList());

        return CreateDataSyncCommandResult.create(results, savedFileQueued);
    }

    @Override
    public CreateSyncFileCommand.CreateSyncFileCommandResult create(CreateSyncFileCommand command) {
        Collection<FileItemRequestDto> fileItemRequestDtos = command.getFileItems();
        Collection<CreateSyncFileCommand.FileItemResponse> fileItemResponses = new ArrayList<>();
        fileItemRequestDtos.forEach(ii -> {
            CreateSyncFileCommand.FileItemResponse fileItemResponse = null;
            try {
                String rootPath = appConfigProperty.getRootPath();
                if (rootPath != null && rootPath.endsWith("/")) {
                    rootPath = rootPath.substring(0, rootPath.length() - 1);
                }

                String targetPath = String.format("%s/%s", rootPath, ii.getTargetPath());
                httpRequestProvider.download(ii.getSourceUrl(), targetPath);

                fileItemResponse = CreateSyncFileCommand.FileItemResponse.create(ii, true, "targetPath");
            } catch (Exception ex) {
                fileItemResponse = CreateSyncFileCommand.FileItemResponse.create(ii, false, ex.toString());
            } finally {
                fileItemResponses.add(fileItemResponse);
            }
        });

        return CreateSyncFileCommand.CreateSyncFileCommandResult.create(command.getId(), fileItemResponses);
    }

    private DataSyncMapEntity doJdbcProcess(DataSyncMapEntity dataSyncMapEntity,
                                            CreateDataSyncCommand.DataSet ds) {
        JdbcMapSnapshot snapshot = calculateJdbcMapSnapshot(dataSyncMapEntity, ds);

        this.dbSyncExecutor.execute(snapshot, ds.getRows(), ds.getDeletedIds());

        return dataSyncMapEntity;
    }

    private Collection<DataSyncMapEntity> getRegexDataSyncMapEntities(CreateDataSyncCommand.DataSet ds) {
        Collection<DataSyncMapEntity> dataSyncMapEntities = this.dataMapSyncRepository.getAllRegexByMapKeys();
        /*
        Map<String, DataSyncMapEntity> dataSyncTableMaps =
                dataSyncMapEntities.stream().filter(ii -> mapKey.matches(ii.getMapKey()))
                        .collect(Collectors.toMap(ix -> {
                            Matcher matcher = Pattern.compile(ix.getMapKey()).matcher(mapKey);
                            return matcher.group(matcher.groupCount() - 1);
                        }, ix -> ix));

        if(dataSyncMapEntities.size()==0) {
            return Collections.EMPTY_LIST;
        }
         */

        return dataSyncMapEntities.stream().filter(ii -> ds.getAutoMapTableName(ii.getMapKey()) != null)
                .collect(Collectors.toList());
    }


    private static Map<String, Collection<TableMapSnapshot.Column>> tableMapSnapshotColumns = new ConcurrentHashMap<>();

    private JdbcMapSnapshot calculateJdbcMapSnapshot(DataSyncMapEntity dataSyncMapEntity,
                                                     CreateDataSyncCommand.DataSet ds) {
        JdbcMapSnapshot snapshot = this.jsonProvider.getData(dataSyncMapEntity.getTargetConfig(),
                JdbcMapSnapshot.class);
        if (snapshot.getTableMap().isAutoMap()) {
            if (StringUtils.isEmpty(snapshot.getTableMap().getTableName()) || "*".equalsIgnoreCase(snapshot.getTableMap().getTableName())) {
                snapshot.getTableMap().setTableName(ds.getAutoMapTableName(dataSyncMapEntity.getMapKey()));
            }

            Collection<TableMapSnapshot.Column> mapColumns = tableMapSnapshotColumns.get(snapshot.getTableMap().getTableName());
            if (mapColumns == null) {
                mapColumns = new ArrayList<>();
                //todo 从数据库获取列
                DataSource dataSource = this.dbSyncExecutor.getDataSource(snapshot.getJdbcConnection());
                String jdbcUrl = "EMPTY";
                try (Connection connection = dataSource.getConnection()) {
                    String pingSql = String.format("select * from %s where 1<>1 limit 1", snapshot.getTableMap().getTableName());
                    jdbcUrl = String.format("JDBC=%s;sql=%s", connection.getMetaData().getURL(), pingSql);
                    try (Statement pingStatement = connection.createStatement()) {
                        ResultSet resultSet = pingStatement.executeQuery(pingSql);
                        for (int columnIndex = 0; columnIndex < resultSet.getMetaData().getColumnCount(); columnIndex++) {
                            TableMapSnapshot.Column tableColumn = TableMapSnapshot.Column.add(
                                    resultSet.getMetaData().getColumnName(columnIndex + 1), null);

                            mapColumns.add(tableColumn);
                        }
                    }
                } catch (Exception ex) {
                    throw new SaasBadException(
                            String.format("[%s]: 找不到表(%s:JDBC=%s)元数据信息(明细: %s)",
                                    dataSyncMapEntity.getId(),
                                    snapshot.getTableMap().getTableName(),
                                    dataSyncMapEntity.getTargetConfig(),
                                    jdbcUrl));
                }

                tableMapSnapshotColumns.put(snapshot.getTableMap().getTableName(), mapColumns);
            }

            for (TableMapSnapshot.Column selColumn : mapColumns) {
                Optional<TableMapSnapshot.Column> columnOptional =
                        snapshot.getTableMap().getColumns().stream().filter(ix -> ix.getFieldName().equalsIgnoreCase(selColumn.getFieldName()))
                                .findFirst();
                if (columnOptional.isPresent()) {
                    selColumn.setDefaultValue(columnOptional.get().getDefaultValue());
                    selColumn.setFixedValue(columnOptional.get().getFixedValue());
                    selColumn.setOnlyForNew(columnOptional.get().isOnlyForNew());
                }
            }

            snapshot.getTableMap().setColumns(mapColumns);
        }

        return snapshot;
    }

    private boolean saveSyncLogAndFiles(CreateDataSyncCommand command, Collection<DataSyncLogEntity> dataSyncLogEntities, FtpConfigInfoDto ftpConfigInfoDto) {
        boolean savedFileQueued = false;
        StringBuilder filePathTracking = new StringBuilder(String.format("待处理文件列表是否为空=%s;", StringUtils.isEmpty(command.getFilePaths())));
        try {
            String firstSynclogId = "EMPTY";
            DataSyncLogEntity syncLogEntity = null;
            if (!dataSyncLogEntities.isEmpty()) {
                syncLogEntity = dataSyncLogEntities.stream().findFirst().get();
                firstSynclogId = syncLogEntity.getId();
            }

            filePathTracking.append(String.format("详细参数=%s;", this.jsonProvider.getJson(command)));

            if (!StringUtils.isEmpty(command.getFilePaths())) {
                Collection<String> paths = Arrays.stream(command.getFilePaths().split(","))
                        .filter(ii -> !StringUtils.isEmpty(ii))
                        .distinct().collect(Collectors.toList());

                filePathTracking.append(String.format("总共%s个文件; urlMapping=%s;",
                        paths.size(), command.getUrlFilePathMappings().size()));

                Collection<FieldNames.PathOption> dataSetFilesPathOptions =
                        command.getDataSets().stream().flatMap(ix -> ix.getRows().stream())
                                .flatMap(ix -> ix.getFieldValues().stream())
                                .filter(ix -> ix.isValidFile())
                                .flatMap(ix -> ix.getPathOptions().stream())
                                .filter(ix -> !StringUtils.isEmpty(ix.getOriginalPath()))
                                .collect(Collectors.toList());

                filePathTracking.append(String.format("dataSetFilesPathOptions=%s;", this.jsonProvider.getJson(dataSetFilesPathOptions)));

                filePathTracking.append("[");
                String finalFirstSynclogId = firstSynclogId;
                Collection<FileSyncQueueEntity> fileSyncQueueEntities =
                        paths.stream().flatMap(downloadUrl -> {
                            try {
                                String finalOriginalDownloadUrl = command.getOriginalDownloadUrl(downloadUrl);
                                Collection<String> targetPaths =
                                        dataSetFilesPathOptions.stream()
                                                .filter(ix -> !StringUtils.isEmpty(finalOriginalDownloadUrl) &&
                                                        ix.getOriginalPath().equalsIgnoreCase(finalOriginalDownloadUrl))
                                                .map(ix -> ix.getPath())
                                                .distinct().collect(Collectors.toList());

                                filePathTracking.append(String.format("映射文件key=%s;匹配结果=%s;", finalOriginalDownloadUrl, targetPaths.size()));

                                Collection<FileSyncQueueEntity> queues = targetPaths.stream().map(tp ->
                                        FileSyncQueueEntity.create(downloadUrl, tp, command.getDataSetId(), finalFirstSynclogId))
                                        .collect(Collectors.toList());

                                return queues.stream();
                            } catch (Exception ex) {
                                return Stream.empty();
                            }
                        }).collect(Collectors.toList());
                filePathTracking.append("]");
                filePathTracking.append(String.format("待提交的fileSyncQueueEntities对象=%s个;", fileSyncQueueEntities.size()));

                if (!fileSyncQueueEntities.isEmpty()) {
                    if (ftpConfigInfoDto != null && ftpConfigInfoDto.getUseFtp()) {
                        List<Future> results = new ArrayList<>();
                        for (FileSyncQueueEntity fileSyncQueueEntity : fileSyncQueueEntities) {
                            FtpUploadTask ftpUploadTask = new FtpUploadTask(fileSyncQueueEntity, ftpConfigInfoDto);
                            Future submit = newFixedThreadPool.submit(ftpUploadTask);
                            results.add(submit);
                        }
                        fileSyncQueueEntities.clear();
                        for (Future result : results) {
                            try {
                                //获取线程结果
                                FileSyncQueueEntity resultFileSyncQueueEntity = (FileSyncQueueEntity) result.get(2, TimeUnit.MINUTES);
                                fileSyncQueueEntities.add(resultFileSyncQueueEntity);
                            } catch (Exception e) {
                                e.printStackTrace();
                            }
                        }
                    }
                    filePathTracking.append("准备保存文件队列信息;");
                    fileSyncQueueRepository.saveAll(fileSyncQueueEntities);
                    filePathTracking.append("执行完文件队列的保存;");
                    savedFileQueued = true;
                }
            }

            try {
                String data = String.format("附件信息:%s", filePathTracking);
                dataSyncLogEntities.add(
                        DataSyncLogEntity.create(syncLogEntity.getDataMapSync(),
                                syncLogEntity.getSyncQueueId(), "FileSyncQueueEntity.Tracking", ProcessedStatus.Done, data));
            } catch (Exception ex) {
                filePathTracking.append(String.format("创建附加信息异常:%s;", ex.toString()));
            }

            filePathTracking.append(String.format("执行dataSyncLogEntities的保存操作=%s;", dataSyncLogEntities.size()));
            if (!dataSyncLogEntities.isEmpty()) {
                dataSyncLogRepository.saveAll(dataSyncLogEntities);

                filePathTracking.append("dataSyncLogEntities保存成功;");
            }

            return savedFileQueued;
        } catch (Exception ex) {
            filePathTracking.append(String.format("执行saveSyncLogAndFiles方法出现异常: %s", ExceptionUtils.getStackMessage(ex)));
            logger.error("执行saveSyncLogAndFiles完毕:{}", filePathTracking);
            throw ex;
        }
    }

}


