package com.bcxin.runtime.domain.syncs.ftp;

import cn.hutool.core.util.StrUtil;
import com.bcxin.runtime.domain.syncs.enums.UploadStatus;
import org.apache.commons.net.ftp.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.net.SocketException;
import java.net.URL;
import java.nio.file.Paths;

/****
 * FTP连接工具类
 * @author subh
 */
public class FtpUtils {
    /***
     * 日志
     */
    protected static Logger logger = LoggerFactory.getLogger(FtpUtils.class);

    /**
     * 获取FTPClient对象
     *
     * @param ftpHost     FTP主机服务器
     * @param ftpPassword FTP 登录密码
     * @param ftpUserName FTP登录用户名
     * @param ftpPort     FTP端口 默认为21
     * @return
     */
    public static FTPClient getFTPClient(String ftpHost, String ftpUserName,
                                         String ftpPassword, int ftpPort) {
        FTPClient ftpClient = new FTPClient();
        try {
            ftpClient = new FTPClient();
            ftpClient.connect(ftpHost, ftpPort);// 连接FTP服务器
            ftpClient.login(ftpUserName, ftpPassword);// 登陆FTP服务器
            if (!FTPReply.isPositiveCompletion(ftpClient.getReplyCode())) {
                logger.info("未连接到FTP，用户名或密码错误。");
                ftpClient.disconnect();
            } else {
                logger.info("FTP连接成功。");
            }
        } catch (SocketException e) {
            logger.error(e.getMessage(), e);
        } catch (IOException e) {
            logger.error(e.getMessage(), e);
        }
        return ftpClient;
    }

    /**
     * 下载文件
     *
     * @param ftpHost     ftp服务器地址
     * @param ftpUserName anonymous匿名用户登录，不需要密码。administrator指定用户登录
     * @param ftpPassword 指定用户密码
     * @param ftpPort     ftp服务员器端口号
     * @param ftpPath     ftp文件存放物理路径
     * @param fileName    文件路径
     */
    public static InputStream downloadFile(String ftpHost, String ftpUserName,
                                           String ftpPassword, int ftpPort, String ftpPath,
                                           String fileName) {
        FTPClient ftpClient = null;
        InputStream is = null;
        try {
            ftpClient = getFTPClient(ftpHost, ftpUserName, ftpPassword, ftpPort);
            ftpClient.setControlEncoding("UTF-8"); // 中文支持
            ftpClient.setFileType(FTPClient.BINARY_FILE_TYPE);
            ftpClient.enterLocalPassiveMode();
            ftpClient.changeWorkingDirectory(ftpPath);
            is = ftpClient.retrieveFileStream(fileName);
        } catch (FileNotFoundException e) {
            logger.error("没有找到" + ftpPath + "文件");
            logger.error(e.getMessage(), e);
        } catch (SocketException e) {
            logger.error("连接FTP失败.");
            logger.error(e.getMessage(), e);
        } catch (IOException e) {
            logger.error("文件读取错误。");
            logger.error(e.getMessage(), e);
        } finally {
            try {
                if (ftpClient != null) {
                    ftpClient.logout();
                    ftpClient.disconnect();
                }
            } catch (Exception e) {
                logger.error(e.getMessage(), e);
            }
        }
        return is;
    }

    /**
     * 下载文件存临时文件
     *
     * @param ftpHost     ftp服务器地址
     * @param ftpUserName anonymous匿名用户登录，不需要密码。administrator指定用户登录
     * @param ftpPassword 指定用户密码
     * @param ftpPort     ftp服务员器端口号
     * @param localPath   本地地址
     * @param ftpPath     ftp文件存放物理路径
     * @param fileName    文件路径
     */
    public static File downloadFile(String ftpHost, String ftpUserName,
                                    String ftpPassword, int ftpPort, String ftpPath, String localPath,
                                    String fileName) {
        FTPClient ftpClient = null;
        OutputStream os = null;
        //判断文件目录是否存在，如果不存在则创建
        File dir = new File(localPath);
        if (!dir.exists()) {
            dir.mkdirs();
        }
        File localFile = new File(localPath + fileName);
        try {
            //连接ftp
            ftpClient = getFTPClient(ftpHost, ftpUserName, ftpPassword, ftpPort);
            //设置编码
            ftpClient.setControlEncoding("UTF-8"); // 中文支持
            ftpClient.setFileType(FTPClient.BINARY_FILE_TYPE);
            //设置linux ftp服务器
            FTPClientConfig conf = new FTPClientConfig(FTPClientConfig.SYST_UNIX);
            ftpClient.configure(conf);
            //设置访问被动模式
            ftpClient.setRemoteVerificationEnabled(false);
            ftpClient.enterLocalPassiveMode();
            //切换FTP目录
            ftpClient.changeWorkingDirectory(ftpPath);
            //获取文件流
            try (InputStream initialStream = ftpClient.retrieveFileStream(new String(fileName.getBytes("UTF-8"), "UTF-8"));
                 OutputStream outStream = new FileOutputStream(localFile)) {
                byte[] buffer = new byte[8 * 1024];
                int bytesRead;
                while ((bytesRead = initialStream.read(buffer)) != -1) {
                    outStream.write(buffer, 0, bytesRead);
                }
            } catch (Exception e) {
                e.printStackTrace();
                logger.error(e.getMessage(), e);
            }
        } catch (FileNotFoundException e) {
            logger.error("没有找到" + ftpPath + "文件");
            logger.error(e.getMessage(), e);
        } catch (SocketException e) {
            logger.error("连接FTP失败.");
            logger.error(e.getMessage(), e);
        } catch (IOException e) {
            logger.error("文件读取错误。");
            logger.error(e.getMessage(), e);
        } finally {
            try {
                if (os != null) os.close();
            } catch (Exception e) {
                e.printStackTrace();
                logger.error(e.getMessage(), e);
            }
            try {
                if (ftpClient != null && ftpClient.isConnected()) {
                    ftpClient.logout();
                    ftpClient.disconnect();
                }

            } catch (Exception e) {
                logger.error(e.getMessage(), e);
            }
        }
        return localFile;
    }

    /**
     * 上传文件
     *
     * @param ftpHost     ftp服务器地址
     * @param ftpUserName anonymous匿名用户登录，不需要密码。administrator指定用户登录
     * @param ftpPassword 指定用户密码
     * @param ftpPort     ftp服务员器端口号
     * @param ftpPath     ftp文件存放物理路径
     * @param fileName    文件路径
     * @param input       文件输入流，即从本地服务器读取文件的IO输入流
     */
    public static void uploadFile(String ftpHost, String ftpUserName,
                                  String ftpPassword, int ftpPort, String ftpPath,
                                  String fileName, InputStream input) {
        FTPClient ftp = null;
        try {
            ftp = getFTPClient(ftpHost, ftpUserName, ftpPassword, ftpPort);
            changeDir(ftp, ftpPath);
            logger.error("当前目录：" + ftp.printWorkingDirectory());
            ftp.setFileType(FTP.BINARY_FILE_TYPE);
            if (ftp.storeFile(fileName, input)) {
                logger.error(fileName + "上传到" + ftpPath + "成功");
            } else {
                logger.error(fileName + "上传到" + ftpPath + "失败");
                fileName = new String(fileName.getBytes("GBK"), "iso-8859-1");
                if (ftp.storeFile(fileName, input)) {
                    logger.error(fileName + "上传到" + ftpPath + "成功");
                } else {
                    logger.error(fileName + "上传到" + ftpPath + "失败");
                }
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        } finally {
            try {
                if (input != null) input.close();
            } catch (Exception e) {
                logger.error(e.getMessage(), e);
            }
            try {
                if (ftp != null) {
                    ftp.logout();
                    ftp.disconnect();
                }
            } catch (Exception e) {
                logger.error(e.getMessage(), e);
            }
        }
    }


    /**
     * 上传文件
     *
     * @param ftpHost     ftp服务器地址
     * @param ftpUserName anonymous匿名用户登录，不需要密码。administrator指定用户登录
     * @param ftpPassword 指定用户密码
     * @param ftpPort     ftp服务员器端口号
     * @param pathFile    ftp文件存放物理路径(带文件名)
     * @param input       文件输入流，即从本地服务器读取文件的IO输入流
     */
    public static boolean uploadFile(String ftpHost, String ftpUserName,
                                     String ftpPassword, int ftpPort,
                                     String pathFile, InputStream input) {
        boolean result = false;
        FTPClient ftp = null;
        try {
            String ftpPath = pathFile.substring(0, pathFile.lastIndexOf("/"));
            String fileName = pathFile.substring(pathFile.lastIndexOf("/") + 1);
            ftp = getFTPClient(ftpHost, ftpUserName, ftpPassword, ftpPort);
            ftp.setFileType(FTP.BINARY_FILE_TYPE);
            logger.error("当前FTP目录：" + ftp.printWorkingDirectory());
            if (ftp.storeFile(pathFile, input)) {
                logger.error(pathFile + "上传成功");
            }

            changeDir(ftp, ftpPath);
            logger.error("当前FTP目录：" + ftp.printWorkingDirectory());
            if (ftp.storeFile(fileName, input)) {
                result = true;
                logger.error(fileName + "上传成功");
            } else {
                logger.error(fileName + "上传失败");
                fileName = new String(fileName.getBytes("GBK"), "iso-8859-1");
                if (ftp.storeFile(fileName, input)) {
                    logger.error(fileName + "上传成功");
                    result = true;
                } else {
                    logger.error(fileName + "上传失败");
                }
            }
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        } finally {
            try {
                if (input != null) {
                    input.close();
                }
            } catch (Exception e) {
                logger.error(e.getMessage(), e);
            }
            try {
                if (ftp != null) {
                    ftp.logout();
                    ftp.disconnect();
                }
            } catch (Exception e) {
                logger.error(e.getMessage(), e);
            }
        }
        return result;
    }


    public static void changeDir(FTPClient ftpClient, String dirPath) {
        try {
            for (String s : dirPath.split("/")) {
                if (StrUtil.isEmpty(s)) {
                    continue;
                }
                if (!ftpClient.changeWorkingDirectory(s)) {
                    logger.error("进入目录失败：" + s);
                    if (!ftpClient.makeDirectory(s)) {
                        logger.error("创建目录失败：" + s);
                    } else {
                        logger.error("创建目录成功：" + s);
                    }
                    if (!ftpClient.changeWorkingDirectory(s)) {
                        logger.error("进入目录失败：" + s);
                    } else {
                        logger.error("进入目录成功：" + s);
                    }
                } else {
                    logger.error("进入目录成功：" + s);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public static UploadStatus bcxUploadFile(String ftpHost, String ftpUserName,
                                             String ftpPassword, int ftpPort,
                                             String pathFile, String localPath) throws Exception {

        BcxinFtpClient ftpClient = null;
        InputStream input = null;
        try {
            File file = new File(localPath);
            if (!file.exists()) {
                return UploadStatus.File_Not_Exits;
            }
            ftpClient = new BcxinFtpClient();
            ftpClient.connect(ftpHost, ftpUserName, ftpPassword, ftpPort);
            String ftpPath = pathFile.substring(0, pathFile.lastIndexOf("/") + 1);
            ftpPath = new String(ftpPath.getBytes("UTF-8"), "iso-8859-1");
            String fileName = pathFile.substring(pathFile.lastIndexOf("/") + 1);
            fileName = new String(fileName.getBytes("UTF-8"), "iso-8859-1");
            boolean isDir = ftpClient.isARemoteDirectory(ftpPath);
            if (!isDir) {
                boolean createDirectoryFlag = ftpClient.createDirectory(ftpPath);
                if (!createDirectoryFlag) {
                    logger.error("目录创建失败：[{}]", ftpPath);
                    return UploadStatus.Create_Directory_Fail;
                } else {
                    logger.debug("目录创建成功：[{}]", ftpPath);
                }
            }
            String filePath = ftpPath + fileName;
            String fileDir = Paths.get(filePath).getParent().toString().replace("\\", "/");
            if (!ftpClient.isARemoteDirectory(fileDir)) {
                boolean createDirectoryFlag = ftpClient.createDirectory(fileDir);
                if (!createDirectoryFlag) {
                    logger.error("目录创建失败：[{}]", ftpPath);
                    return UploadStatus.Create_Directory_Fail;
                } else {
                    logger.debug("目录创建成功：[{}]", ftpPath);
                }
            }

            filePath = filePath.replace("\\", "/");
            boolean existFlag = existFile(filePath, ftpClient, file);
            if (existFlag) {
                return UploadStatus.File_Exits;
            }
            input = new FileInputStream(file);
            boolean result = ftpClient.uploadFile(input, filePath);
            if (result) {
                return UploadStatus.Upload_New_File_Success;
            }
        } catch (Exception e) {
            e.printStackTrace();
            throw e;
        } finally {
            if (ftpClient != null) {
                try {
                    ftpClient.disconnect();
                } catch (Exception e) {
                    logger.error(e.getMessage());
                }
            }
        }

        return UploadStatus.Upload_New_File_Failed;

    }

    public static boolean checkUrlFileExists(InputStream inputStream) {
        boolean flag = false;
        try {
            byte[] bytes = new byte[1024];
            if (inputStream.read(bytes) != -1) {
                flag = true;
            }
            return flag;
        } catch (Exception e) {
            logger.error("检查摆渡服务文件是否存在异常", e.getMessage());
            return false;
        }
    }

    public static boolean existFile(String ftpPath, BcxinFtpClient ftpClient, File localFile) {
        boolean flag = false;
        try {
            FTPFile[] ftpFileArr = ftpClient.getClient().listFiles(ftpPath);

            if (ftpFileArr.length > 0) {
                long remoteSize = ftpFileArr[0].getSize();
                if (remoteSize >= localFile.length()) {
                    flag = true;
                } else {
                    ftpClient.deleteFile(ftpPath);
                }
            }
        } catch (Exception e) {
            logger.debug("判断ftp服务器文件是否存在出错：[{}]", ftpPath);
        }
        return flag;
    }

    public static long getFileLength(String ftpPath, BcxinFtpClient ftpClient) {
        long length = 0;
        try {
            FTPFile[] files = ftpClient.getClient().listFiles(ftpPath);
            if (files != null && files.length > 0) {
                length = files[0].getSize();
            }
        } catch (Exception e) {
            logger.debug("获取ftp服务器文件大小出错：[{}]", ftpPath);
        }
        return length;
    }
}