package com.bcxin.saas.core.utils;

import com.bcxin.saas.core.components.ActionFunction;
import com.bcxin.saas.core.exceptions.SaasBadException;
import com.bcxin.saas.core.exceptions.SaasRetryableException;
import com.github.rholder.retry.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;

public class RetryUtil {
    private static final Logger logger = LoggerFactory.getLogger(RetryUtil.class);

    public static <T> T  execute(Callable<T> function, int retryCount) {
        try {
            Retryer<T> retryer = RetryerBuilder.<T>newBuilder()
                    .retryIfExceptionOfType(SaasRetryableException.class)
                    .withStopStrategy(StopStrategies.stopAfterAttempt(retryCount))
                    .withWaitStrategy(WaitStrategies.fixedWait(5, TimeUnit.MILLISECONDS))
                    .build();

            return retryer.call(() -> {
                try {
                    return function.call();
                } catch (Exception e) {
                    if (!ExceptionUtils.getStackMessage(e).contains("504")) {
                        throw new SaasRetryableException(e);
                    }

                    throw e;
                }
            });
        } catch (Exception e) {
            throw new SaasBadException(String.format("failed to execute max retries(%s)", retryCount), e);
        }
    }

    public static <T> T execute(Callable<T> function) {
        return RetryUtil.execute(function, 2);
    }
}
