package com.bcxin.saas.core.components;
import com.bcxin.saas.core.models.CacheItemAbstract;
import lombok.Getter;

import java.sql.Timestamp;
import java.time.Instant;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

public interface CacheProvider {
    <T> T getData(String key, Supplier<T> supplier);

    <T> T getData(String key, Supplier<T> supplier, int expiredInSeconds);

    void reset(String key);

    <T> void update(String key, T data);

    <T> void update(String key, T data, int expiredInSeconds);

    static class CacheProviderImpl implements CacheProvider
    {
        private final int DEFAULT_EXPIRED_IN_SECONDS=2 * 24 * 60 * 60;
        private static ConcurrentHashMap<String, CacheDataWrapper> concurrentHashMap = new ConcurrentHashMap<>();
        @Override
        public <T> T getData(String key, Supplier<T> supplier) {
            return getData(key, supplier, DEFAULT_EXPIRED_IN_SECONDS);
        }

        @Override
        public <T> T getData(String key, Supplier<T> supplier, int expiredInSeconds) {
            assert supplier != null;

            CacheDataWrapper wrapperData = concurrentHashMap.get(key);
            Object data = null;
            if (wrapperData == null || wrapperData.isExpired()) {
                data = supplier.get();
                update(key, data);
            } else {
                data = wrapperData.getData();
            }

            return (T) data;
        }

        @Override
        public void reset(String key) {
            concurrentHashMap.remove(key);
        }

        @Override
        public <T> void update(String key, T data) {
            update(key, data, DEFAULT_EXPIRED_IN_SECONDS);
        }

        @Override
        public <T> void update(String key, T data, int expiredInSeconds) {
            if (data == null) {
                if (concurrentHashMap.contains(key)) {
                    concurrentHashMap.remove(key);
                }
                return;
            }

            concurrentHashMap.put(key, CacheDataWrapper.create(data, expiredInSeconds));
        }

        @Getter
        public static class CacheDataWrapper
        {
            private final Object data;
            private final int expiredInSeconds;
            private final Timestamp expiredTimestamp;

            public CacheDataWrapper(Object data, int expiredInSeconds, Timestamp expiredTimestamp) {
                this.data = data;
                this.expiredInSeconds = expiredInSeconds;
                this.expiredTimestamp = expiredTimestamp;
            }

            public static CacheDataWrapper create(Object data, int expiredInSeconds) {
                return new CacheDataWrapper(data, expiredInSeconds, Timestamp.from(Instant.now().plusSeconds(expiredInSeconds)));
            }

            public boolean isExpired() {
                return Timestamp.from(Instant.now()).after(this.getExpiredTimestamp());
            }
        }
    }
}
