package com.bcxin.rest.web.apis.controllers;

import com.bcxin.Infrastructures.TenantContext;
import com.bcxin.Infrastructures.TenantUserContext;
import com.bcxin.Infrastructures.components.JsonProvider;
import com.bcxin.Infrastructures.exceptions.NotFoundTenantException;
import com.bcxin.Infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.api.interfaces.tenants.ExternalMemberRpcProvider;
import com.bcxin.api.interfaces.tenants.requests.externalMembers.CreateExternalMemberRequest;
import com.bcxin.api.interfaces.tenants.requests.externalMembers.ExternalMemberJoinRecordSearchRequest;
import com.bcxin.api.interfaces.tenants.responses.ExternalMemberJoinRecordResponse;
import com.bcxin.rest.web.apis.caches.ExternalMemberInviteCodeCache;
import com.bcxin.rest.web.apis.requests.JoinOrganizationAsExternalMemberRequest;
import com.bcxin.rest.web.apis.responses.ExternalMemberInviteInfoResponse;
import io.swagger.annotations.*;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import java.io.IOException;
import java.util.Collection;

/**
 * 我与团体的相关操作
 */
@Api("我与团体的相关操作")
@RestController
@RequestMapping("/external-members")
public class ExternalMemberController extends ControllerAbstract {

    private final RedisTemplate redisTemplate;
    private final JsonProvider jsonProvider;

    private final ExternalMemberRpcProvider externalMemberRpcProvider;

    public ExternalMemberController(RedisTemplate redisTemplate, JsonProvider jsonProvider, ExternalMemberRpcProvider externalMemberRpcProvider) {
        this.redisTemplate = redisTemplate;
        this.jsonProvider = jsonProvider;
        this.externalMemberRpcProvider = externalMemberRpcProvider;
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("查看邀请码/二维码的基本信息")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @GetMapping("/invite-codes/{code}/info")
    public ResponseEntity<ExternalMemberInviteInfoResponse> getInviteBaiscInfo(
            @PathVariable String code,
            HttpServletResponse response) throws IOException {
        TenantUserContext.UserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException("当前用户无效");
        }

        String cacheKey = ExternalMemberInviteCodeCache.getInviteCodeCacheKey(code);
        String jsonData = (String) this.redisTemplate.opsForValue().get(cacheKey);
        ExternalMemberInviteCodeCache codeCache = this.jsonProvider.toObject(ExternalMemberInviteCodeCache.class, jsonData);

        if (codeCache == null) {
            throw new NotFoundTenantException();
        }

        return this.ok(ExternalMemberInviteInfoResponse.create(codeCache.getEmployeeId(), codeCache.getName(), codeCache.getOrgId(), codeCache.getOrgName()));
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("通过团体的邀请码/二维码加入团体")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @PostMapping("/join")
    public ResponseEntity join(
            @RequestBody JoinOrganizationAsExternalMemberRequest request,
            HttpServletResponse response) throws IOException {
        TenantUserContext.UserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException("未授权用户");
        }

        String cacheKey = ExternalMemberInviteCodeCache.getInviteCodeCacheKey(request.getCode());
        String jsonData = (String) this.redisTemplate.opsForValue().get(cacheKey);
        ExternalMemberInviteCodeCache codeCache = this.jsonProvider.toObject(ExternalMemberInviteCodeCache.class, jsonData);
        if(codeCache==null) {
            throw new NotFoundTenantException();
        }

        this.externalMemberRpcProvider.create(
                CreateExternalMemberRequest.create(
                        request.getInviteType(),
                        request.getCode(),
                        codeCache.getOrgId(),
                        codeCache.getEmployeeId(),
                        codeCache.getGroupId()
                ));

        return this.ok();
    }

    @ApiImplicitParams({
            @ApiImplicitParam(
                    name = "Authorization",
                    value = "Bearer 租户token",
                    paramType = "header",
                    dataType = "String"
            )
    })
    @ApiOperation("查询我的加入团体请求记录")
    @ApiResponses({
            @ApiResponse(code = 200, message = "无", response = Valid.class),
            @ApiResponse(code = 404, message = "当前用户无效", response = Valid.class)
    })
    @PostMapping("/my-records")
    public ResponseEntity<Collection<ExternalMemberJoinRecordResponse>> getMyJoinRecords(@RequestBody ExternalMemberJoinRecordSearchRequest request) {
        Collection<ExternalMemberJoinRecordResponse> data
                = this.externalMemberRpcProvider.getMyJoinRecords(request);

        return this.ok(data);
    }
}
