package com.bcxin.tenant.backend.tasks;

import com.bcxin.tenant.backend.components.SubscriberEventActionComponent;
import com.bcxin.tenant.domain.entities.EventSubscriberEntity;
import com.bcxin.tenant.domain.readers.TenantDbReader;
import com.bcxin.tenant.domain.repositories.EventSubscriberRepository;
import org.springframework.core.task.TaskExecutor;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.concurrent.CountDownLatch;

public abstract class V5SyncTenantEventTaskAbstract extends TaskAbstract {
    protected final EventSubscriberRepository subscriberRepository;
    protected final SubscriberEventActionComponent subscriberEventActionComponent;
    protected final TaskExecutor taskExecutor;
    protected final TenantDbReader dbReader;

    public V5SyncTenantEventTaskAbstract(EventSubscriberRepository subscriberRepository,
                                       SubscriberEventActionComponent subscriberEventActionComponent,
                                       TaskExecutor taskExecutor, TenantDbReader dbReader) {
        this.subscriberRepository = subscriberRepository;
        this.subscriberEventActionComponent = subscriberEventActionComponent;
        this.taskExecutor = taskExecutor;
        this.dbReader = dbReader;
    }

    @Override
    protected void runCore() {
        try {
            Collection<EventSubscriberEntity> subscriberEntities = getOnlineSubscribers();

            CountDownLatch countDownLatch = new CountDownLatch(subscriberEntities.size());

            subscriberEntities.parallelStream().forEach(subscriber -> {
                taskExecutor.execute(() -> {
                    try {
                        Collection<String> eventIds = this.getPendingEventIds(Collections.singleton(subscriber.getSelector()));
                        if (!CollectionUtils.isEmpty(eventIds)) {
                            this.subscriberEventActionComponent.execute(subscriber, eventIds);
                        }
                    } catch (Exception ex) {
                        logger.error("订阅({})发生异常",subscriber,ex);
                    } finally {
                        countDownLatch.countDown();
                    }
                });
            });

            countDownLatch.await();
        } catch (InterruptedException e) {
            logger.error("阻扰有异常", e);
        }
    }

    private static Collection<EventSubscriberEntity> onlineSubscriberEntities = new ArrayList<>();

    protected Collection<EventSubscriberEntity> getOnlineSubscribers() {

        if (CollectionUtils.isEmpty(onlineSubscriberEntities)) {
            onlineSubscriberEntities.addAll(this.subscriberRepository.getOnline());
        }

        return onlineSubscriberEntities;
    }

    protected abstract Collection<String> getPendingEventIds(Collection<String> mapKeys);
}
