/*
 * Copyright (c) 2015 xuerdongcom@126.com All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 */
package com.bcxin.api.interfaces.tenants.responses;

import lombok.Data;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 该对象只是一个约定的返回到前台的json分页对象，按该对象设置可基本满足分页需求，
 * 若不满足需求可自行返回json对象
 * <p>
 * code 表示请求状态编码
 * </p>
 * <p>
 * message 提示的信息
 * </p>
 * <p>
 * data 要返回的数据
 * </p>
 *
 * @author subinghui
 * @since 1.1
 */
@Data
public class AjaxPageResponse<T extends Serializable> implements Serializable,Cloneable {
    //该请求状态编码
    private String code;
    //请求提示信息
    private String message;
    //数据对象
    private Collection<T> data;
    //第几页
    private int pageNumber = 1;
    //每页多少条
    private int pageSize = 20;
    //总共多少条
    private long total = 0l;
    //总共多少页
    private long totalPage = 0l;

    //是否分页 默认true分页
    private boolean pagination = true;

    private String sort;//排序字段

    private String order;//排序 asc 顺序 or 倒序desc
    /**
     * 创建一个新的{@link AjaxPageResponse}，code为0,message为操作成功
     */
    public AjaxPageResponse() {
        this(ResponseCode.SUCCESS, "操作成功");
    }

    /**
     * 创建一个新的{@link AjaxPageResponse}
     *
     * @param code {@link ResponseCode}
     */
    public AjaxPageResponse(String code) {
        this(code, null);
    }

    /**
     * 创建一个新的{@link AjaxPageResponse}
     *
     * @param code    {@link ResponseCode}
     * @param message
     */
    public AjaxPageResponse(String code, String message) {
        this.code = (code == null ? ResponseCode.SUCCESS : code);
        this.message = message;
        if (this.message == null) {
            if (code.equals(ResponseCode.SUCCESS))
                this.message = "操作成功";
            if (code.equals(ResponseCode.FAIL))
                this.message = "操作失败";
            if (code.equals(ResponseCode.UNAUTHORIZED))
                this.message = "用户验证失败";
        }
        this.data = new ArrayList<>();
    }

    /**
     * 创建一个新的{@link AjaxPageResponse}，success为false
     *
     * @return
     */
    public static AjaxPageResponse fail() {
        return fail(null);
    }

    /**
     * 创建一个新的{@link AjaxPageResponse}，success为false
     *
     * @param message 消息
     * @return
     */
    public static AjaxPageResponse fail(String message) {
        return new AjaxPageResponse(ResponseCode.FAIL, message);
    }

    /**
     * 创建一个新的{@link AjaxPageResponse}，code为ResponseCode.SUCCESS
     *
     * @return
     */
    public static AjaxPageResponse success() {
        return success(null);
    }

    /**
     * 创建一个新的{@link AjaxPageResponse}，success为true
     *
     * @param message 消息
     * @return
     */
    public static AjaxPageResponse success(String message) {
        return new AjaxPageResponse(ResponseCode.SUCCESS, message);
    }


    public void setTotal(long total) {
        this.total = total;
        this.totalPage = this.total / this.getPageSize();
        if (this.total % this.getPageSize() != 0)
            this.totalPage++;
    }


    //自定义类实现自己的clone()方法
    @Override
    public AjaxPageResponse clone() throws CloneNotSupportedException {
        //通过super.clone()实现clone()方法
        return (AjaxPageResponse) super.clone();
    }

    public String getSort() {
        return sort;
    }

    public void setSort(String sort) {
        this.sort = sort;
    }

    public String getOrder() {
        return order;
    }

    public void setOrder(String order) {
        this.order = order;
    }
}
