package com.bcxin.rbac.domain.entities;

import com.bcxin.Infrastructures.entities.IAggregate;
import com.bcxin.Infrastructures.exceptions.NotAllowedTenantException;
import com.bcxin.Infrastructures.utils.UUIDUtil;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.Setter;

import javax.persistence.*;
import java.sql.Timestamp;
import java.time.Instant;
import java.util.Collection;

/**
 * 资源管理
 * 功能点, 数据, 应用包
 */
@Getter
@Setter(AccessLevel.PROTECTED)
@Table(name = "rbac_resources")
@Entity
public class ResourceEntity extends MetaEntityAbstract implements IAggregate {
    @Id
    @Column(name = "id", length = 200)
    private String id;

    @Column(name = "code", length = 200, nullable = false)
    private String code;

    /**
     * 资源名称
     */
    @Column(name = "name", length = 500)
    private String name;

    /**
     * 资源对应的数据
     */
    @Column(name = "data", length = 4000)
    private String data;

    @Column(name = "tree_path_id", nullable = false, length = 255)
    private String treePathId;

    @Column(name = "level", nullable = false)
    private int level;

    /**
     * 数据类型： 功能点, 菜单，数据权限
     */
    @Column(name = "data_type", length = 100, nullable = false)
    private String dataType;

    @ManyToOne
    @JoinColumn(name = "parent_id", referencedColumnName = "id", nullable = true, foreignKey = @ForeignKey(name = "fk_resource_parent_id", value = ConstraintMode.CONSTRAINT))
    private ResourceEntity parent;

    @OneToMany(mappedBy = "parent",cascade = CascadeType.ALL)
    private Collection<ResourceEntity> resources;

    @Column(name = "note", length = 500)
    private String note;

    @ManyToOne
    @JoinColumn(name = "category_id", referencedColumnName = "id", foreignKey = @ForeignKey(name = "fk_resource_category_id", value = ConstraintMode.CONSTRAINT))
    private CategoryEntity category;

    public void change(String code, String name, String data, String dataType) {
        this.setCode(code);
        this.setName(name);
        this.setData(data);
        this.setDataType(dataType);
    }

    public void assignParent(ResourceEntity parent) {
        this.setParent(parent);
        if (parent == null) {
            this.setLevel(0);
            this.setTreePathId(this.getId());
        } else {
            if(parent.getId().equals(this.getId())) {
                throw new NotAllowedTenantException("自身不能作为自己的父资源信息");
            }

            this.setLevel(parent.getLevel() + 1);
            this.setTreePathId(String.format("%s-%s", parent.getTreePathId(), this.getId()));
        }
    }

    protected ResourceEntity() {
        this.setCreatedTime(Timestamp.from(Instant.now()));
    }

    protected ResourceEntity(CategoryEntity category) {
        this();
        this.setCategory(category);
    }

    public static ResourceEntity create(CategoryEntity category, String id, String code, String name, String data, String dataType) {
        ResourceEntity resource = new ResourceEntity(category);
        resource.setId(id);
        resource.setTreePathId(resource.getId());
        resource.setLevel(0);
        resource.change(code, name, data, dataType);

        return resource;
    }

    public static ResourceEntity create(CategoryEntity category, String code, String name, String data, String dataType) {
        return create(category,UUIDUtil.getShortUuid(),code,name,data,dataType);
    }
}
