package com.zbkj.service.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.zbkj.common.model.bcx.BcxChannel;
import com.zbkj.common.model.bcx.BcxDepartment;
import com.zbkj.common.response.BcxDepartmentMemberResponse;
import com.zbkj.common.response.BcxDepartmentResponse;
import com.zbkj.service.dao.BcxDepartmentDao;
import com.zbkj.service.service.BcxDepartmentService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * description: 部门表 服务实现类
 * author : linchunpeng
 * date : 2023/8/3
 */
@Service
public class BcxDepartmentServiceImpl extends ServiceImpl<BcxDepartmentDao, BcxDepartment> implements BcxDepartmentService {
    
    @Resource
    private BcxDepartmentDao dao;

    /**
     * description：获取部门tree列表
     * author：linchunpeng
     * date：2023/8/4
     */
    @Override
    public List<BcxDepartmentResponse> findAllTree() {
        List<BcxDepartment> allList = this.list();
        List<BcxDepartmentResponse> responseList = allList.stream().map(this::getBcxDepartmentResponse).collect(Collectors.toList());
        //生成树形结构
        List<BcxDepartmentResponse> rootList = getBcxDepartmentResponseTree(responseList);
        return rootList;
    }


    /**
     * description：获取部门tree列表（含部门成员）
     * author：linchunpeng
     * date：2023/8/4
     */
    @Override
    public List<BcxDepartmentResponse> findAllTreeWithMember(List<BcxDepartmentMemberResponse> memberResponseList) {
        List<BcxDepartment> allList = this.list();
        List<BcxDepartmentResponse> responseList = allList.stream().map(this::getBcxDepartmentResponse).collect(Collectors.toList());

        for (BcxDepartmentResponse response : responseList) {
            for (BcxDepartmentMemberResponse memberResponse : memberResponseList) {
                if (response.getId().longValue() == memberResponse.getDeptId().longValue()) {
                    response.addMember(memberResponse);
                }
            }
        }
        //生成树形结构
        List<BcxDepartmentResponse> rootList = getBcxDepartmentResponseTree(responseList);
        return rootList;
    }

    /**
     * description：生成树形结构
     * author：linchunpeng
     * date：2023/8/7
     */
    private List<BcxDepartmentResponse> getBcxDepartmentResponseTree(List<BcxDepartmentResponse> responseList) {
        Map<String, BcxDepartmentResponse> responseMap = responseList.stream().collect(Collectors.toMap(BcxDepartmentResponse::getDeptCode, Function.identity()));
        for (BcxDepartmentResponse response : responseList) {
            BcxDepartmentResponse parent = responseMap.get(response.getParentCode());
            if (parent != null) {
                parent.addChildren(response);
            }
        }
        return responseList.stream().filter(dept -> dept.getParentCode().equals("0")).collect(Collectors.toList());
    }

    /**
     * description：获取BcxDepartmentResponse对象
     * author：linchunpeng
     * date：2023/8/4
     */
    private BcxDepartmentResponse getBcxDepartmentResponse(BcxDepartment department) {
        BcxDepartmentResponse response = new BcxDepartmentResponse();
        response.setId(department.getId());
        response.setDeptCode(department.getCode());
        response.setParentCode(department.getParentCode());
        response.setDeptName(department.getName());
        return response;
    }

    /**
     * description：根据v5id查询部门
     * author：linchunpeng
     * date：2023/8/21
     */
    @Override
    public BcxDepartment findByBcxId(String bcxId) {
        QueryWrapper<BcxDepartment> queryWrapper = Wrappers.query();
        queryWrapper.eq("bcx_id", bcxId);
        return this.getOne(queryWrapper);
    }
}