package com.bcxin.tenant.bcx.infrastructures.enums;

import lombok.Getter;
import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import java.util.stream.Collectors;

@Getter
public enum MenuMediumType {
    PC(1<<0, "适用于PC端"),
    APP(1<<1, "适用于APP端");

    private final int value;
    private final String name;

    MenuMediumType(int value, String name) {
        this.value = value;
        this.name = name;
    }

    /**
     * 取得显示名称（与字段 name 等价，便于与参考实现对齐）
     */
    public String getTypeName() {
        return this.name;
    }

    /**
     * 取得位值（便于与参考实现对齐）
     */
    public int getTypeValue() {
        return this.value;
    }

    /**
     * 将字符串集合（使用枚举常量名：如 "PC"、"APP"）转换为枚举集合
     */
    public static Set<MenuMediumType> toMenuMediumTypes(Set<String> mediumTypeNames) {
        if (mediumTypeNames == null || mediumTypeNames.isEmpty()) {
            return Collections.emptySet();
        }
        return mediumTypeNames.stream()
                .filter(name -> name != null && !name.isEmpty())
                .map(name -> {
                    try {
                        return MenuMediumType.valueOf(name);
                    } catch (IllegalArgumentException ex) {
                        return null;
                    }
                })
                .filter(mm -> mm != null)
                .collect(Collectors.toSet());
    }

    /**
     * 判断是否包含指定媒介类型（传入的集合是字符串常量名集合）
     */
    public static boolean hasMenuMediumType(Set<String> mediumTypeNames, MenuMediumType target) {
        if (target == null) {
            return false;
        }
        Set<MenuMediumType> set = toMenuMediumTypes(mediumTypeNames);
        return set.contains(target);
    }

    /**
     * 计算位值：对传入的枚举集合按位或累积（组合多个位标志）
     */
    public static int getCalculatedMediumValue(Set<MenuMediumType> mediums) {
        int calculatedValue = 0;
        if (mediums == null || mediums.isEmpty()) {
            return calculatedValue;
        }
        for (var m : mediums) {
            calculatedValue = calculatedValue | m.getTypeValue();
        }
        return calculatedValue;
    }

    /**
     * 根据传入的字符串常量名集合计算位值
     */
    public static int getCalculatedMediumValueByNames(Collection<String> mediumTypeNames) {
        if (mediumTypeNames == null || mediumTypeNames.isEmpty()) {
            return 0;
        }
        Set<MenuMediumType> mediums = mediumTypeNames.stream()
                .filter(name -> name != null && !name.isEmpty())
                .map(name -> {
                    try {
                        return MenuMediumType.valueOf(name);
                    } catch (IllegalArgumentException ex) {
                        return null;
                    }
                })
                .filter(mm -> mm != null)
                .collect(Collectors.toSet());
        return getCalculatedMediumValue(mediums);
    }

}
