package com.bcxin.tenant.bcx.domains.services.impls;

import com.bcxin.tenant.bcx.domains.entities.ExamEnrollmentCriteriaEntity;
import com.bcxin.tenant.bcx.domains.entities.MetaModuleEntity;
import com.bcxin.tenant.bcx.domains.repositories.ExamEnrollmentCriteriaRepository;
import com.bcxin.tenant.bcx.domains.repositories.MetaModuleRepository;
import com.bcxin.tenant.bcx.domains.services.ExamEnrollmentCriteriaService;
import com.bcxin.tenant.bcx.domains.services.MetaPageService;
import com.bcxin.tenant.bcx.domains.services.commands.DeleteDataCommand;
import com.bcxin.tenant.bcx.domains.services.commands.ExamEnrollmentCriterias.CreateExamEnrollmentCriteriaCommand;
import com.bcxin.tenant.bcx.domains.services.commands.ExamEnrollmentCriterias.UpdateExamEnrollmentCriteriaCommand;
import com.bcxin.tenant.bcx.domains.services.commands.pages.CreateMetaPageCommand;
import com.bcxin.tenant.bcx.domains.services.commands.pages.UpdateMetaPageCommand;
import com.bcxin.tenant.bcx.domains.utils.BusinessConstants;
import com.bcxin.tenant.bcx.infrastructures.TenantContext;
import com.bcxin.tenant.bcx.infrastructures.TenantEmployeeContext;
import com.bcxin.tenant.bcx.infrastructures.UnitWork;
import com.bcxin.tenant.bcx.infrastructures.components.IdWorker;
import com.bcxin.tenant.bcx.infrastructures.enums.PageBusinessType;
import com.bcxin.tenant.bcx.infrastructures.enums.PageType;
import com.bcxin.tenant.bcx.infrastructures.exceptions.ForbidTenantException;
import com.bcxin.tenant.bcx.infrastructures.exceptions.NoFoundTenantException;
import com.bcxin.tenant.bcx.infrastructures.exceptions.UnAuthorizedTenantException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.util.concurrent.atomic.AtomicReference;

@Service
public class ExamEnrollmentCriteriaServiceImpl implements ExamEnrollmentCriteriaService {
    private static final Logger logger = LoggerFactory.getLogger(ExamEnrollmentCriteriaServiceImpl.class);

    private final MetaPageService metaPageService;
    private final ExamEnrollmentCriteriaRepository examEnrollmentCriteriaRepository;
    private final MetaModuleRepository moduleRepository;
    private final UnitWork unitWork;
    private final IdWorker idWorker;

    public ExamEnrollmentCriteriaServiceImpl(MetaPageService metaPageService,
                                             ExamEnrollmentCriteriaRepository examEnrollmentCriteriaRepository,
                                             MetaModuleRepository moduleRepository, UnitWork unitWork, IdWorker idWorker) {
        this.metaPageService = metaPageService;
        this.examEnrollmentCriteriaRepository = examEnrollmentCriteriaRepository;
        this.moduleRepository = moduleRepository;
        this.unitWork = unitWork;
        this.idWorker = idWorker;
    }

    @Override
    public String dispatch(CreateExamEnrollmentCriteriaCommand command) {
        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }

        MetaModuleEntity moduleEntity =
        this.moduleRepository.getById(BusinessConstants.SYS_DEFAULT_EXAM_ENROLLMENT_CRITERIA_MODULE_ID);
        if(moduleEntity==null) {
            throw new ForbidTenantException("找不到此系统模块, 无法进行自定义报名条件页面创建");
        }

        if(!moduleEntity.getProjectId().equals(BusinessConstants.SYS_DEFAULT_PROJECT_ID)) {
            throw new ForbidTenantException("找不到此系统项目");
        }

        AtomicReference<String> pageId = new AtomicReference<>();
        this.unitWork.executeTran(() -> {
            String id =
                    this.metaPageService.dispatch(
                            CreateMetaPageCommand.create(
                                    BusinessConstants.SYS_DEFAULT_PROJECT_ID,
                                    BusinessConstants.SYS_DEFAULT_EXAM_ENROLLMENT_CRITERIA_MODULE_ID,
                                    command.getName(),
                                    command.getNote(),
                                    command.getDefaultFormJson(),
                                    command.getSlug(),
                                    command.getLogo(),
                                    PageType.Form,
                                    PageBusinessType.ExamEnrollmentCriteria,
                                    BusinessConstants.SYS_DYNAMIC_EXAM_ENROLLMENT_MAPPING_ID,
                                    command.isEnableBPMFlow(),
                                    command.isEnableAnonymousAccess(),
                                    command.isSync2Identity(),
                                    command.getBeginTime(),
                                    command.getEndTime(),
                                    command.getAttachment(),
                                    command.getLimitedCount()
                            )
                    );

            pageId.set(id);
            ExamEnrollmentCriteriaEntity entity = ExamEnrollmentCriteriaEntity.create(
                    pageId.get(),
                    command.getSubjectType(),
                    command.getSubjectNumber(),
                    userModel
            );
            entity.change(command.getName(), command.getNote(),
                    userModel);
            this.examEnrollmentCriteriaRepository.insert(entity);
        });

        return pageId.get();
    }

    @Override
    public void dispatch(UpdateExamEnrollmentCriteriaCommand command) {
        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }

        ExamEnrollmentCriteriaEntity entity =
                this.examEnrollmentCriteriaRepository.getById(command.getId());
        if (entity == null) {
            throw new UnAuthorizedTenantException();
        }

        if (!entity.getReferenceNumber().equalsIgnoreCase(userModel.getOrganizationId())) {
            throw new ForbidTenantException(String.format("不允许操作非本组织(%s)的数据(%s)", userModel.getOrganizationId(), entity.getReferenceNumber()));
        }

        this.unitWork.executeTran(() -> {
            entity.change(command.getName(), command.getNote(),
                    userModel);
            this.examEnrollmentCriteriaRepository.update(entity);

            this.metaPageService.dispatch(UpdateMetaPageCommand.create(
                    entity.getId(),
                    command.getName(),
                    command.getNote(),
                    command.getSlug(),
                    command.getLogo(),
                    PageType.Form,
                    PageBusinessType.ExamEnrollmentCriteria,
                    BusinessConstants.SYS_DYNAMIC_EXAM_ENROLLMENT_MAPPING_ID,
                    command.isEnableBPMFlow(),
                    command.isEnableAnonymousAccess(),
                    command.isSync2Identity(),
                    command.getBeginTime(),
                    command.getEndTime(),
                    command.getAttachment(),
                    command.getLimitedCount()
            ));
        });
    }

    @Override
    public void dispatch(DeleteDataCommand command) {
        ExamEnrollmentCriteriaEntity entity = this.examEnrollmentCriteriaRepository.getById(command.getId());
        if (entity == null) {
            throw new NoFoundTenantException();
        }
        TenantEmployeeContext.TenantUserModel userModel = TenantContext.getInstance().getUserContext().get();
        if (userModel == null) {
            throw new UnAuthorizedTenantException();
        }

        if (!userModel.getOrganizationId().equals(entity.getReferenceNumber())) {
            throw new ForbidTenantException("无权限删除该表单");
        }

        this.unitWork.executeTran(() -> {
            this.examEnrollmentCriteriaRepository.delete(entity);
        });
    }
}
