package com.bcxin.risk.sys;

import cn.hutool.core.map.MapUtil;
import com.bcxin.risk.common.dao.DictDao;
import com.bcxin.risk.common.domain.Dict;
import com.bcxin.risk.common.mapper.JsonMapper;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;

/**
 * 字典工具类
 * @author luopeng
 */
@Service
@Transactional
@Slf4j
public class SysDictUtil {

	@Resource
	private DictDao dictDao;

	private Map<String,Collection<Dict>> CACHE_KEY_DICT = Maps.newHashMap();

	public String getDictLabel(String value, String type, String defaultValue){
		if (StringUtils.isNotBlank(type) && StringUtils.isNotBlank(value)){
			Collection<Dict> list = dictMap().get(type.toUpperCase());
			for (Dict dict : list){
				if (value.equals(dict.getValue())){
					return dict.getLabel();
				}
			}
		}
		return defaultValue;
	}

	public String getDictLabelByLikeType(String value, String type, String defaultValue){
		if (StringUtils.isNotBlank(type) && StringUtils.isNotBlank(value)){
			Dict dict = getDictByLikeType(value,type);
			if(dict != null) {
				return dict.getLabel();
			}
		}
		return defaultValue;
	}


	public Collection<Dict> getDictList(String type){
		return dictMap().get(type.toUpperCase()); //因为数据字典类型均为大写
	}


	public Map<String,String> getDictMap(String type){
		Collection<Dict> list = dictMap().get(type.toUpperCase()); //因为数据字典类型均为大写
		Map<String,String> map = new HashMap<>();
		for(Dict dict : list){
			map.put(dict.getValue(),dict.getLabel());
		}
		return map;
	}

	/**
	 * 获取dictList
	 * @return
	 */
	private Map<String,Collection<Dict>> dictMap(){
		if (MapUtil.isNotEmpty(CACHE_KEY_DICT)) {
			return CACHE_KEY_DICT;
		}
		List<Dict> dictList = dictDao.selectDictList();
		Multimap<String, Dict> map = ArrayListMultimap.create();
		for(Dict dict: dictList){
			map.put(dict.getType().toUpperCase(),dict);
		}
		CACHE_KEY_DICT = map.asMap();
		return CACHE_KEY_DICT;
	}

	/**
	 * 返回字典列表（JSON）
	 * @param type
	 * @return
	 */
	public String getDictListJson(String type){
		return JsonMapper.toJsonString(getDictList(type.toUpperCase()));
	}

	/**
	 * 根据type去数据库模糊查询
	 * @param type
	 * @return
	 */
	public List<Dict> getDictListByLikeType(String type) {
		List<Dict> dictList = Lists.newArrayList();
		Set<String> keyset = CACHE_KEY_DICT.keySet();
		for (String key : keyset) {
			if (key.startsWith(type)) {
				dictList.addAll(CACHE_KEY_DICT.get(key));
			}
		}
		return dictList;
	}

	/**
	 * 根据type去数据库模糊查询
	 * @param type
	 * @return
	 */
	public Dict getDictByLikeType(String data,String type) {
		List<Dict> dictList = getDictListByLikeType(type);
		for (Dict dict:dictList) {
			if (Objects.equals(dict.getValue(),data)) {
				return dict;
			}
		}
		return null;
	}
}
