package com.bcxin.risk.common.util;

import cn.hutool.core.date.DatePattern;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.io.FileUtil;
import cn.hutool.poi.excel.ExcelUtil;
import cn.hutool.poi.excel.ExcelWriter;
import com.bcxin.risk.base.domain.util.StringUtil;
import com.bcxin.risk.common.obs.OBSUtil;
import com.bcxin.risk.common.oss.OSSServiceExecutor;
import com.bcxin.risk.constant.Const;
import com.bcxin.risk.exception.FileException;
import com.bcxin.risk.sys.resources.ConfigResources;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.io.Files;
import lombok.Cleanup;
import lombok.extern.slf4j.Slf4j;
import net.coobird.thumbnailator.Thumbnails;
import org.apache.commons.lang3.StringUtils;
import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipOutputStream;
import org.springframework.web.multipart.MultipartFile;

import javax.imageio.ImageIO;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.util.*;
import java.util.List;
import java.util.regex.Pattern;

@Slf4j
public class FileUtils {

	/**
	 * 文件加时间戳的连接符
	 */
	private final static String TI = "_t_";

	private static Pattern FilePattern = Pattern.compile("[\\\\/:*?\"<>|]");

	public static String filenameFilter(String str) {
		return str==null?null:FilePattern.matcher(str).replaceAll("");
	}
	/**
	 * 默认文件上传路径
	 */
	private final static String DEFAULT_UPLOAD_DIR = "upload";

	private final static String DEFAULT_UPLOAD_LINUX = "/data/upload/pss/";

	/**
	 * 保存上传的文件
	 * 返回：/getResource.do?path=20180901/abc.doc
	 * @param file
	 * @return
	 */
	public static String storeFile(MultipartFile file){
		String upload_path = ConfigUtil.getValue("materialUpload_path");
		Map<String, String> rMap = FileUtils.saveFile(upload_path, file);
		if (null != rMap) {
			return rMap.get("url");
		} else {
			throw new RuntimeException("上传失败:" + file.getOriginalFilename());
		}
	}

	/**
	 * 华为OBS文件上传存储（同步）
	 *
	 * @return
	 */
	public static String huaweiOBSFileUpload(MultipartFile xFile,String dir,boolean radomFile,String customFileName) {
		/* 获得文件后缀名 jpg*/
		String fileType = Files.getFileExtension(xFile.getOriginalFilename());
		String fileName = customFileName;
		if ( radomFile ) {
			String randomNumber = BcxinDateUtil.generatorRadomNumber();
			fileName = randomNumber + Const.DOT + fileType;
		}
		if(StringUtils.isNotEmpty(dir)){
			dir = dir.startsWith("/")?dir:"/"+dir;
			dir = dir.endsWith("/")?dir.substring(0,dir.length()-1):dir;
		}
		/* 生成目录 */
		String directoryName = DEFAULT_UPLOAD_DIR+ dir + "/" + DateUtil.today() + "/";
		String ossKey = DEFAULT_UPLOAD_DIR+ dir + "/" + DateUtil.today() + "/" + fileName;
		OBSUtil obsUtil = new OBSUtil();
		/* 利用jdk7的特性，执行完后自动关闭流对象 */
		try (InputStream inputStream = xFile.getInputStream()) {
			return obsUtil.put(directoryName, ossKey, inputStream);
		} catch (Exception e) {
			e.printStackTrace();
			log.error("huaweiOBSFileUpload:"+e.getMessage(), e);
		}
		return "";
	}

	/**
	 * @param materialFile
	 * @param directoryName
	 * @param radomFile
	 * @param customFileName
	 * @return
	 */
	public static Map<String,String> ossFileUpload(MultipartFile materialFile,String directoryName,boolean radomFile,String customFileName){
		Map<String,String> map = Maps.newHashMap();
		String materialFileName = materialFile.getOriginalFilename(); // a.jpg
		String fileType = StringUtils.substringAfterLast(materialFileName, Const.DOT).toLowerCase(); // jpg
		String fileName = Const.BLANK_CHAR;
		if ( radomFile ) {
			String radomNumber = BcxinDateUtil.generatorRadomNumber(); // 12345
			fileName = radomNumber + Const.DOT + fileType; // 12345.jpg
		} else {
			fileName = customFileName;
		}
		String ossKey = directoryName + fileName;
		OSSServiceExecutor ossExecutor = new OSSServiceExecutor(Const.IMAGE_FORM.contains(fileType));
		try {
			if ( !ossExecutor.isExitsDirectory(directoryName) ) {
				ossExecutor.createDirectory(directoryName);
			}
			new Thread(()->{ //异步上传处理
				try {
					ossExecutor.put(ossKey, materialFile.getInputStream(), true);
				} catch(Exception e) {
					e.printStackTrace();
					log.error(e.getMessage(), e);
				}
			}).start();

			String url = ossExecutor.getServer() + ossKey;
			map.put("url", url);
			map.put("key", ossKey);
		} catch ( Exception e) {

		}
		return map;
	}


	/**
	 * @param materialFile
	 * @param directoryName
	 * @param radomFile
	 * @param customFileName  (abc)
	 * @return
	 */
	public static Map<String,String> ossFileUpload_sysc(MultipartFile materialFile,String directoryName,boolean radomFile,String customFileName){
		Map<String,String> map = new HashMap<String,String>();
		String materialFileName = materialFile.getOriginalFilename(); // a.jpg
		String fileType = StringUtils.substringAfterLast(materialFileName, Const.DOT).toLowerCase(); // jpg
		String fileName = Const.BLANK_CHAR;
		if ( radomFile ) {
			String radomNumber = BcxinDateUtil.generatorRadomNumber(); // 12345
			fileName = radomNumber + Const.DOT + fileType; // 12345.jpg
		} else {
			fileName = customFileName;
		}
		String ossKey = directoryName + fileName;
		OSSServiceExecutor ossExecutor = new OSSServiceExecutor(Const.IMAGE_FORM.contains(fileType));
		try {
			if ( !ossExecutor.isExitsDirectory(directoryName) ) {
				ossExecutor.createDirectory(directoryName);
			}
			try {
				ossExecutor.put(ossKey, materialFile.getInputStream(), true);
			} catch(Exception e) {
				e.printStackTrace();
				log.error(e.getMessage(), e);
			}

			String url = ossExecutor.getServer() + ossKey;
			map.put("url", url);
			map.put("key", ossKey);
		} catch ( Exception e) {
			e.printStackTrace();
			log.error(e.getMessage(), e);
		}
		return map;
	}

	/**
	 * 创建ZIP文件
	 * @param sourcePath 文件或文件夹路径
	 * @param zipPath 生成的zip文件存在路径（包括文件名）
	 */
	public static void createZip(String sourcePath, String zipPath) {
		FileOutputStream fos = null;
		ZipOutputStream zos = null;
		try {
			fos = new FileOutputStream(zipPath);
			zos = new ZipOutputStream(fos);
			zos.setEncoding("utf8");//此处修改字节码方式。
			writeZip(new File(sourcePath), "", zos);
		} catch (FileNotFoundException e) {
			log.error("创建ZIP文件失败",e);
			throw new RuntimeException(e);
		} finally {
			try {
				if (zos != null) {
					zos.close();
				}
			} catch (IOException e) {
				log.error("创建ZIP文件失败",e);
				throw new RuntimeException(e);
			}

		}
	}

	private static void writeZip(File file, String parentPath, ZipOutputStream zos) {
		if(file.exists()){
			if(file.isDirectory()){//处理文件夹
				parentPath+=file.getName()+File.separator;
				File [] files=file.listFiles();
				if(files.length != 0)
				{
					for(File f:files){
						writeZip(f, parentPath, zos);
					}
				}
				else
				{       //空目录则创建当前目录
					try {
						zos.putNextEntry(new ZipEntry(parentPath));
					} catch (IOException e) {
						e.printStackTrace();
						throw new RuntimeException(e);
					}
				}
			}else{
				FileInputStream fis=null;
				try {
					fis=new FileInputStream(file);
					ZipEntry ze = new ZipEntry(parentPath + file.getName());
					zos.putNextEntry(ze);
					byte [] content=new byte[1024];
					int len;
					while((len=fis.read(content))!=-1){
						zos.write(content,0,len);
						zos.flush();
					}

				} catch (FileNotFoundException e) {
					log.error("创建ZIP文件失败",e);
					throw new RuntimeException(e);
				} catch (IOException e) {
					log.error("创建ZIP文件失败",e);
					throw new RuntimeException(e);
				}finally{
					try {
						if(fis!=null){
							fis.close();
						}
					}catch(IOException e){
						log.error("创建ZIP文件失败",e);
						throw new RuntimeException(e);
					}
				}
			}
		}
	}

	public static String zipFiles(List<String> filePaths, String zipName) {
		byte[] buffer = new byte[1024];
		String strZipPath = zipName;
		try {
			ZipOutputStream out = new ZipOutputStream(new FileOutputStream(strZipPath));
			List<File> files = Lists.newArrayList();
			for (String filePath : filePaths) {
				if (StringUtil.isNotEmpty(filePath)){
					File policyFile = new File(filePath);
					if(policyFile.exists()) {
						files.add(policyFile);
					}
				}
			}

			for (File file : files) {
				FileInputStream fis = new FileInputStream(file);
				out.putNextEntry(new ZipEntry(file.getName()));
				// 设置压缩文件内的字符编码，不然会变成乱码
				out.setEncoding("utf8");
				int len;
				// 读入需要下载的文件的内容，打包到zip文件
				while ((len = fis.read(buffer)) > 0) {
					out.write(buffer, 0, len);
				}
				out.closeEntry();
				fis.close();
			}
			out.close();

			for (File file : files) {
				if (file.isFile() && file.exists()) {
					file.delete();
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
		}

		return null;
	}

	public static void downloadFile(String fileName, String downloadName, HttpServletResponse response){
		try {
			response.setContentType("octets/stream");
			response.addHeader("Content-Type", "text/html; charset=utf-8");
			String downLoadName = new String(downloadName.getBytes("gbk"), "iso8859-1");
			response.addHeader("Content-Disposition", "attachment;filename=" + downLoadName);
			FileInputStream fileInputStream = new FileInputStream(fileName);
			OutputStream out = response.getOutputStream();
			int i = 0;
			while ((i = fileInputStream.read()) != -1) {
				out.write(i);
			}
			fileInputStream.close();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}


	/**
	 *
	 * <b>Function </b> 本地文件存储
	 *
	 * @return
	 * @author LuoPeng
	 * @date 2017年1月3日 下午8:34:36
	 * @注意事项 </b>
	 * <b>
	 */
	public static Map<String, String> saveFile(String uploadPath,MultipartFile multipartFile) {
		if ( multipartFile != null ) {
			if (StringUtil.isNotEmpty(multipartFile.getOriginalFilename())) {
				try {
					Date now = new Date();
					String materialFileName = now.getTime() + StringUtil.randomString(5) + Const.DOT; // a.jpg
					String fileType = Files.getFileExtension(multipartFile.getOriginalFilename()); // jpg
					materialFileName = materialFileName + fileType;
					String dirDate = DateUtil.today();
					File dir = new File(uploadPath + dirDate);
					if(!dir.exists()) {
						dir.mkdirs();
					}
					String url = "/getResource.do?path=" + dirDate + "/" + materialFileName;
					String targetPath = uploadPath + dirDate + "/" + materialFileName;
					File targetFile = new File(targetPath);
					log.info("执行本地文件上传..." + targetFile);
					multipartFile.transferTo(targetFile);
					Map<String,String> map = Maps.newHashMap();
					map.put("url", url);
					map.put("key", "local_"+now.getTime());
					log.info("本地文件上传成功!");
					return map;
				} catch(Exception e) {
					e.printStackTrace();
					log.error(e.getMessage(), e);
					return null;
				}
			}
		}
		return null;
	}


	/**
	 * <b>Function </b> 本地文件存储，缩放
	 *
	 * @return
	 * @author 张建华
	 * @date 2018年10月19日11:02:00
	 * @注意事项 </b>
	 * <b>
	 */
	public static Map<String, String> saveFileByScale(String uploadPath, MultipartFile multipartFile) {
		if (multipartFile != null) {
			if (StringUtil.isNotEmpty(multipartFile.getOriginalFilename())) {
				try {
					Date now = new Date();
					String materialFileName = now.getTime() + StringUtil.randomString(5) + Const.DOT;
					String fileType = Files.getFileExtension(multipartFile.getOriginalFilename());
					materialFileName = materialFileName + fileType;
					String scaleName = now.getTime() + StringUtil.randomString(5) + Const.DOT + fileType;
					String dirDate = DateUtil.today();
					File dir = new File(uploadPath + dirDate);
					if (!dir.exists()) {
						dir.mkdirs();
					}
					String url = "/getResource.do?path=" + dirDate + "/" + materialFileName;
					String scaleUrl = "/getResource.do?path=" + dirDate + "/" + scaleName;
					String targetPath = uploadPath + dirDate + "/" + materialFileName;
					String scaleNameTargetPath = uploadPath + dirDate + "/" + scaleName;
					File targetFile = new File(targetPath);
					File scaleFile = new File(scaleNameTargetPath);
					log.info("执行本地文件上传..." + targetFile);
					multipartFile.transferTo(targetFile);

					//是图片文件则进行压缩
					Image image = ImageIO.read(targetFile);

					Map<String, String> map = Maps.newHashMap();
					if (image != null) {
						//缩放图片
						compressPictures(targetPath, scaleNameTargetPath, targetFile.length());
						if (targetFile.length() > scaleFile.length()) {
							//删除原图，只保留缩放后的图片
							targetFile.delete();
							map.put("url", scaleUrl);
						} else {
							//如果缩放后的图片质量更大，则删除之后的图片。在测试中会出现降低图片质量后，文件大小更大的情况，有待优化。
							scaleFile.delete();
							map.put("url", url);
						}
					} else {
						map.put("url", url);
					}

					map.put("key", "local_" + now.getTime());
					log.info("本地文件上传成功!");
					return map;
				} catch (Exception e) {
					e.printStackTrace();
					log.error(e.getMessage(), e);
					return null;
				}
			}
		}
		return null;
	}


	/**
	 * 其中的scale是可以指定图片的大小，值在0到1之间，1f就是原图大小，0.5就是原图的一半大小，这里的大小是指图片的长宽。
	 * <p>
	 * 而outputQuality是图片的质量，值也是在0到1，越接近于1质量越好，越接近于0质量越差。
	 *
	 * @param oldPicPath 原图文件的路径
	 * @param newPicPath 压缩后文件的路径
	 * @param fileLength 文件大小 b
	 * @Author 张建华
	 */
	public static void compressPictures(String oldPicPath, String newPicPath, Long fileLength) {

		float quality = 1f;

		if (fileLength > 102400 && fileLength < 512000) {
			quality = 0.6f;
		} else if (
				fileLength > 512000 && fileLength < 1024000) {
			quality = 0.5f;
		} else if (fileLength > 1024000 && fileLength < 5120000) {
			quality = 0.3f;
		} else if (fileLength > 10240000) {
			quality = 0.1f;
		}
		try {
			Thumbnails.of(oldPicPath)
					.scale(1f)
					.outputQuality(quality)
					.toFile(newPicPath);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}


	/**
	 * 缩放图像（按比例缩放）
	 * <p>
	 * scale        缩放比例
	 *
	 * @param srcImageFile 源图像文件地址
	 * @param result       缩放后的图像地址
	 * @param flag         缩放选择:true 放大; false 缩小;
	 * @param length       文件大小
	 * @Author 张建华
	 */
	public static void scale(String srcImageFile, String result,
							 float length, boolean flag) {

		float scale = 1;

		if (length > 20000) {
			scale = 1.5f;
		} else if (length > 50000) {
			scale = 2.0f;
		} else if (length > 100000) {
			scale = 3.0f;
		}
		try {
			// 读入文件
			BufferedImage src = ImageIO.read(new File(srcImageFile));
			// 得到源图宽
			int width = src.getWidth();
			// 得到源图长
			int height = src.getHeight();
			// 放大
			if (flag) {
				width = (int) (width * scale);
				height = (int) (height * scale);
			} else {
				// 缩小
				width = (int) (width / scale);
				height = (int) (height / scale);
			}
			Image image = src.getScaledInstance(width, height,
					Image.SCALE_DEFAULT);
			BufferedImage tag = new BufferedImage(width, height,
					BufferedImage.TYPE_INT_RGB);
			Graphics g = tag.getGraphics();
			// 绘制缩小后的图
			g.drawImage(image, 0, 0, null);
			g.dispose();
			// 输出到文件流
			ImageIO.write(tag, "JPEG", new File(result));
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 *
	 * <b>Function </b> 本地文件存储
	 * @return 文件全url
	 * <b>
	 */
	public static String saveFile(MultipartFile multipartFile) {
		if ( multipartFile != null ) {
			if (StringUtil.isNotEmpty(multipartFile.getOriginalFilename())) {
				try {
					Date now = new Date();
					String materialFileName = now.getTime() + StringUtil.randomString(5) + Const.DOT; // a.jpg
					String fileType = Files.getFileExtension(multipartFile.getOriginalFilename()); // jpg
					materialFileName = materialFileName + fileType;
					String dirDate = DateUtil.today();
					File dir = new File(ConfigUtil.material() + dirDate);
					if(!dir.exists()) {
						dir.mkdirs();
					}
					String url = ConfigUtil.webSiteUrl() + "/getResource.do?path=" + dirDate + "/" + materialFileName;
					String targetPath = ConfigUtil.material() + dirDate + "/" + materialFileName;
					File targetFile = new File(targetPath);
					log.info("执行本地文件上传..." + targetFile);
					multipartFile.transferTo(targetFile);
					log.info("本地文件上传成功!");
					return url;
				} catch(Exception e) {
					e.printStackTrace();
					log.error(e.getMessage(), e);
					return null;
				}
			}
		}
		return null;
	}


	/**
	 *
	 * <b>Function </b> 本地文件存储
	 *
	 * @return
	 * @author LuoPeng
	 * @date 2017年1月3日 下午8:34:36
	 * @注意事项 </b>
	 * <b>
	 */
	public static Map<String, String> saveFile(String uploadPath,File file) {
		if( file != null ) {
			try {
				Date now = new Date();
				String materialFileName = now.getTime() + StringUtil.randomString(5) + Const.DOT; // 20100101123123.
				String fileType = Files.getFileExtension(file.getPath()); // jpg
				materialFileName = materialFileName + fileType; // 20100101123123.jpg
				String dirDate = DateUtil.today();
				File dir = new File(uploadPath + dirDate);
				if(!dir.exists()) {
					dir.mkdirs();
				}
				String url = "/getResource.do?path=" + dirDate + "/" + materialFileName;
				String targetPath = uploadPath + dirDate + "/" + materialFileName;
				File targetFile = new File(targetPath);
				log.info("执行本地文件上传..." + targetFile);
				Files.copy(file,targetFile);
				Map<String,String> map = Maps.newHashMap();
				map.put("url", url);
				map.put("key", "local_"+now.getTime());
				log.info("本地文件上传成功!");
				return map;
			} catch(Exception e) {
				e.printStackTrace();
				log.error(e.getMessage(), e);
				return null;
			}
		}
		return null;
	}


	/**
	 * 上传文件
	 *
	 * @param file
	 *            文件
	 * @param radomfileName
	 *            保存文件名
	 * @param materialFilePath
	 *            保存路径
	 * @return
	 */
	public static boolean fileUpload(MultipartFile file, String radomfileName,
									 String materialFilePath) {
		boolean flag = true;
		File baseFile = new File(materialFilePath);
		File trainFile = new File(baseFile, radomfileName);
		if (!baseFile.exists()) {
			baseFile.mkdirs();
		}
		try {
			file.transferTo(trainFile);
		} catch (IllegalStateException e) {
			flag = false;
		} catch (IOException e) {
			flag = false;
		}// 保存文件
		return flag;
	}


	public static void deleteFolder(File file) {
		if(file.exists()) {
			if(file.isDirectory()) {
				for(File f : file.listFiles()) {
					deleteFolder(f);
				}
				file.delete();
			} else {
				file.delete();
			}
		}
	}


	/**
	 * 根据系统的地址，下载文件到新的路径
	 * @param urlStr
	 * @param fileName 指定的文件名
	 * @param savePath
	 * @return
	 * @throws IOException
	 */
	public static List<String> downLoadFromUrl(String urlStr, String fileName, String savePath){
		List<String> pathList = Lists.newArrayList();
		try {
			if (StringUtil.isEmpty(urlStr)) {
				return Lists.newArrayList();
			}
			if (StringUtil.isNotEmpty(fileName)) {
				fileName = fileName.replace("/", "");
			}
			/* 分解成多个可以解析的url */
			String[] imageValues = urlStr.split("&&");
			int count = 1;
			for (String imageValue : imageValues) {
				String disposeFileName = fileName;
				String imagePath = "";
				/* 使用的本地的那种组装式url */
				if (imageValue.contains("||")) {
					String[] splitStr = StringUtil.split(imageValue, "||");  //再从一张图片url里面切分到url
					if (splitStr.length >= 3) {
						imagePath = splitStr[1];
					} else {
						imagePath = urlStr;
					}
				}
				/* 使用正常的 url */
				else {
					imagePath = imageValue;
				}
				/* 处理多个url时，组装不同的名称 */
				if (count > 1) {
					if (disposeFileName.contains(".")) {
						disposeFileName = disposeFileName.substring(0, disposeFileName.lastIndexOf(".")) + count;
					} else {
						disposeFileName = disposeFileName + count;
					}
				}
				System.out.println("------------------重新计算的name是"+disposeFileName+"------------------");
				/* http的请求 */
				if (imagePath.startsWith("http")) {
					disposeFileName = disposeFileName + imagePath.substring(imagePath.lastIndexOf("."), imagePath.length());
					String filePath = download(imagePath, disposeFileName, savePath);
					if (StringUtil.isNotEmpty(filePath)) {
						pathList.add(filePath);
					}
				}
				/* 本地组装式的请求 */
				else if (imagePath.indexOf("path=") > 0) {
					String web_url = ConfigUtil.webSiteUrl();
					/* 如果是开发环境，则获取本地的地址 */
					String PROFILE_ENVI = ConfigResources.PROFILE_ENVI;
					if (Objects.equals(PROFILE_ENVI, "dev")) {
						web_url = ConfigResources.DEV_URL;
					}
					if (StringUtil.isNotEmpty(web_url)) {
						if (web_url.endsWith("/") && imagePath.endsWith("/")) {
							web_url = web_url.substring(0, web_url.length() - 1);
						}
					}
					imagePath = web_url + imagePath;
					disposeFileName = disposeFileName + imagePath.substring(imagePath.lastIndexOf("."), imagePath.length());
					String filePath = download(imagePath, disposeFileName, savePath);
					if (StringUtil.isNotEmpty(filePath)) {
						pathList.add(filePath);
					}
				}
				count ++;
			}
			return pathList;
		} catch (Exception e) {
			e.printStackTrace();
			log.error("文件下载异常："+e.getMessage());
			return pathList;
		}
	}

	/**
	 * 根据 url 下载
	 * @param urlStr
	 * @param fileName
	 * @param savePath
	 * @return
	 * @throws FileException
	 */
	public static String download(String urlStr, String fileName, String savePath) {
		FileOutputStream fos = null;
		InputStream inputStream = null;
		try {
			URL url = new URL(urlStr);
			HttpURLConnection conn = (HttpURLConnection) url.openConnection();
			//设置超时间为10秒
			conn.setConnectTimeout(10 * 1000);
			//防止屏蔽程序抓取而返回403错误
			conn.setRequestProperty("User-Agent", "Mozilla/4.0 (compatible; MSIE 5.0; Windows NT; DigExt)");
			//得到输入流
			inputStream = conn.getInputStream();
			//获取自己数组
			byte[] getData = readInputStream(inputStream);
			//文件保存位置
			File saveDir = new File(savePath);
			if (!saveDir.exists()) {
				saveDir.mkdirs();
			}
			File file = new File(saveDir + File.separator + fileName);
			fos = new FileOutputStream(file);
			fos.write(getData);

			String filePath = saveDir + "/" + fileName;
			log.info("文件下载，url路径[{}]，文件路径[{}]", url,filePath);
			return filePath;
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			try {
				if (fos != null) {
					fos.close();
				}
				if (inputStream != null) {
					inputStream.close();
				}
			} catch (IOException e){
				e.printStackTrace();
			}
		}
		return "";
	}


	/**
	 * 根据系统默认存储路径的方式，获取其中的文件路径
	 * example : 参数 local_1497333054610||/getResource.do?path=2017-06-13/1497333054610.jpg||4.jpg&&local_1497333054610||/getResource.do?path=2017-06-13/1497333054611.jpg||4.jpg
	 * result ： 2017-06-13/1497333054610.jpg  2017-06-13/1497333054611.jpg
	 * @param localFilePath
	 * @return
	 */
	public static List<String> filterMultiFilePath(String localFilePath){
		List<String> list = Lists.newArrayList();
		if (StringUtil.isEmpty(localFilePath)) {
			return list;
		}
		String[] filePaths = StringUtils.split(localFilePath, "&&");
		for(String filePath : filePaths){
			if (filePath.indexOf("||")>-1) {
				String[] explain = StringUtils.split(filePath, "||");
				if (explain.length < 3) {
					continue;
				}
				String localUrl = explain[1];
				if (localUrl.indexOf("path=") == -1) {
					continue;
				}
				localUrl = localUrl.substring(localUrl.indexOf("path=")+5,localUrl.length());
				list.add(localUrl);
			}
		}
		return list;
	}


	/**
	 * 根据系统默认存储路径的方式，获取其中的文件路径
	 * example : 参数 local_1497333054610||/getResource.do?path=2017-06-13/1497333054610.jpg||4.jpg&&local_1497333054610||/getResource.do?path=2017-06-13/1497333054611.jpg||4.jpg
	 * result ： /getResource.do?path=2017-06-13/1497333054610.jpg  /getResource.do?path=2017-06-13/1497333054611.jpg
	 * @param localFilePath
	 * @return
	 */
	public static List<String> filterMultiFileUrl(String localFilePath){
		List<String> list = Lists.newArrayList();
		if (StringUtil.isEmpty(localFilePath)) {
			return list;
		}
		String[] filePaths = StringUtils.split(localFilePath, "&&");
		for(String filePath : filePaths){
			if (filePath.indexOf("||")>-1) {
				String[] explain = StringUtils.split(filePath, "||");
				if (explain.length < 3) {
					continue;
				}
				String localUrl = explain[1];
				list.add(localUrl);
			}
		}
		return list;
	}


	/**
	 * 根据系统默认存储路径的方式，获取其中的文件路径
	 * example : 参数 local_1497333054610||/getResource.do?path=2017-06-13/1497333054610.jpg||4.jpg
	 * result ： /getResource.do?path=2017-06-13/1497333054610.jpg
	 * @param localFilePath
	 * @return
	 */
	public static String filterSingleLocalFilePath(String localFilePath){
		if (StringUtil.isEmpty(localFilePath)) {
			return localFilePath;
		}
		if (localFilePath.indexOf("||")>-1) {
			String[] explain = StringUtils.split(localFilePath, "||");
			if (explain.length < 3) {
				return localFilePath;
			}
			return explain[1];
		}
		return localFilePath;
	}


	/**
	 * 根据系统默认存储路径的方式，获取其中的文件路径
	 * example : 参数 local_1497333054610||/getResource.do?path=2017-06-13/1497333054610.jpg||4.jpg&&local_1497333054610||/getResource.do?path=2017-06-13/1497333054610.jpg
	 * result ： http://www.bcxin.com.cn/getResource.do?path=2017-06-13/1497333054610.jpg&&http://www.bcxin.com.cn/getResource.do?path=2017-06-13/1497333054610.jpg
	 * @param localFilePath
	 * @return
	 */
	public static String filterMultiLocalFilePath(String channel,String localFilePath){
		if (StringUtil.isEmpty(localFilePath)) {
			return localFilePath;
		}
		List<String> files = Lists.newArrayList();
		String[] stringList = localFilePath.split("&&");
		for (String path:stringList) {
			String newFilePath=filterSingleLocalFilePath(path);
			if(newFilePath.startsWith("http://") || newFilePath.startsWith("https://")){
				files.add(newFilePath);
			}else {
				files.add(channel + newFilePath);
			}
		}
		return Joiner.on("&&").join(files);
	}


	/**
	 * 根据系统默认存储路径的方式，获取其中的文件路径
	 * example : 参数 /getResource.do?path=2017-06-13/1497333054610.jpg
	 * result ： 2017-06-13/1497333054610.jpg
	 * @param localFilePath
	 * @return
	 */
	public static String filterSingleFilePath(String localFilePath){
		if (StringUtil.isEmpty(localFilePath)) {
			return Const.BLANK_CHAR;
		}
		if ( localFilePath.contains("path=")) {
			return localFilePath.substring(localFilePath.indexOf("path=")+5,localFilePath.length());
		}
		return localFilePath;
	}

	/**
	 * 根据文件存储全路径，获取其中的文件文件夹路径
	 * example : 参数 /data/pss/material/2017-06-13/1497333054610.jpg
	 * result ： /data/pss/material/2017-06-13/
	 * @param filePath
	 * @return
	 */
	public static String filterFileDir(String filePath){
		if (StringUtil.isEmpty(filePath)) {
			return Const.BLANK_CHAR;
		}
		String fileName = Files.getNameWithoutExtension(filePath);
		String type = Files.getFileExtension(filePath);
		return filePath.replaceAll(fileName+"\\."+type,"");
	}


	/**
	 * 根据文件存储URL路径，获取其中的文件文件夹路径
	 * example : 参数 /getResource.do?path=2017-06-13/1497333054610.jpg
	 * result ： 2017-06-13
	 * @param filePath
	 * @return
	 */
	public static String filterUrlFileDir(String filePath){
		if (StringUtil.isEmpty(filePath)) {
			return Const.BLANK_CHAR;
		}
		String localFilePath = filterSingleFilePath(filePath);
		String fileName = Files.getNameWithoutExtension(localFilePath);
		String type = Files.getFileExtension(localFilePath);
		return localFilePath.replaceAll(fileName+"\\."+type,"");
	}

	/**
	 * 根据系统默认存储路径的方式，获取其中的文件名称
	 * example : 参数 /getResource.do?path=2017-06-13/1497333054610.jpg
	 * result ： 1497333054610.jpg
	 * @param localFilePath
	 * @return
	 */
	public static String filterSingleFileName(String localFilePath){
		if (StringUtil.isEmpty(localFilePath)) {
			return Const.BLANK_CHAR;
		}
		if (localFilePath.indexOf("path=") == -1) {
			return Const.BLANK_CHAR;
		}
		String filePath = localFilePath.substring(localFilePath.indexOf("path=")+5,localFilePath.length());
		if (filePath.contains("/")) {
			filePath = filePath.substring(filePath.lastIndexOf("/")+1,filePath.length());
		}
		return filePath;
	}

	private static byte[] readInputStream(InputStream inputStream) throws IOException {
		byte[] buffer = new byte[1024];
		int len = 0;
		ByteArrayOutputStream bos = new ByteArrayOutputStream();
		while((len = inputStream.read(buffer)) != -1) {
			bos.write(buffer, 0, len);
		}
		bos.close();
		return bos.toByteArray();
	}


	/**
	 * 为文件名增加时间戳
	 * @return
	 */
	public synchronized static String packTimestamp(String fileName){
		if (StringUtil.isEmpty(fileName)) {
			return Const.BLANK_CHAR;
		}
		if (fileName.contains(Const.DOT)) {
			return Files.getNameWithoutExtension(fileName) + TI + cn.hutool.core.date.DateUtil.format(new Date(), DatePattern.PURE_DATETIME_MS_PATTERN) + Files.getFileExtension(fileName);
		} else {
			return fileName + TI + cn.hutool.core.date.DateUtil.format(new Date(), DatePattern.PURE_DATETIME_MS_PATTERN) ;
		}
	}


	/**
	 * 处理文件夹路径，并根据文件夹路径创建文件夹
	 * @param paths
	 * @return
	 */
	public static String appendFolderPath(String[] paths){
		StringBuilder sb = new StringBuilder();
		for (String path:paths) {
			if (sb.length()==0) {
				sb.append(path);
			} else {
				if (sb.toString().endsWith("/")) {
					sb.append(path).append("/");
				} else {
					sb.append("/").append(path).append("/");
				}
			}
		}
		makeDir(sb.toString());//创建文件夹
		return sb.toString();
	}


	/**
	 * 创建文件夹
	 * @param dir
	 */
	public static void makeDir(String dir){
		File fileDir = new File(dir);
		if(!fileDir.exists()) {
			fileDir.mkdirs();
		}
	}


	/**
	 * 导出xls数据
	 * @param rename 重命名
	 * @param list N个数据行
	 * @throws IOException
	 */
	public static void downloadXlS(HttpServletResponse response,String rename,List<Map<String, Object>> list) throws IOException {
		downloadXLSPublic(response, rename, list,null);
	}


	private static void downloadXLSPublic(HttpServletResponse response, String rename, List<Map<String, Object>> list, Map<String,String> headers) throws IOException {
		@Cleanup ServletOutputStream out = response.getOutputStream();

		File dir = Files.createTempDir();
		File file = new File(dir.getPath() + "/" + System.currentTimeMillis() + ".xls");
		// 通过工具类创建writer，默认创建xls格式
		ExcelWriter writer = ExcelUtil.getWriter(file);
		if ( headers != null) {
			writer.setHeaderAlias(headers);
		}

		//一次性写出内容，使用默认样式
		writer.write(list);
		//out为OutputStream，需要写出到的目标流
		response.setContentType("application/vnd.ms-excel;charset=utf-8");
		response.setHeader("Content-Disposition", "attachment;filename=" + URLEncoder.encode( rename + ".xls", "utf-8"));

		writer.flush(out);
		writer.close();
		FileUtil.del(file);
	}



	public static void main(String[] args) {
		//System.out.println(filterSingleFilePath("http://bcxin-risk-extranet-test.oss-cn-shanghai.aliyuncs.com/correction/589827/201705311654149062.jpg"));
		//System.out.println(Files.getFileExtension("http://bcxin-risk-extranet-test.oss-cn-shanghai.aliyuncs.com/correction/589827/201705311654149062.jpg"));
		//System.out.println(Files.getNameWithoutExtension("http://bcxin-risk-extranet-test.oss-cn-shanghai.aliyuncs.com/correction/589827/201705311654149062.jpg"));
		//List<String> list = filterMultiFilePath("local_1499751468351||/getResource.do?path=2017-07-11/1499751468351.jpg||code.jpg&&local_1499751468351||/getResource.do?path=2017-07-11/1499751468351.jpg||360.jpg");
		//for(String s:list){
		//	System.out.println(s);
		//}

		//File file = new File("");
		//System.out.print(file instanceof File);
		//List<String> files = Lists.newArrayList();
		//files.add("D:\\data\\pss\\material\\cashier.jsp");
		//files.add("D:\\data\\pss\\material\\shanxi\\2017-08-25\\sp");
		//files.add("D:\\data\\pss\\material\\2017-09-06\\150466837569025235.jpg");
		//System.out.println(filterFileDir("/data/pss/material/2017-06-13/1497333054610.jpg"));

		//zipFiles(files,"D:/我是压缩包.zip");
	}
}
