package com.bcxin.tenant.domain.services.commands.organizations;

import com.bcxin.Infrastructures.commands.CommandAbstract;
import com.bcxin.Infrastructures.enums.CredentialType;
import com.bcxin.Infrastructures.exceptions.BadTenantException;
import lombok.Getter;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;

@Getter
public class RegisterCompanyCommand extends CommandAbstract {
    /**
     * 机构信息
     */
    @NotEmpty
    private final String institutionalCode;
    /**
     * 行业类型
     */
    @NotEmpty
    private final String industryCode;

    /**
     * 组织统一社会信用代码
     */
    @NotEmpty
    private final String unifySocialCreditCode;

    /**
     * 组织统一社会信用代码扫描件
     */
    @NotEmpty
    private final String unifySocialCreditCodeFile;

    /**
     * 组织名称
     */
    @NotEmpty
    private final String name;

    /**
     * 注册地
     */
    @NotNull
    private final LocationCommandItem placeOfRegister;


    /**
     * 经营地
     */
    @NotNull
    private final LocationCommandItem placeOfBusiness;

    /**
     * 管理员
     */
    @NotNull
    private final ContactCommandItem administrator;

    /**
     * 总经理
     */
    @NotNull
    private final ContactCommandItem generalManager;

    /**
     * 副总经理
     */
    @NotNull
    private final ContactCommandItem deputyGeneralManager;

    @Override
    public void validate() {
        if (this.getAdministrator() == null) {
            throw new BadTenantException("管理员不能为空!");
        }

        if (this.getGeneralManager() == null) {
            throw new BadTenantException("总经理不能为空!");
        }

        if (this.getDeputyGeneralManager() == null) {
            throw new BadTenantException("副总经理不能为空!");
        }

        if (!StringUtils.hasLength(this.getInstitutionalCode())) {
            throw new BadTenantException("行业类型!");
        }

        if (!StringUtils.hasLength(this.getIndustryCode())) {
            throw new BadTenantException("机构信息不能为空!");
        }

        if (!StringUtils.hasLength(this.getUnifySocialCreditCode())) {
            throw new BadTenantException("组织统一社会信用代码不能为空!");
        }

        if (!StringUtils.hasLength(this.getUnifySocialCreditCodeFile())) {
            throw new BadTenantException("组织统一社会信用代码扫描件不能为空!");
        }

        if (!StringUtils.hasLength(this.getName())) {
            throw new BadTenantException("组织名称不能为空!");
        }

        this.validateLocation(this.getPlaceOfRegister(), "注册地");
        this.validateLocation(this.getPlaceOfBusiness(), "经营地");
    }

    public RegisterCompanyCommand(String institutionalCode, String industryCode,
                                  String unifySocialCreditCode, String unifySocialCreditCodeFile,
                                  String name, LocationCommandItem placeOfRegister,
                                  LocationCommandItem placeOfBusiness, ContactCommandItem administrator,
                                  ContactCommandItem generalManager, ContactCommandItem deputyGeneralManager) {
        this.institutionalCode = institutionalCode;
        this.industryCode = industryCode;
        this.unifySocialCreditCode = unifySocialCreditCode;
        this.unifySocialCreditCodeFile = unifySocialCreditCodeFile;
        this.name = name;
        this.placeOfRegister = placeOfRegister;
        this.placeOfBusiness = placeOfBusiness;
        this.administrator = administrator;
        this.generalManager = generalManager;
        this.deputyGeneralManager = deputyGeneralManager;
    }


    private void validateLocation(LocationCommandItem commandItem,String category) {
        if (commandItem == null) {
            throw new BadTenantException(String.format("%s不能为空!", category));
        }

        if (commandItem.getProvince() == null && !StringUtils.hasLength(commandItem.getProvince().getCode())) {
            throw new BadTenantException(String.format("%s的省份不能为空!", category));
        }

        if (commandItem.getCity() == null && !StringUtils.hasLength(commandItem.getCity().getCode())) {
            throw new BadTenantException(String.format("%s的城市不能为空!", category));
        }

        if (commandItem.getDistrict() == null && !StringUtils.hasLength(commandItem.getDistrict().getCode())) {
            throw new BadTenantException(String.format("%s的区域不能为空!", category));
        }

        if (!StringUtils.hasLength(commandItem.getAddress())) {
            throw new BadTenantException(String.format("%s的地址不能为空!", category));
        }
    }

    public static RegisterCompanyCommand create(String institutionalCode, String industryCode,
                                                String unifySocialCreditCode, String unifySocialCreditCodeFile,
                                                String name, LocationCommandItem placeOfRegister,
                                                LocationCommandItem placeOfBusiness, ContactCommandItem administrator,
                                                ContactCommandItem generalManager, ContactCommandItem deputyGeneralManager) {
        return new RegisterCompanyCommand(institutionalCode, industryCode, unifySocialCreditCode, unifySocialCreditCodeFile,
                name, placeOfRegister, placeOfBusiness, administrator, generalManager, deputyGeneralManager);
    }

    @Getter
    public static class ContactCommandItem {
        @NotEmpty
        private final String name;
        @NotEmpty
        private final String telephone;
        @NotNull
        private final CredentialType credentialType;
        @NotEmpty
        private final String credentialNumber;

        public ContactCommandItem(String name, String telephone, CredentialType credentialType,
                                  String credentialNumber) {
            this.name = name;
            this.telephone = telephone;
            this.credentialType = credentialType;
            this.credentialNumber = credentialNumber;
        }

        public static RegisterCompanyCommand.ContactCommandItem create(String name, String telephone, CredentialType credentialType,
                                                                       String credentialNumber) {
            return new ContactCommandItem(name, telephone, credentialType, credentialNumber);
        }
    }

    @Getter
    public static class LocationCommandItem
    {
        /**
         * 省份
         */
        private final ItemValueCommandItem province;

        /**
         * 城市
         */
        private final ItemValueCommandItem city;

        /**
         * 区域
         */
        private final ItemValueCommandItem district;

        /**
         * 详细地址
         */
        private final String address;

        public LocationCommandItem(ItemValueCommandItem province,
                                   ItemValueCommandItem city,
                                   ItemValueCommandItem district,
                                   String address) {
            this.province = province;
            this.city = city;
            this.district = district;
            this.address = address;
        }

        public static LocationCommandItem create(ItemValueCommandItem province,
                                                 ItemValueCommandItem city,
                                                 ItemValueCommandItem district,
                                                 String address) {
            return new LocationCommandItem(province, city, district, address);
        }
    }

    @Getter
    public static class ItemValueCommandItem {
        private final String code;
        private final String name;

        public ItemValueCommandItem(String code, String name) {
            this.code = code;
            this.name = name;
        }

        public static ItemValueCommandItem create(String code, String name) {
            return new ItemValueCommandItem(code, name);
        }
    }
}
