package com.bcxin.ars.aspect;

import cn.hutool.core.util.ObjectUtil;
import com.abcxin.smart.validator.annotation.DataChangeAnnotation;
import com.abcxin.smart.validator.enums.ParamType;
import com.alibaba.fastjson.JSON;
import com.bcxin.ars.dto.DataChangeDto;
import com.bcxin.ars.model.sys.DataChange;
import com.bcxin.ars.service.sys.DataChangeService;
import com.bcxin.ars.util.Constants;
import com.bcxin.ars.util.DateUtil;
import com.bcxin.ars.util.StringUtil;
import com.bcxin.ars.util.spring.util.SpringUtils;
import com.bcxin.ars.util.thread.ThreadPoolManager;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.ReflectionUtils;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * 保安员证修改保存记录
 * @author linqinglin
 * @date 2019/08/12 0012 10:54
 */
@Component
@Aspect
public class DataChangeAspect {

    /***
     * 日志
     */
    private static final Logger logger = LoggerFactory.getLogger(DataChangeAspect.class);

    @Autowired
    private DataChangeService dataChangeService;

    @Around("@annotation(com.abcxin.smart.validator.annotation.DataChangeAnnotation)")
    public Object doAround(ProceedingJoinPoint pjp) throws Throwable {

        //前置增强开始
        Signature sig = pjp.getSignature();
        MethodSignature msig = null;
        if (!(sig instanceof MethodSignature)) {
            logger.error("该注解只能用于方法");
            return pjp.proceed();
        }
        msig = (MethodSignature) sig;
        Object target = pjp.getTarget();
        Method currentMethod = target.getClass().getMethod(msig.getName(), msig.getParameterTypes());
        DataChangeAnnotation annotation = currentMethod.getAnnotation(DataChangeAnnotation.class);

        Class entityClazz = annotation.getEntityClazz();
        String fieldName = annotation.getFieldName();
        String methodName = msig.getDeclaringTypeName() + Constants.POINT + msig.getName();
        Object oldValue = null;
        Object newValue = null;
        String tableName = null;
        Object value = null;

        try {
            if(msig.getParameterTypes() != null && msig.getParameterTypes().length > 0){
                value = pjp.getArgs()[0];
                oldValue = getDBValue(value,annotation);

                String tableNameFieldName = annotation.getTableNameField();
                if(StringUtil.isNotEmpty(tableNameFieldName)) {
                    Field tableNameField = ReflectionUtils.findField(entityClazz, tableNameFieldName);
                    tableNameField.setAccessible(true);
                    Object fieldValue = ReflectionUtils.getField(tableNameField, value);
                    if(fieldValue != null){
                        tableName = fieldValue.toString();
                    }
                }
            }
        }catch (Exception e){
            logger.error(e.getMessage(),e);
            e.printStackTrace();
        }

        //前置增强结束
        Object obj = pjp.proceed();

        //后置增强开始
        try {
            if(oldValue != null){
                newValue = getDBValue(value,annotation);
                if(newValue != null) {
                    DataChangeDto dataChangeDto = new DataChangeDto(annotation,oldValue, newValue, methodName, entityClazz.getName(), tableName, fieldName);

                    ThreadPoolManager threadPoolManager = ThreadPoolManager.newInstance();
                    threadPoolManager.addExecuteTask(new Runnable() {
                        @Override
                        public void run() {
                            saveLog(dataChangeDto);
                        }
                    });
                }

            }
        }catch (Exception e){
            logger.error(e.getMessage(),e);
            e.printStackTrace();
        }

        //后置增强结束
        return obj;
    }

    private void saveLog(DataChangeDto dataChangeDto) {
        if(dataChangeDto != null){
            DataChangeAnnotation annotation = dataChangeDto.getAnnotation();
            Object oldValue = dataChangeDto.getOldValue();
            Object newValue = dataChangeDto.getNewValue();
            String methodName = dataChangeDto.getMethodName();
            String fieldName = dataChangeDto.getFieldName();
            Class entityClazz = annotation.getEntityClazz();
            Field field = ReflectionUtils.findField(entityClazz,fieldName);
            field.setAccessible(true);
            if(List.class.isInstance(oldValue)){
                for(Object oldObject: (List)oldValue){
                    Object keyValue = ReflectionUtils.getField(field,oldObject);
                    for(Object newObject: (List)newValue) {
                        if(ObjectUtil.equal(keyValue,ReflectionUtils.getField(field,newObject))){
                            saveDataChangeLog(entityClazz.getName(),oldObject, newObject, methodName, field,null, fieldName);
                            break;
                        }
                    }
                }
            }else{
                saveDataChangeLog(entityClazz.getName(),oldValue,newValue,methodName,field,dataChangeDto.getTableName(),fieldName);
            }
        }
    }

    /**
     * 获取数据库记录
     * @param value
     * @param annotation
     * @return
     */
    private Object getDBValue(Object value,DataChangeAnnotation annotation){
        ParamType paramType = annotation.getParamType();
        Class daoClazz = annotation.getDaoClazz();
        Class paramClazz = annotation.getParamClazz();
        String searchMethodName = annotation.getSearchMethodName();
        String fieldName = annotation.getFieldName();
        Class entityClazz = annotation.getEntityClazz();

        Object entityDao = SpringUtils.getBean(daoClazz);

        //获取查询数据的方法
        Method method = ReflectionUtils.findMethod(daoClazz, searchMethodName,paramClazz);
        Object result = null;
        switch (paramType) {
            case FIELD:
                Field field = ReflectionUtils.findField(entityClazz,fieldName);
                field.setAccessible(true);
                result = ReflectionUtils.invokeMethod(method, entityDao, ReflectionUtils.getField(field,value));
                break;
            default:
                if (method == null){
                    logger.error("method为空");
                    break ;

                }
                if (entityDao == null){
                    logger.error("entityDao为空");
                    break ;

                }
                if (value == null){
                    logger.error("value为空");
                    break ;
                }

                result = ReflectionUtils.invokeMethod(method, entityDao, value);
                break;
        }
        return result;
    }

    private void saveDataChangeLog(String entityClazz,Object oldValue,Object newValue,String methodName,Field field,String tableName,String fieldName) {
        List<DataChangeDto> changeList = new ArrayList<>();
        geChangeFieldValue(changeList, oldValue, newValue, oldValue.getClass());
        if (changeList.size() > 0) {
            dataChangeService.save(new DataChange(methodName, entityClazz, tableName,fieldName,
                    ReflectionUtils.getField(field, oldValue).toString(),
                    JSON.toJSONString(changeList).toString()));
        }
    }

    private void geChangeFieldValue(List<DataChangeDto> changeList,Object oldValue, Object newValue, Class clazz){
        Field[] fields = clazz.getDeclaredFields();
        DataChangeDto changeDto = null;
        for (Field field : fields) {
            if ("handler".equals(field.getName())) {
                continue;
            }
            if (!checkType(field)) {
                continue;
            }
            field.setAccessible(true);
            Object oldFieldValue = ReflectionUtils.getField(field, oldValue);
            Object newFieldValue = ReflectionUtils.getField(field, newValue);
            if (!ObjectUtil.equal(oldFieldValue, newFieldValue)) {
                changeDto = new DataChangeDto();
                changeDto.setFieldName(field.getName());
                if (Date.class.isInstance(oldFieldValue)) {
                    try {
                        changeDto.setOldValue(DateUtil.convertDateToString((Date) oldFieldValue, DateUtil.FORMAT1));
                    }catch (Exception e){
                        logger.error(e.getMessage(),e);
                    }
                    try {
                        changeDto.setNewValue(DateUtil.convertDateToString((Date) newFieldValue, DateUtil.FORMAT1));
                    }catch (Exception e){
                        logger.error(e.getMessage(),e);
                    }
                } else {
                    changeDto.setOldValue(oldFieldValue);
                    changeDto.setNewValue(newFieldValue);
                }
                changeList.add(changeDto);
            }
        }
        if (clazz.getSuperclass() != null) {
            geChangeFieldValue(changeList, oldValue, newValue, clazz.getSuperclass());
        }
    }

    private Boolean checkType(Field field){
        if(field.getType().isPrimitive()){
            return true;
        }

        if(field.getType().equals(Long.class)
            ||field.getType().equals(String.class)
            ||field.getType().equals(Integer.class)
            ||field.getType().equals(Date.class)
            ||field.getType().equals(LocalDate.class)
            ||field.getType().equals(LocalDateTime.class)
            ||field.getType().equals(Boolean.class)
            ){
            return true;
        }

        return false;
    }

}
