package com.bcxin.rest.web.apis.filters;

import com.bcxin.Infrastructures.TenantContext;
import com.bcxin.Infrastructures.TenantUserContext;
import com.bcxin.Infrastructures.components.CacheProvider;
import com.bcxin.Infrastructures.components.CacheV2Provider;
import com.bcxin.Infrastructures.components.JsonProvider;
import com.bcxin.Infrastructures.enums.PersonResourceType;
import com.bcxin.Infrastructures.enums.ResourceReferenceType;
import com.bcxin.Infrastructures.exceptions.BadTenantException;
import com.bcxin.Infrastructures.exceptions.TenantExceptionAbstract;
import com.bcxin.Infrastructures.exceptions.UnAuthorizedTenantException;
import com.bcxin.Infrastructures.utils.JwtUtil;
import com.bcxin.api.interfaces.tenants.UserRpcProvider;
import com.bcxin.api.interfaces.tenants.responses.UserOrganBasicGetResponse;
import com.bcxin.rest.web.apis.ResponseBuilder;
import com.bcxin.rest.web.apis.caches.UserOrganBasic;
import com.bcxin.rest.web.apis.utils.ServletRequestUtil;
import org.springframework.boot.autoconfigure.http.HttpMessageConverters;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;
import org.springframework.web.filter.OncePerRequestFilter;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;

@Component
@Order(Ordered.HIGHEST_PRECEDENCE-1)
public class AuthFilter extends OncePerRequestFilter {
    private final static Collection<String> excludedUrls = new ArrayList<>();
    static {
        excludedUrls.add("/identity/shanxi/out/sso/login");
        excludedUrls.add("/identity/beijing/in/sso/login");
        excludedUrls.add("/identity/hunan/out/sso/login");
        excludedUrls.add("/identity/hunan/sso/login");
        excludedUrls.add("/identity/rt/sso/login");
        excludedUrls.add("/identity/shanxi/pki/login");
        excludedUrls.add("/identity/gansu/pki/login");
        excludedUrls.add("/identity/jitGWAuth");
        excludedUrls.add("/identity/sign-in");
        excludedUrls.add("/wechat/cgi-bin/auto-login");
        excludedUrls.add("/wechat/cgi-bin/confirm-check-status");
        excludedUrls.add("/identity/sso");
//        excludedUrls.add("/identity/sso/login/naturalPerson");
//        excludedUrls.add("/identity/sso/login/legalPerson");
//        excludedUrls.add("/identity/sso/redirectUrl/naturalPerson");
//        excludedUrls.add("/identity/sso/redirectUrl/legalPerson");
        excludedUrls.add("/identity/ca/login");
        excludedUrls.add("/identity/beijing/xlcp/callback");
        excludedUrls.add("/identity/beijing/xlcp/zc");
        excludedUrls.add("/tenant/organizations/pre_registration/");
        excludedUrls.add("/tenant/organizations/name/exist");
        excludedUrls.add("/tenant/organizations/registration");
        excludedUrls.add("/tenant/organizations/update/");
        excludedUrls.add("/swagger-ui/");
        excludedUrls.add("/doc.html");
        excludedUrls.add("/webjars/");
        excludedUrls.add("/sso/test/");
        excludedUrls.add("/v2/api-docs");
        excludedUrls.add("/common/resources");
        excludedUrls.add("/common/dict");
        excludedUrls.add("/identity/wechat-in");
        excludedUrls.add("/identity/prepare-reset-password");
        excludedUrls.add("/identity/reset-password");
        excludedUrls.add("/identity/wechat-in");
        excludedUrls.add("/tenant/inviteDepart");
        excludedUrls.add("/tenant/inviteAttendSite");
        excludedUrls.add("/public/qualification-credentials");
        excludedUrls.add("/public/company-info");
        excludedUrls.add("/tenant/register");
    }

    /**
     * todo 统一修复序列化方式
     */
    private final JsonProvider jsonProvider;
    private final HttpMessageConverters converters;
    private final CacheProvider cacheProvider;
    private final UserRpcProvider userRpcProvider;
    private final CacheV2Provider cacheV2Provider;

    public AuthFilter(JsonProvider jsonProvider, HttpMessageConverters converters,
                      CacheProvider cacheProvider, UserRpcProvider userRpcProvider,
                      CacheV2Provider cacheV2Provider) {
        this.jsonProvider = jsonProvider;
        this.converters = converters;
        this.cacheProvider = cacheProvider;
        this.userRpcProvider = userRpcProvider;
        this.cacheV2Provider = cacheV2Provider;
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        try {
            TenantUserContext userContext = TenantContext.getInstance().getUserContext();
            if (userContext.get() == null || !StringUtils.hasLength(userContext.get().getToken())) {
                throw new UnAuthorizedTenantException("当前无有效Authorization信息");
            }

            /**
             * 还是需要查询用户信息
             */
            this.doCheckSelectedOrganization(request);

            filterChain.doFilter(request, response);
        } catch (Exception ex) {
            HttpStatus status = HttpStatus.BAD_REQUEST;
            if (ex instanceof TenantExceptionAbstract) {
                status = HttpStatus.UNAUTHORIZED;
            }

            ResponseEntity responseEntity =
                    ResponseBuilder.build(status, ex.getMessage());
            response.setStatus(responseEntity.getStatusCode().value());

            String error = this.jsonProvider.getJson(responseEntity.getBody());
            response.setHeader(HttpHeaders.CONTENT_TYPE, "application/json");

            response.getOutputStream().write(error.getBytes(StandardCharsets.UTF_8));
            response.flushBuffer();
        }
    }

    @Override
    protected boolean shouldNotFilter(HttpServletRequest request) throws ServletException {
        /**
         * 重置当前用户上下文信息
         */
        TenantContext.getInstance().getUserContext().reset();

        try {
            String token = ServletRequestUtil.getBearerToken(request);
            String tenantUserId = JwtUtil.getContentFromToken(token);
            /**
             * 初始化当前token和对应的tenant user id
             */
            TenantContext.getInstance().getUserContext().init(token, tenantUserId);
        } catch (Exception ex) {
            //todo 获取token或者解析token失败
        }

        String requestUri = request.getRequestURI();
        if (!StringUtils.hasLength(requestUri)) {
            return true;
        }

        boolean matchedResult =
                excludedUrls.stream().anyMatch(ii -> requestUri.startsWith(String.format("%s%s", request.getContextPath(), ii)));

        return matchedResult || requestUri.contains(String.format("%s/swagger", request.getContextPath())) || requestUri.contains("/download/resources");
    }

    /**
     * 验证employee token是否跟当前登入的为同一个用户
     * @param request
     */
    private void validateIsAllowedAction(HttpServletRequest request) {
        String employee_Token = ServletRequestUtil.getEmployeeToken(request);
        if (!StringUtils.hasLength(employee_Token)) {
            return;
        }

        /**
         *             Map<String, Object> map = new HashMap<>();
         *             map.put("employeeId", ii.getEmployeeId());
         *             map.put("tenantUserId", ii.getTenantUserId());
         *             map.put("organizationId", ii.getId());
         *             map.put("createdTime", Timestamp.from(Instant.now()));
         *             String json = jsonProvider.getJson(map);
         */
        String employeeJson=null;
        try {
            employeeJson = JwtUtil.getContentFromToken(employee_Token);
            Map<String, Object> employeeMap = jsonProvider.toObject(Map.class, employeeJson);
            if (employeeMap == null || employeeMap.size() == 0) {
                throw new BadTenantException("无效的employee_token");
            }
            boolean matched = TenantContext.getInstance().getUserContext().get().getId().equals(employeeMap.get("tenantUserId"));
            if (!matched) {
                throw new BadTenantException("当前登入用户与employee_token不匹配");
            }
        }
        catch (Exception ex)
        {
            throw new BadTenantException(String.format("employee_token(%s)无效",employeeJson));
        }
    }

    private void doCheckSelectedOrganization(HttpServletRequest request) {
        String requestUri = request.getRequestURI();
        if (StringUtils.hasLength(requestUri) && requestUri.contains("organizations/") && !requestUri.contains("/approved") && !requestUri.contains("/organizations/search")) {
            String[] pathSections = requestUri.split("/");
            String selectedOrganId = null;
            String selectedEmployeeId=null;
            for (int index = 0; index < pathSections.length; index++) {
                if ("organizations".equals(pathSections[index]) && (index + 1) < pathSections.length) {
                    selectedOrganId = pathSections[index + 1];
                    break;
                }
            }
            for (int index = 0; index < pathSections.length; index++) {
                if ("employees".equals(pathSections[index]) && (index + 1) < pathSections.length) {
                    selectedEmployeeId = pathSections[index + 1];
                    break;
                }

            }
            if(requestUri.contains("/organizations/")
                    && requestUri.contains("/employees/")
                    && !requestUri.contains("exportEmployeeTemp")
                    && StringUtils.hasLength(selectedEmployeeId)
                    && "get".equalsIgnoreCase(request.getMethod())
            ) {
                return;
            }
            if(requestUri.contains("/organizations/") &&
                    requestUri.contains("/credentials") &&
                    "post".equalsIgnoreCase(request.getMethod())
            ){
                return;
            }

            if (StringUtils.hasLength(selectedOrganId)) {
                String finalSelectedOrganId = selectedOrganId;
                String tenantUserId = TenantContext.getInstance().getUserContext().get().getId();

                PersonResourceType resourceType = PersonResourceType.Employee;
                if(requestUri.contains("external-members")) {
                    resourceType = PersonResourceType.Member;
                }
                //todo 修复
                String myOrgCache = String.format("myOrganInfo:v2:%s-%s", finalSelectedOrganId, tenantUserId);
                PersonResourceType finalResourceType = resourceType;
                UserOrganBasicGetResponse organBasicGetResponse =
                        cacheV2Provider.get(myOrgCache, () -> {
                            UserOrganBasicGetResponse getResponse = this.userRpcProvider.getByIdAndOrganId(
                                    finalSelectedOrganId,
                                    tenantUserId, finalResourceType);

                            /**
                             * 针对只查询 企业信息的情况 /v3/tenant/organizations/aSuXCEfz
                             */
                            if(finalResourceType==PersonResourceType.Employee && getResponse==null) {
                                getResponse = this.userRpcProvider.getByIdAndOrganId(
                                        finalSelectedOrganId,
                                        tenantUserId, PersonResourceType.Member);
                            }

                            return getResponse;
                        });

                if (organBasicGetResponse == null) {
                    throw new UnAuthorizedTenantException(String.format("当前用户(%s)不属于本公司(%s)或者已经办理离职手续!", tenantUserId, finalSelectedOrganId));
                }


                UserOrganBasic basic = UserOrganBasic.create(
                        organBasicGetResponse.getOrganizationId(),
                        organBasicGetResponse.getOrgName(),
                        organBasicGetResponse.getEmployeeId(),
                        organBasicGetResponse.getName(),
                        organBasicGetResponse.getMasterSlaveType(),
                        organBasicGetResponse.getAdditional(),
                        organBasicGetResponse.isDomainAdmin(),
                        organBasicGetResponse.isDepartAdmin(),
                        organBasicGetResponse.getAreaCode());

                TenantContext.getInstance().getUserContext()
                        .assignDetail(basic.getName(), basic.getOrganizationId(),
                                basic.getOrgName(),
                                basic.getEmployeeId(), basic.getMasterSlaveType(),
                                basic.isDomainAdmin(),basic.isDepartAdmin(),
                                basic.getAreaCode());
            }
        }
    }
}
