package com.bcxin.tenant.open.infrastructures.utils;

import com.bcxin.tenant.open.infrastructures.constants.BusinessConstants;
import com.bcxin.tenant.open.infrastructures.enums.DeskType;
import com.bcxin.tenant.open.infrastructures.enums.DispatchAccountType;
import com.bcxin.tenant.open.infrastructures.exceptions.BadTenantException;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class BusinessUtil {
    /**
     * 判断是否为有效的引用Id; 比如: 驻勤点,人员等
     *
     * @param value
     * @return
     */
    public static boolean validReferenceNumber(String value) {
        if (StringUtil.isEmpty(value)) {
            return false;
        }

        return value.length() > 2;
    }

    public static boolean isSupervise(String institutional) {
        return isSecuritySupervise(institutional) || isProprietorSupervise(institutional) || isPopulationSupervise(institutional) || isOtherSupervise(institutional);
    }

    public static boolean isSecuritySupervise(String institutional) {
        if (StringUtil.isEmpty(institutional)) {
            return false;
        }

        /**
         * 01保安
         */
        final Collection<String> superviseInstitutionals =
                Stream.of("0107", "0108", "0109","0110")
                        .collect(Collectors.toList());

        return superviseInstitutionals.stream()
                .anyMatch(ii -> ii.equalsIgnoreCase(institutional));
    }

    public static boolean isProprietorSupervise(String institutional) {
        if (StringUtil.isEmpty(institutional)) {
            return false;
        }

        /**
         * 04内保单位
         */
        final Collection<String> superviseInstitutionals =
                Stream.of("0407", "0408", "0409", "0410")
                        .collect(Collectors.toList());

        return superviseInstitutionals.stream()
                .anyMatch(ii -> ii.equalsIgnoreCase(institutional));
    }

    public static boolean isPopulationSupervise(String institutional) {
        if (StringUtil.isEmpty(institutional)) {
            return false;
        }

        /**
         * 06 群防群治
         */
        final Collection<String> superviseInstitutionals =
                Stream.of("0607", "0608", "0609", "0610")
                        .collect(Collectors.toList());

        return superviseInstitutionals.stream()
                .anyMatch(ii -> ii.equalsIgnoreCase(institutional));
    }

    public static boolean isOtherSupervise(String institutional) {
        if (StringUtil.isEmpty(institutional)) {
            return false;
        }

        institutional = institutional.trim();
        if (institutional.length() != 4) {
            return false;
        }

        return institutional.endsWith("07") || institutional.endsWith("08") || institutional.endsWith("09") || institutional.endsWith("10");
    }

    public static boolean isExpectedOrganization(String expectedOrgId, String orgId) {
        /**
         * 这边判断为空是为了兼容旧数据, 再次版本之前, 这个对象是没有存储organizationId, 因此, 都是本公司的数据
         */
        return StringUtil.isEmpty(orgId) || StringUtil.isEqual(expectedOrgId, orgId);
    }

    public static String[] getDefaultInstitutionals(Collection<DeskType> deskTypes) {
        if (deskTypes == null || deskTypes.isEmpty()) {
            throw new BadTenantException("调度台类型不能为空");
        }

        Set<String> institutionals = new HashSet<>();
        for (DeskType deskType : deskTypes) {
            if (deskType == DeskType.Proprietor) {
                institutionals.addAll(getProprietorInstitutional());
            } else {
                institutionals.add("0102");
                institutionals.add("0103");
                institutionals.add("0104");
            }
        }

        return institutionals.toArray(ix -> new String[ix]);
    }

    public static String getInstitutionalName(String institutional) {
        if (StringUtil.isEmpty(institutional)) {
            return "";
        }

        switch (institutional) {
            case "0102" -> {
                return "保安服务公司";
            }
            case "0103" -> {
                return "保安培训单位";
            }
            case "0104" -> {
                return "自行招用保安员单位";
            }
            case "0403" -> {
                return "中小学、幼儿园";
            }
            case "0404" -> {
                return "高校";
            }
            case "0401" -> {
                return "医疗单位";
            }
            case "0405" -> {
                return "金融单位";
            }
            case "0402" -> {
                return "文博单位";
            }
            case "0406" -> {
                return "党政机关";
            }
            case "0201" -> {
                return "活动承办商";
            }
            case "0212" -> {
                return "组委会";
            }
            case "0214" -> {
                return "体育局";
            }
            case "0215" -> {
                return "文旅局";
            }
            case "0216" -> {
                return "卫健委";
            }
            case "0217" -> {
                return "政法委";
            }
            case "0218" -> {
                return "共青团";
            }
            case "0219" -> {
                return "社团志愿者团队";
            }
            case "0221" -> {
                return "无人机反制公司";
            }
        }

        return institutional;
    }

    /**
     * 是否动态计算在线保安人员以及驻勤数量
     * @param institutional
     * @return
     */
    public static boolean checkIfDynamicCountOfStationsAndMen(String institutional) {
        return getProprietorInstitutional().contains(institutional);
    }

    private static Collection<String> getProprietorInstitutional() {
        Set<String> institutional = new HashSet<>();
        institutional.add("0403");
        institutional.add("0404");
        institutional.add("0401");
        institutional.add("0405");
        institutional.add("0402");
        institutional.add("0406");

        return institutional;
    }

    /**
     * 内保单位
     * @param institutional
     * @return
     */
    public static boolean isProprietor(String institutional) {
        if (StringUtil.isEmpty(institutional)) {
            return false;
        }

        return getProprietorInstitutional().stream()
                .anyMatch(ii -> ii.equalsIgnoreCase(institutional));
    }

    public static boolean checkIfSecurityStationRelativeUsers(String institutional) {
        if (StringUtil.isEmpty(institutional)) {
            return false;
        }
        institutional = institutional.trim();
        if (institutional.startsWith("01") || checkIfCommunityPolicing(institutional)) {
            return true;
        }

        return false;
    }

    public static boolean checkIfCommunityPolicing(String institutional) {
        if (StringUtil.isEmpty(institutional)) {
            return false;
        }

        institutional = institutional.trim();

        return "0601".equalsIgnoreCase(institutional);
    }
    
    public static String getRollCallGeTuiTitle(String orgName) {
        return orgName;
    }

    public static String getRollCallGeTuiContent(String callerName) {
        return String.format("%s正向您发起点名轮询, 请接听..", callerName);
    }

    public static boolean isStationResponsible(String responsibleOfStationId) {
        if (StringUtil.isEmpty(responsibleOfStationId) || BusinessConstants.DefaultEmptyValue.equalsIgnoreCase(responsibleOfStationId)) {
            return false;
        }

        return true;
    }

        /**
     * 判断是否为调度台
     * @param accountType
     * @return
     */
    public static boolean checkIfDesk(DispatchAccountType accountType) {
        return accountType == DispatchAccountType.AppDesk || accountType == DispatchAccountType.Desk;
    }

    public static boolean isEventOrganizer(String institutional) {
        /**
         * 0102 保安公司
         */
        return isDaHuoManagedOrg(institutional) || "0102".equalsIgnoreCase(institutional);
    }

    private static final Collection<String> securityAssiciationInstitutionals =Stream.of("0106","0105","0101").collect(Collectors.toList());
    public static boolean isSecurityAssociation(String institutional) {
        return securityAssiciationInstitutionals.stream().anyMatch(ii -> ii.equalsIgnoreCase(institutional));
    }

    public static boolean isEventSponsor(String institutional) {
        return "0201".equalsIgnoreCase(institutional);
    }

    public static boolean isOrgMember(String id)
    {
        return !StringUtil.isEmpty(id) && id.startsWith("1$");
    }

    public static String extractMemberId(String id) {
        return StringUtil.isEmpty(id) ? "" : id.substring(2);
    }

    public static boolean isDaHuoManagedOrg(String institutional) {
        return daHuoManagedOrgInstitutionals().contains(institutional);
    }

    public static Set<String> daHuoManagedOrgInstitutionals() {
        return Stream.of("0201","0203","0211",  "0212", "0214", "0215", "0216", "0217", "0218", "0219", "0221")
                .collect(Collectors.toSet());
    }

    public static boolean isEventProject(String id)
    {
        return !StringUtil.isEmpty(id) && id.endsWith(BusinessConstants.EventProjectIdStuffId);
    }

    public static Collection<String> getSplitValues(String input) {
        if (StringUtil.isEmpty(input)) {
            return Collections.EMPTY_LIST;
        }

        return Arrays.stream(input.split(";|,")).collect(Collectors.toSet());
    }

    public static String getPaymentOrgId(String orgId,String paymentOrgId) {
        if (!StringUtil.isEmpty(paymentOrgId) && !BusinessConstants.DefaultEmptyValue.equalsIgnoreCase(paymentOrgId)) {
            return paymentOrgId;
        }

        return orgId;
    }
}
