package com.bcxin.risk.sys;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.map.MapUtil;
import com.bcxin.risk.common.dao.MarketRegionDao;
import com.bcxin.risk.common.domain.MarketRegion;
import com.bcxin.risk.common.domain.Region;
import com.bcxin.risk.constant.DictConst;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 可销售城市地区类
 * @author 罗鹏
 */
@Service
@Transactional
@Slf4j
public class SysMarketRegionUtil {

    @Resource
    private SysRegionUtil sysRegionUtil;

    @Resource
    private MarketRegionDao marketRegionDao ;

    private static List<MarketRegion> CACHE_KEY_MARKET_REGION_PROVINCE = Lists.newArrayList();

    private static List<MarketRegion> CACHE_KEY_MARKET_REGION_CITY = Lists.newArrayList();

    private static List<MarketRegion> CACHE_KEY_MARKET_REGION_INDEPENDENT = Lists.newArrayList();

    private static Map<String,MarketRegion> CACHE_KEY_BLAZER_MARKET_REGION = Maps.newHashMap();

    /**
     * 查询可销售区域
     */
    public List<MarketRegion> findRegionByLevel(String level){
        if (DictConst.REGIONLEVEL_PROVINCE.equals(level)) {
            if (CollectionUtil.isNotEmpty(CACHE_KEY_MARKET_REGION_PROVINCE)) {
                return CACHE_KEY_MARKET_REGION_PROVINCE;
            }
        } else if (DictConst.REGIONLEVEL_CITY.equals(level)){
            if (CollectionUtil.isNotEmpty(CACHE_KEY_MARKET_REGION_CITY)) {
                return CACHE_KEY_MARKET_REGION_CITY;
            }
        }
        List<MarketRegion> rList = marketRegionDao.findRegionByLevel(level);
        if (DictConst.REGIONLEVEL_PROVINCE.equals(level)) {
            CACHE_KEY_MARKET_REGION_PROVINCE = rList;
        } else if (DictConst.REGIONLEVEL_CITY.equals(level)){
            CACHE_KEY_MARKET_REGION_CITY = rList;
        }
        return rList;
    }

    /**
     * 查询办事区域
     */
    public List<MarketRegion> findRegionByIndependent(){
        if (CollectionUtil.isNotEmpty(CACHE_KEY_MARKET_REGION_INDEPENDENT)) {
            return CACHE_KEY_MARKET_REGION_INDEPENDENT;
        }
        Collection<MarketRegion> rList = findBlazerMarketList();
        CACHE_KEY_MARKET_REGION_INDEPENDENT = Lists.newArrayList(rList);
        return CACHE_KEY_MARKET_REGION_INDEPENDENT;
    }

    /**
     * 获得已经开拓的市场
     * @return 返回map
     */
    private Map<String,MarketRegion> findBlazerMarketRedis(){
        if (MapUtil.isNotEmpty(CACHE_KEY_BLAZER_MARKET_REGION)) {
            return CACHE_KEY_BLAZER_MARKET_REGION;
        }
        Collection<MarketRegion> rList = findBlazerMarketList();
        for( MarketRegion marketRegion:rList){
            CACHE_KEY_BLAZER_MARKET_REGION.put(marketRegion.getOid()+"",marketRegion);
        }
        return CACHE_KEY_BLAZER_MARKET_REGION;
    }

    /**
     * 获得已经开拓的市场
     * @return 返回集合
     */
    public Collection<MarketRegion> findBlazerMarketList(){
        return marketRegionDao.findBlazerMarket().values();
    }

    /**
     * 获得已经开拓的市场
     * @return 根据区域id返回单个
     */
    public MarketRegion getBlazerMarket(long regionCode){
        Map<String,MarketRegion> marketRegionMap = findBlazerMarketRedis();
        MarketRegion obj = marketRegionMap.get(regionCode + "");
        if (obj == null) {
            return new MarketRegion();
        }
        return obj;
    }

    public boolean validBlazerMarket(long regionCode){
        Map<String,MarketRegion> marketRegionMap = findBlazerMarketRedis();
        MarketRegion obj = marketRegionMap.get(regionCode + "");
        if (obj == null) {
            return false;
        }
        return true;
    }


    /**
     * 获取已开拓市场
     * @param cityCode
     * @return
     */
    public MarketRegion getBlazerMarketByCity(long cityCode){
        Region region=sysRegionUtil.getRegionById(cityCode+"");
        Map<String,MarketRegion> marketRegionMap = findBlazerMarketRedis();
        // 获取市
        MarketRegion obj = marketRegionMap.get(cityCode + "");

        if (obj == null) {
            // 特殊市区
            if (isSpecialProvince(region.getParent_id(),cityCode+"")){
                return obj;
            }
            // 获取省
            return marketRegionMap.get(region.getParent_id() + "");
        }
        return obj;
    }

    /**
     * 根据市获取落地省份
     * @param cityCode
     * @return
     */
    public boolean validBlazerMarketByCity(long cityCode){
        MarketRegion marketRegion=getBlazerMarketByCity(cityCode);
        if(marketRegion==null){
            return false;
        }
        return true;
    }

    /**
     * 获取channel
     * @param cityRegionId
     * @return
     */
    public String getChannel(String cityRegionId){
        MarketRegion marketRegion = getBlazerMarketByCity(Long.parseLong(cityRegionId));
        if(marketRegion!=null){
            return marketRegion.getWebsite();
        }
        return null;
    }

    /**
     * 是否特殊市区
     * 如：浙江已经全省落地
     * @param provinceId
     * @return
     */
    public boolean isSpecialProvince(String provinceId,String cityId){
        String [] specialCityList=new String[]{""};
        for(String l : specialCityList){
            if(l.equals(provinceId)){
                return true;
            }
        }
        return false;
    }
}
