package com.bcxin.runtime.apis.controllers;

import cn.myapps.common.Environment;
import cn.myapps.common.util.DefaultProperty;
import cn.myapps.common.util.PropertiesConfig;
import cn.myapps.util.CollectionUtil;
import cn.myapps.util.file.RemoteFileCompressor;
import cn.myapps.util.file.ZipUtil;
import com.bcxin.runtime.apis.configs.RegionConfig;
import com.bcxin.runtime.apis.requests.CompressFileToZipRequest;
import com.bcxin.runtime.apis.requests.CompressFileToZipV2Request;
import com.bcxin.saas.core.exceptions.SaasBadException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.net.ssl.HttpsURLConnection;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.*;
import java.net.URLEncoder;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import static com.bcxin.runtime.domain.constants.FieldNames.extractPathValues;

@RestController
@RequestMapping("/v3/extends/compress")
public class CompressController extends SecurityControllerAbstract {
    private static final Logger logger = LoggerFactory.getLogger(CompressController.class);

    private final RegionConfig regionConfig;
    private final PropertiesConfig propertiesConfig;

    public CompressController(RegionConfig regionConfig, PropertiesConfig propertiesConfig) {
        this.regionConfig = regionConfig;
        this.propertiesConfig = propertiesConfig;
        HttpsURLConnection.setDefaultHostnameVerifier((hostname, session) -> true);
    }

    @PostMapping("/fileIntoZip")
    public void getFileIntoZIP(@RequestBody CompressFileToZipRequest compressFileToZipRequest, HttpServletResponse response){
        try {
            Collection<String> allExtractPaths =new ArrayList<>();
            for (String path:compressFileToZipRequest.getFilePaths()){
                if (path != null) {
                    if (!path.startsWith("http")) {
                        path = this.regionConfig.getSourceSite()+String.format("/obpm/%s", path).replace("//", "/");
                    }
                    allExtractPaths.add(path);
                }
            }
            SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmssSSS");
            String timestamp = sdf.format(new Date());
            String zipPath=propertiesConfig.getStorageroot()+compressFileToZipRequest.getDestDir()+timestamp+"/"+compressFileToZipRequest.getNewFileName()+".zip";
            RemoteFileCompressor.compressRemoteFiles(allExtractPaths.toArray(new String[0]),zipPath);
            response.setCharacterEncoding("UTF-8");
            response.getWriter().write(zipPath);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @PostMapping("/file-zip")
    public void postFileIntoZIP(@RequestBody CompressFileToZipV2Request compressFileToZipV2Request, HttpServletResponse response) {

        try {
            if(compressFileToZipV2Request.getFolders()==null) {
                compressFileToZipV2Request.setFolders(new ArrayList<>());
            }

            SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmssSSS");
            String timestamp = sdf.format(new Date());
            String zipPath = propertiesConfig.getStorageroot() + compressFileToZipV2Request.getDestDir() + timestamp + "/" + compressFileToZipV2Request.getNewFileName() + ".zip";
            compressRemoteFiles(regionConfig,compressFileToZipV2Request.getFolders(), zipPath);
            response.setCharacterEncoding("UTF-8");
            response.getWriter().write(zipPath);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public static String compressRemoteFiles(RegionConfig regionConfig,Collection<CompressFileToZipV2Request.CompressFileItemFolder> folders, String targetZipPath) {

        String temp = DefaultProperty.getProperty("myapps.storage.root") +
                File.separator + "uploads" + File.separator + "temp"+ File.separator+UUID.randomUUID();
        File tempDir = new File(temp,UUID.randomUUID().toString());
        Path zipFilePath = Paths.get(targetZipPath);
        if (!tempDir.exists()) {
            tempDir.mkdirs();
        }

        try {
            Files.createDirectories(zipFilePath.getParent());
        } catch (IOException e) {
            throw new SaasBadException("创建ZIP文件的父目录失败：", e.getMessage());
        }

        try (FileOutputStream fos = new FileOutputStream(targetZipPath);
             ZipOutputStream zos = new ZipOutputStream(new BufferedOutputStream(fos))) {
            File notFoundFile = new File(tempDir, "未找到文件路径.txt");
            for (CompressFileToZipV2Request.CompressFileItemFolder folder : folders) {
                Collection<String> remoteUrls = folder.getFilePaths();
                if(CollectionUtils.isEmpty(remoteUrls)) {
                    continue;
                }

                // 下载远程文件到临时目录
                File targetTmpDir = new File(temp, folder.getFolderName());
                if (!targetTmpDir.exists()) {
                    targetTmpDir.mkdirs();
                }

                for (String remoteUrl : remoteUrls) {
                    try {
                        if (remoteUrl != null) {
                            if (!remoteUrl.startsWith("http")) {
                                remoteUrl = regionConfig.getSourceSite() + String.format("/obpm/%s", remoteUrl).replace("//", "/");
                            }
                        }
                        File downloadedFile = RemoteFileCompressor.downloadFile(remoteUrl, targetTmpDir);

                        if (downloadedFile != null) {
                            logger.error("当前要加入的文件为:{} / {}",folder.getFolderName(),downloadedFile.getName());
                            // 创建ZipEntry，并添加到ZipOutputStream
                            ZipEntry zipEntry = new ZipEntry(String.format("%s/%s", folder.getFolderName(), downloadedFile.getName()));
                            zos.putNextEntry(zipEntry);

                            try (FileInputStream fis = new FileInputStream(downloadedFile)) {
                                byte[] buffer = new byte[1024];
                                int length;
                                while ((length = fis.read(buffer)) >= 0) {
                                    zos.write(buffer, 0, length);
                                }
                            }
                            zos.closeEntry();
                        }
                    }
                    catch (Exception ex) {
                        logger.error("文件下载失败:{}", remoteUrl, ex);
                    }
                }
            }

            if (notFoundFile.exists()) {
                ZipEntry zipEntry = new ZipEntry(notFoundFile.getName());
                zos.putNextEntry(zipEntry);

                try (FileInputStream fis = new FileInputStream(notFoundFile)) {
                    byte[] buffer = new byte[1024];
                    int length;
                    while ((length = fis.read(buffer)) >= 0) {
                        zos.write(buffer, 0, length);
                    }
                }
                zos.closeEntry();
            }

            return targetZipPath;

        } catch (IOException e) {
            throw new SaasBadException("压缩远程文件失败：", e.getMessage());
        } finally {
            // 清理临时文件（可选，根据实际需求）
            RemoteFileCompressor.cleanUpTempFiles(tempDir);
        }
    }
}
