package com.bcxin.runtime.apis.controllers;
import cn.myapps.authtime.common.service.AuthTimeServiceManager;
import cn.myapps.authtime.domain.model.DomainVO;
import cn.myapps.authtime.domain.service.DomainProcess;
import cn.myapps.authtime.user.model.UserVO;
import cn.myapps.authtime.user.service.UserProcess;
import cn.myapps.common.FileSystemDesignTimeSerializable;
import cn.myapps.common.data.DataPackage;
import cn.myapps.common.model.application.Application;
import cn.myapps.common.model.role.Role;
import cn.myapps.common.util.SpringApplicationContextUtil;
import cn.myapps.designtime.application.service.ApplicationDesignTimeService;
import cn.myapps.designtime.common.cache.DesignTimeSerializableCache;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.role.service.RoleDesignTimeService;
import cn.myapps.runtime.macro.RhinoJavaScriptCompiler;
import com.bcxin.runtime.apis.requests.BatchAssignUserRoleRequest;
import com.bcxin.runtime.apis.requests.BatchClearUserRoleRequest;
import com.bcxin.runtime.apis.requests.SearchUserRoleRequest;
import com.bcxin.runtime.apis.responses.AppRoleResponse;
import com.bcxin.runtime.apis.responses.UserRoleResponse;
import com.bcxin.saas.core.components.DistributedCacheProvider;
import com.bcxin.saas.core.exceptions.SaasBadException;
import com.bcxin.saas.core.exceptions.SaasNofoundException;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.sql.ResultSet;
import java.util.*;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/v3/extends")
public class AppController extends ControllerAbstract{
    private final NamedParameterJdbcTemplate jdbcTemplate;
    private final DistributedCacheProvider distributedCacheProvider;
    public AppController(NamedParameterJdbcTemplate jdbcTemplate, DistributedCacheProvider distributedCacheProvider) {
        this.jdbcTemplate = jdbcTemplate;
        this.distributedCacheProvider = distributedCacheProvider;
    }

    @GetMapping("/domains/{domainId}")
    public ResponseEntity getMyApps(@PathVariable String domainId) throws Exception {
        try {
            Collection<AppRoleResponse> appRoleResponses = this.getAppsByDomainId(domainId);

            return ResponseEntity.ok(appRoleResponses);
        } catch (Exception ex) {
            if (ex instanceof SaasNofoundException) {
                return ResponseEntity.status(HttpStatus.NOT_FOUND).body(ex.getMessage());
            }

            throw ex;
        }
    }

    @PutMapping("/domains/{domainId}/batch-assign-permissions")
    public ResponseEntity put(HttpServletRequest servletRequest, @PathVariable String domainId, @RequestBody BatchAssignUserRoleRequest request) throws Exception {
        this.validateAllowed(servletRequest,domainId);

        UserProcess userProcess = AuthTimeServiceManager.userRuntimeService();
        Collection<UserVO> userVOS = userProcess.getAllByIds(domainId, request.getUserIds());

        Collection<String> notExistsUserIds = request.getUserIds().stream().filter(ii ->
                        !userVOS.stream().anyMatch(ix -> ix.getId().equals(ii)))
                .collect(Collectors.toList());
        if (notExistsUserIds.size() > 0) {
            return ResponseEntity.status(HttpStatus.NOT_FOUND).body(String.format("找不到用户(%s)",
                    notExistsUserIds.stream().collect(Collectors.joining(","))));
        }

        Collection<AppRoleResponse.RoleItem> allowedAppRoles = this.getAppsByDomainId(domainId).stream()
                .filter(ii -> ii.getRoles() != null)
                .flatMap(ii -> ii.getRoles().stream()).collect(Collectors.toList());

        Collection<String> notExistsRoleIds = request.getRoleIds().stream().filter(ii -> !allowedAppRoles.stream().anyMatch(ix -> ix.getId().equals(ii)))
                .collect(Collectors.toList());
        if (notExistsRoleIds.size() > 0) {
            return ResponseEntity.status(HttpStatus.NOT_ACCEPTABLE).body(String.format("无法授权允许的应用以外的角色信息(%s)",
                    notExistsRoleIds.stream().collect(Collectors.joining(","))));
        }

        doUpdateUserRoles(domainId, userVOS, request.getRoleIds());

        return ResponseEntity.ok(null);
    }


    @PostMapping("/domains/{domainId}/batch-clear-permissions")
    public ResponseEntity batchDelete(HttpServletRequest servletRequest,
                              @PathVariable String domainId,
                                      @RequestBody BatchClearUserRoleRequest request) {

        this.validateAllowed(servletRequest,domainId);

        Map deleteMap = new HashMap();
        deleteMap.put("id", request.getUserIds());
        deleteMap.put("domainId", domainId);

        int affectedCount = jdbcTemplate.update(
                "delete from t_user_department_role_set where USERID in (select t.id from t_user t where t.id in (:id) and t.DOMAINID=:domainId)",
                deleteMap);

        return ResponseEntity.ok(affectedCount);
    }

    @PostMapping("/domains/{domainId}/search-permissions")
    public ResponseEntity batchClear(@PathVariable String domainId,
                                     @RequestBody SearchUserRoleRequest request) throws Exception {
        Collection<String> userIds = request.getUserIds();
        Map<String, Collection<String>> userRoleIdMaps = new HashMap<>();
        if (!CollectionUtils.isEmpty(userIds)) {
            Map<String, Object> parameters = new HashMap<>();
            parameters.put("id", userIds);
            parameters.put("domainId", domainId);

            this.jdbcTemplate.query(
                    "select USERID,ROLEID from t_user_department_role_set where USERID in (select t.id from t_user t where t.id in (:id) and t.DOMAINID=:domainId)",
                    parameters, (ResultSet rs) -> {
                        while (rs.next()) {
                            String uId = rs.getString("USERID");
                            String rId = rs.getString("ROLEID");

                            Collection<String> userRoleIds = null;
                            if (!userRoleIdMaps.containsKey(uId)) {
                                userRoleIds = new ArrayList<>();
                                userRoleIdMaps.put(uId, userRoleIds);
                            } else {
                                userRoleIds = userRoleIdMaps.get(uId);
                            }

                            userRoleIds.add(rId);
                        }
                    });
        }

        Collection<UserRoleResponse> userRoleResponses = new ArrayList<>();
        if (userRoleIdMaps.size() == 0) {
            return ResponseEntity.ok(userRoleResponses);
        }

        Collection<AppRoleResponse> appRoleResponses = this.getAppsByDomainId(domainId);
        for (String uId : userRoleIdMaps.keySet()) {
            Optional<UserRoleResponse> selectedUserRoleOptional = userRoleResponses.stream().filter(ii -> ii.getUserId().equals(uId))
                    .findFirst();
            UserRoleResponse selectedUserRole = null;
            if (!selectedUserRoleOptional.isPresent()) {
                selectedUserRole = UserRoleResponse.create(uId);
                userRoleResponses.add(selectedUserRole);
            } else {
                selectedUserRole = selectedUserRoleOptional.get();
            }

            Collection<String> roleIds = userRoleIdMaps.get(uId);
            for (AppRoleResponse app : appRoleResponses) {
                Collection<AppRoleResponse.RoleItem> selectedRoles = app.getRoles().stream().filter(ii -> roleIds.contains(ii.getId()))
                        .collect(Collectors.toList());

                if (selectedRoles.size() > 0) {
                    UserRoleResponse.UserAppItem appItem = UserRoleResponse.UserAppItem.create(app.getId(), app.getName());
                    selectedRoles.forEach(ix -> {
                        appItem.assignRole(ix.getId(), ix.getName());
                    });

                    selectedUserRole.assignApps(appItem);
                }
            }
        }

        return ResponseEntity.ok(userRoleResponses);
    }

    @GetMapping("/app-roles")
    public ResponseEntity getAllApps() throws Exception {
        ApplicationDesignTimeService appService = DesignTimeServiceManager.applicationDesignTimeService();
        List<Application> apps = appService.list(null, null);

        RoleDesignTimeService roleDesignTimeService = DesignTimeServiceManager.roleDesignTimeService();
        Collection<Role> roles = roleDesignTimeService.list(null, null);

        Collection<AppRoleResponse> appRoleResponses =
                apps.stream().map(ii -> {
                    AppRoleResponse arr = AppRoleResponse.create(ii.getId(), ii.getName());
                    Collection<AppRoleResponse.RoleItem> selectedRoles =
                            roles.stream().filter(ir -> ir.getApplicationid().equals(ii.getId()))
                                    .map(ir -> AppRoleResponse.RoleItem.create(ir.getId(), ir.getName()))
                                    .collect(Collectors.toList());

                    arr.assignRole(selectedRoles);
                    return arr;
                }).collect(Collectors.toList());

        return ResponseEntity.ok(appRoleResponses);
    }

    @GetMapping("/components/{id}")
    public ResponseEntity getComponent(@PathVariable String id) {
        FileSystemDesignTimeSerializable component = DesignTimeSerializableCache.get(id);


        HashMap map = new HashMap();
        if (component == null) {
            map.put("data", "找不到组件");
        } else {
            map.put("id", component.getId());
            map.put("url", component.getUri());
            map.put("appId", component.getApplicationid());
            map.put("data", component);
        }

        return ResponseEntity.ok(map);
    }

    @GetMapping("/components-redis/{id}")
    public ResponseEntity getComponentFromRedis(@PathVariable String id) {

        FileSystemDesignTimeSerializable component = this.distributedCacheProvider.get(DesignTimeSerializableCache.getSeriallizableRedisKey(id, false));


        HashMap map = new HashMap();
        if (component == null) {
            map.put("data", "找不到组件");
        } else {
            map.put("data", component);
        }

        return ResponseEntity.ok(map);
    }

    @GetMapping("/components/{id}/children")
    public ResponseEntity getComponentChildren(@PathVariable String id) {
        List<FileSystemDesignTimeSerializable> components = DesignTimeSerializableCache.getChildens(id);


        HashMap map = new HashMap();
        if (CollectionUtils.isEmpty(components)) {
            map.put("data", "找不到组件");
        } else {
            map.put("parentId", id);
            map.put("data", components);
        }

        return ResponseEntity.ok(map);
    }

    @GetMapping("/components/{id}/{name}")
    public ResponseEntity getComponentChildrenByName(@PathVariable String id,@PathVariable String name) {
        List<FileSystemDesignTimeSerializable> components =
                DesignTimeSerializableCache.getChildens(id);

        Collection<FileSystemDesignTimeSerializable> selectedBeans
                = components.stream().filter(ii -> ii.getName().equalsIgnoreCase(name))
                .collect(Collectors.toList());

        return ResponseEntity.ok(selectedBeans);
    }

    @GetMapping("/cache-index")
    public ResponseEntity getCacheIndex() {
        Map<String, String> component = DesignTimeSerializableCache.getCacheIndex();

        return ResponseEntity.ok(component);
    }

    @GetMapping("/cache-pid-index")
    public ResponseEntity getCachePidIndex() {
        Map<String, String> component = DesignTimeSerializableCache.getCachePidIndex();

        return ResponseEntity.ok(component);
    }

    @GetMapping("/cache-info")
    public ResponseEntity getCacheInfo() {
        String cacheInfo = DesignTimeSerializableCache.getCacheInfo();

        return ResponseEntity.ok(cacheInfo);
    }

    @GetMapping("/cache-menus")
    public ResponseEntity getMenus() {
        Map<String, List<FileSystemDesignTimeSerializable>> components = DesignTimeSerializableCache.getCachedMenus();

        return ResponseEntity.ok(components);
    }

    @PostMapping("/flush-second-caches")
    public ResponseEntity flush2SecondCache() {
        boolean flag = DesignTimeSerializableCache.flush2SecondCached();

        return ResponseEntity.ok(flag);
    }

    @PostMapping("/disabled-second-caches")
    public ResponseEntity disabled2SecondCache() {
        boolean flag = DesignTimeSerializableCache.disableSecondCache();

        return ResponseEntity.ok(flag);
    }

    @PostMapping("/components/redis/keys")
    public ResponseEntity getRedisKeys() {
        Collection<String> keys = DesignTimeSerializableCache.getAllRedisKeys();

        return ResponseEntity.ok(keys);
    }

    @PostMapping("/javascript-execute/{appId}/unregister")
    public ResponseEntity unregister(@PathVariable String appId) {
        RhinoJavaScriptCompiler rhinoJavaScriptCompiler =
                SpringApplicationContextUtil.getBean(RhinoJavaScriptCompiler.class);
        rhinoJavaScriptCompiler.unRegister(appId);

        Collection<String> registeredKeys =
                rhinoJavaScriptCompiler.getRegisteredKeys();

        return ResponseEntity.ok(registeredKeys);
    }

    @GetMapping("/javascript-execute/{appId}")
    public ResponseEntity getRegistered(@PathVariable String appId) {
        RhinoJavaScriptCompiler rhinoJavaScriptCompiler =
                SpringApplicationContextUtil.getBean(RhinoJavaScriptCompiler.class);

        Collection<String> registeredKeys =
                rhinoJavaScriptCompiler.getRegisteredKeys(appId);

        return ResponseEntity.ok(registeredKeys);
    }

    private Collection<AppRoleResponse> getAppsByDomainId(String domainId) throws Exception {
        DomainProcess domainProcess = AuthTimeServiceManager.domainRuntimeService();
        DomainVO domain = (DomainVO) domainProcess.doView(domainId);
        if (domain == null) {
            throw new SaasNofoundException("找不到组织信息");
        }

        ApplicationDesignTimeService service = DesignTimeServiceManager.applicationDesignTimeService();
        List<Application> list = service.list(null, null);
        Collection<Application> bindApplications = domain.getApplications();

        List<Application> items = new ArrayList<Application>();
        for (Application app : list) {
            //排除系统软件软件
            if (app.getType() == Application.SYSTEM_TYPE) {
                continue;
            }
            // 软件被启用
            if (app.isActivated()) {
                boolean isBind = false;
                for (Application bindApp : bindApplications) {
                    if (app.getId().equals(bindApp.getId())) {
                        isBind = true;
                        break;
                    }
                }
                // 软件没有被企业域绑定
                if (isBind) {
                    items.add(app);
                }
            }
        }

        return getAppRoles(items);
    }

    private Collection<AppRoleResponse> getAppRoles(Collection<Application> myApps) throws Exception {
        RoleDesignTimeService roleProcess = AuthTimeServiceManager.roleRuntimeService();
        return myApps.stream().map(appVo -> {

            try {
                AppRoleResponse appR = AppRoleResponse.create(appVo.getId(), appVo.getName());
                DataPackage<Role> dataPackage = null;
                dataPackage = roleProcess.query(appVo.getId(), null, 1, Integer.MAX_VALUE);

                Collection<AppRoleResponse.RoleItem> roles = dataPackage.getDatas().stream().map(r -> AppRoleResponse.RoleItem.create(r.getId(), r.getName()))
                        .collect(Collectors.toList());

                appR.assignRole(roles);

                return appR;
            } catch (Exception e) {
                throw new SaasBadException("获取应用角色异常");
            }
        }).collect(Collectors.toList());
    }

    /**
     * 批量处理用户角色问题
     * @param domainId
     * @param userVOS
     * @param roleIds
     */
    @Transactional
    protected void doUpdateUserRoles(String domainId, Collection<UserVO> userVOS, Collection<String> roleIds) {
        Collection<String> userIds = userVOS.stream().map(ii -> ii.getId()).filter(ii -> StringUtils.hasLength(ii))
                .distinct().collect(Collectors.toList());
        roleIds = roleIds.stream().filter(ii -> StringUtils.hasLength(ii)).distinct().collect(Collectors.toList());

        Map deleteMap = new HashMap();
        deleteMap.put("id", userIds);
        deleteMap.put("domainId", domainId);

        int affectedCount = jdbcTemplate.update(
                "delete from t_user_department_role_set where USERID in (select t.id from t_user t where t.id in (:id) and t.DOMAINID=:domainId)",
                deleteMap);

        List<Map> parameters = new ArrayList<>();
        for (UserVO userVO : userVOS) {
            for (String roleId : roleIds) {
                Map<String, String> pm = new HashMap<>();

                String selectedDepartId = null;
                if (userVO.getUserDepartmentRoleSets() != null && userVO.getUserDepartmentRoleSets().size() > 0) {
                    selectedDepartId = userVO.getUserDepartmentRoleSets().stream().findFirst().get().getDepartmentId();
                }

                if (!StringUtils.hasLength(selectedDepartId) && userVO.getDepartments() != null && userVO.getDepartments().size() > 0) {
                    selectedDepartId = userVO.getDepartments().stream().findFirst().get().getId();
                }

                if (!StringUtils.hasLength(selectedDepartId)) {
                    selectedDepartId = userVO.getDefaultDepartment();
                }

                pm.put("id", UUID.randomUUID().toString());
                pm.put("departId", selectedDepartId);
                pm.put("userId", userVO.getId());
                pm.put("roleId", roleId);

                parameters.add(pm);
            }
        }

        int[] affectedCounts = jdbcTemplate.batchUpdate(
                "insert into t_user_department_role_set(id,DEPARTMENTID,USERID,ROLEID) values(:id,:departId,:userId,:roleId)",
                parameters.toArray(new Map[parameters.size()])
        );
    }
}
