package com.bcxin.web.commons.utils;

import com.bcxin.saas.core.InjectResolverFactory;
import com.bcxin.saas.core.components.ClientAccessProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.Arrays;
import java.util.Optional;

public class RequestUtil {
    private static Logger logger = LoggerFactory.getLogger(RequestUtil.class);

    public static String extractValue(HttpServletRequest request, String headerName) {
        String accessToken = request.getParameter(headerName);
        if (accessToken == null) {
            Cookie[] cookies = request.getCookies();
            if (cookies != null) {
                for (Cookie cookie : cookies) {
                    String name = cookie.getName();
                    if (headerName.equals(name)) {
                        accessToken = cookie.getValue();
                    }
                }
            }
        }

        if (accessToken == null) {
            accessToken = request.getHeader(headerName);
        }

        return accessToken;
    }

    public static String extractAclAdminAccountId(HttpServletRequest request) {
        return extractAclAccountId(request, "x-g-app-token", "x-g-app-time");
    }

    public static String extractAclAppAccountId(HttpServletRequest request) {
        return extractAclAccountId(request, "x-a-app-token", "x-a-app-time");
    }

    private static String extractAclAccountId(HttpServletRequest request, String token_key, String time_key) {
        String appTime = extractValue(request, time_key);
        if (appTime == null || appTime == "") {
            return null;
        }

        String token = extractValue(request, token_key);

        ClientAccessProvider clientAccessProvider =
                InjectResolverFactory.resolve(ClientAccessProvider.class);

        return clientAccessProvider.getUserId(token);
    }

    public static String getCookieValue(HttpServletRequest request, String key) {
        try {
            Cookie[] cookies = request.getCookies();
            if (cookies == null || cookies.length == 0) {
                return null;
            }

            Optional<String> cookieValue =
                    Arrays.stream(cookies).filter(ii -> ii.getName().equalsIgnoreCase(key))
                            .map(ii -> ii.getValue()).findFirst();
            if (cookieValue.isPresent()) {
                return URLDecoder.decode(cookieValue.get(), "utf-8");
            }
        }
        catch (Exception ex) {
            logger.error("getCookieValue异常", ex);
        }

        return null;
    }

    public static void setCookieValue(HttpServletResponse response, String key, String value) throws UnsupportedEncodingException {
        try {
            Cookie newCookie = new Cookie(key, URLEncoder.encode(value, "utf-8"));
            //1小时
            newCookie.setMaxAge(60 * 60);
            response.addCookie(newCookie);
        } catch (Exception ex) {
            logger.error("setCookieValue异常", ex);
        }
    }

    public static void clearCookie(HttpServletRequest request, HttpServletResponse response, String prefix)
    {
        try {
            Cookie[] cookies = request.getCookies();
            if (cookies == null || cookies.length == 0) {
                return;
            }
            Arrays.stream(cookies).filter(ii -> ii.getName().startsWith(prefix))
                    .forEach(ci -> {
                        Cookie rmCookie = new Cookie(ci.getName(), null);
                        rmCookie.setMaxAge(0);
                        response.addCookie(rmCookie);
                    });
        }
        catch (Exception ex) {
            logger.error("clearCookie异常", ex);
        }
    }
}
