package cn.myapps.designtime.task.controller;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import cn.myapps.common.util.SequenceException;
import cn.myapps.runtime.scheduler.engine.RegularScheduler;
import net.sf.json.JSONArray;
import org.apache.commons.lang.StringUtils;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import cn.myapps.common.data.DataPackage;
import cn.myapps.common.data.ParamsTable;
import cn.myapps.common.exception.OBPMValidateException;
import cn.myapps.common.model.task.Task;
import cn.myapps.common.model.task.TaskConstants;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.task.service.TaskDesignTimeService;
import cn.myapps.runtime.common.service.RunTimeServiceManager;
import cn.myapps.runtime.scheduler.ejb.TriggerProcess;
import cn.myapps.runtime.scheduler.ejb.TriggerVO;
import cn.myapps.runtime.scheduler.engine.job.IscriptTaskJob;
import cn.myapps.util.StringUtil;
import cn.myapps.util.sequence.Sequence;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONObject;

@Api(tags = "任务模块")
@Component
@RequestMapping(path = "/api/designtime/applications", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class TaskController extends AbstractDesignTimeController {

    private TaskDesignTimeService taskDesignTimeService;

    public TaskController() throws Exception {
        taskDesignTimeService = DesignTimeServiceManager.taskDesignTimeService();
    }

    /**
     * 获取任务列表（可根据名字或者备注查询）
     *
     * @param applicationId 软件id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/tasks")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取任务列表", notes = "获取任务列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "pageNo", value = "页码", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "linesPerPage", value = "页条数", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "searchword", value = "名字", required = true, paramType = "query", dataType = "string")
    })
    public Resource getTasksList(@PathVariable String applicationId, String searchword){
        ParamsTable params = getParams();
        String pageNo =  params.getParameterAsString("pageNo");
        String linesPerPage =  params.getParameterAsString("linesPerPage");
        int page = (pageNo != null && pageNo.length() > 0) ? Integer.parseInt(pageNo) : 1;
        int line = (linesPerPage != null && linesPerPage.length() > 0) ? Integer.parseInt(linesPerPage) : 10;
        try {
            DataPackage<Task> data = taskDesignTimeService.queryByNameOrDescript(applicationId, searchword, page, line);

            return success("ok", data);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取任务详情
     *
     * @param taskId 任务id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/tasks/{taskId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取任务详情", notes = "获取任务详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "taskId", value = "任务id", required = true, paramType = "path", dataType = "string")
    })
    public Resource doGetTask(@PathVariable String taskId) throws Exception {
        try {
            Task task = taskDesignTimeService.findById(taskId);
            return success("ok", task);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建任务
     *
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/tasks")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "新建任务", notes = "新建任务")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "rTime", value = "运行时间", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "rDate", value = "运行日期", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCreateTasks(@PathVariable String applicationId, @RequestBody String content,@RequestParam(required = false,defaultValue = "0")int runState,@RequestParam String rTime,@RequestParam String rDate) throws Exception {
        try {

            JSONObject json = JSONObject.fromObject(content);
            Task task = (Task)json2obj(json,Task.class);
            task.setrTime(rTime);
            task.setrDate(rDate);
            this.setDate(task,rTime,rDate);
            task.setApplicationid(applicationId);
            task.setParentId(applicationId);
            JSONArray array = (JSONArray) json.get("daysOfWeek");
            Collection<Integer> dayOfWeek= new ArrayList<>();
            if(!array.isEmpty()){
                for (int i = 0;i<array.size();i++){
                    dayOfWeek.add(array.getInt(i));
                }
            }
            task.setDaysOfWeek(dayOfWeek);
            task.setDayOfMonth(1);
            if(StringUtil.isBlank(task.getId())){
                task.setId(Sequence.getDesignTimeSequence());
            }
            if (task.getName().equals("")) {
                throw new OBPMValidateException(
                        "{*[page.name.notexist]*}");
            }
            //判断名称是否含有特殊字符
            String regEx = "[ \\~\\!\\/\\@\\#\\$\\%\\^\\&\\*\\(\\)\\-\\=\\+\\\\\\|\\[\\{\\}\\]\\;\\:\\\'\\\"\\,\\<\\.\\>\\/\\?]";
            Pattern p = Pattern.compile(regEx);
            Matcher m = p.matcher(task.getName());

            if (m.find()){
                throw new OBPMValidateException("{*[Name]*}{*[can.not.exist.invalidchar]*}");
            }
            task.setModifyTime(new Date());
//            if (task.getStartupType() == TaskConstants.STARTUP_TYPE_BANNED) {
//                runState = this.stopTask(task);
//            }
            if(exist(task)) {
                throw new OBPMValidateException("任务名称已存在!");
            }
            taskDesignTimeService.save(task);
            insertOrUpdate(task);
//            if(task.getStartupType() == TaskConstants.STARTUP_TYPE_AUTO){
//                this.stopTask(task);
//                runState = this.doStart(task.getId(),rTime,rDate);
//            }
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("id", task.getId());
            jsonObject.put("runState",runState);
            return success("ok", jsonObject);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新任务
     *
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/tasks")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "更新任务", notes = "更新任务")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource doUpdateTasks(@RequestBody String content) throws Exception {
        try {
            JSONObject json = JSONObject.fromObject(content);
            String applicationId = (String) json.get("appId");
            Task task = (Task)json2obj(json,Task.class);
//            TriggerProcess triggerProcess = RunTimeServiceManager.triggerProcess(task.getApplicationid());
//            TriggerVO trigger = (TriggerVO)triggerProcess.doView(task.getId());
//            JSONObject jsonObject = new JSONObject();
//            if (trigger!=null) {
//                jsonObject.put("runState",1);
//            }

            JSONArray array = (JSONArray) json.get("daysOfWeek");
            Collection<Integer> dayOfWeek= new ArrayList<>();
            if(!array.isEmpty()){
                for (int i = 0;i<array.size();i++){
                    dayOfWeek.add(array.getInt(i));
                }
            }
            String rTime = json.getString("rTime");
            String rDate = json.getString("rDate");
            if("null".equalsIgnoreCase(rDate)) {
                rDate = null;
            }
            if("null".equalsIgnoreCase(rTime)) {
                rTime = null;
            }
            task.setrTime(rTime);
            task.setrDate(rDate);
            this.setDate(task,rTime,rDate);
            task.setApplicationid(applicationId);
            task.setParentId(applicationId);
            task.setDaysOfWeek(dayOfWeek);

            taskDesignTimeService.saveOrUpdate(task);
            insertOrUpdate(task);
            return success("ok", task);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除任务（可批量）
     *
     * @param ids 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/{applicationId}/tasks")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "删除任务", notes = "删除任务（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doDeleteTasks(@PathVariable String applicationId,@RequestBody String[] ids) throws Exception {
        try {
            TriggerProcess triggerProcess = RunTimeServiceManager.triggerProcess(applicationId);
            for(String id :ids){
                String token = TriggerVO.generateIscriptTaskJobToken(id);
                triggerProcess.removeByToken(token);
            }
            taskDesignTimeService.delete(ids);
            return success("ok", "删除成功");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    public int doStart(String id,String rTime,String rDate) {
        try {
            Task task = taskDesignTimeService.findById(id);
            task.setState(1);
            this.setDate(task,rTime,rDate);
//            this.startTask(task);
            return 1;
        } catch (Exception e) {
            e.printStackTrace();
            return 0;
        }
    }

    /**
     * 运行任务
     *
     * @return
     * @throws Exception
     */
    @PutMapping("/task/start")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "运行任务", notes = "运行任务")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "任务id", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "rTime", value = "运行时间", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "rDate", value = "运行日期", required = true, paramType = "query", dataType = "string"),
    })
    public Resource startTask(String id,String rTime,String rDate) throws Exception {
        try{
            doStart(id,rTime,rDate);
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("runState", 1);
            jsonObject.put("msg","定时任务已经启动");
            return success("ok",jsonObject);
        }catch (Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 停止任务
     *
     * @return
     * @throws Exception
     */
    @PutMapping("/task/stop")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "停止任务", notes = "停止任务")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "任务id", required = true, paramType = "query", dataType = "string"),
    })
    public Resource stopToTask(String id) throws Exception {
        try{
            Task task = taskDesignTimeService.findById(id);
            stopTask(task);
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("runState", 0);
            jsonObject.put("msg","任务已经停止");
            return success("ok",jsonObject);
        }catch (Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 判断是否存在同名的任务
     * @return
     * @throws Exception
     */
    private Boolean exist(Task task) throws Exception {
        //获取Id
        String id = task.getId();

        //列出该应用下所有的任务
        List<Task> tasks =  taskDesignTimeService.list(task.getApplicationid(),null);

        if(StringUtils.isBlank(id)) {
            //新建
            for(Task t :tasks) {
                if(t.getName().equals(task.getName())) {
                    return true;
                }
            }
        }else {
            //编辑
            for(Task t :tasks) {
                if(task.getId().equals(t.getId())) {
                    continue;
                }
                if(t.getName().equals(task.getName())) {
                    return true;
                }
            }
        }

        return false;
    }
    /**
     * 停止任务
     */
    private int stopTask(Task task) {
        int runState = 0;

        try {

//            TriggerProcess triggerProcess = RunTimeServiceManager.triggerProcess(task.getApplicationid());
//            triggerProcess.doRemove(task.getId());
            //终于明白token的作用，同于当不知道trigger的时候作为唯一标识取消任务
            String token = TriggerVO.generateIscriptTaskJobToken(task.getId());
            //在执行器中取消
            // RegularScheduler.cancelTrigger(token,task.getApplicationid());
            task.setState(runState);
            taskDesignTimeService.doCumulativeTask(task);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return runState;
    }

    /**
     * 设置日期
     * @throws Exception
     */
    private void setDate(Task task,String rTime,String rDate) throws Exception {
        Date date = new Date();
        try {
            int period = task.getPeriod();
            if (period == TaskConstants.REPEAT_TYPE_DAILY
                    || period == TaskConstants.REPEAT_TYPE_WEEKLY
                    || period == TaskConstants.REPEAT_TYPE_MONTHLY) {
                // 当为每天，每周，每月的时候。选择运行日期的时候
                SimpleDateFormat formater = new SimpleDateFormat();
                formater.applyPattern("HH:mm:ss");
                if ("".equals(rTime)) {
                    throw new OBPMValidateException("{*[core.task.choosetime]*}");
                }
                String dateStr = rTime;
                date = formater.parse(dateStr);
                Calendar c = Calendar.getInstance();
                c.set(Calendar.HOUR_OF_DAY, date.getHours());
                c.set(Calendar.MINUTE, date.getMinutes());
                c.set(Calendar.SECOND, date.getSeconds());
                date = c.getTime();
            } else if (period == TaskConstants.REPEAT_TYPE_NONE) {
                // 当为不重复的时候。
                if (rDate.equals("")) {
                    throw new OBPMValidateException(
                            "{*[cn.myapps.core.task.select_running_time]*}");
                } else if (rTime.equals("")) {
                    throw new OBPMValidateException("{*[core.task.choosetime]*}");
                }
                String dateStr = rDate + " " + rTime;
                SimpleDateFormat formatter = new SimpleDateFormat(
                        "yyyy-MM-dd HH:mm:ss");
                date = formatter.parse(dateStr);
            } else if (period == TaskConstants.REPEAT_TYPE_DAILY_MINUTES
                    || period == TaskConstants.REPEAT_TYPE_DAILY_HOURS) {
                SimpleDateFormat formatter = new SimpleDateFormat(
                        "yyyy-MM-dd HH:mm:ss");
                date = formatter.parse(formatter.format(date));
            } else {
                // 当为立即的时候
                SimpleDateFormat formatter = new SimpleDateFormat(
                        "yyyy-MM-dd HH:mm:ss");
                date = formatter.parse(formatter.format(date));
            }
        }catch (OBPMValidateException e) {
            e.printStackTrace();
            throw e;
        }catch (Exception e) {

            e.printStackTrace();
            throw e;
        }
        // 把时间放入到运行的那里去
        task.setRunningTime(date);
    }

    /**
     * 插入或者更新trigger
     * @throws Exception
     */
    private void insertOrUpdate(Task task) throws Exception {

        TriggerProcess triggerProcess = RunTimeServiceManager.triggerProcess(task.getApplicationid());
        //这里用task的id是查不到trigger的
        //        TriggerVO trigger = (TriggerVO)triggerProcess.doView(task.getId());
        TriggerVO trigger = (TriggerVO)triggerProcess.selectByToken(TriggerVO.generateIscriptTaskJobToken(task.getId()));

        if(task.getStartupType()!= TaskConstants.STARTUP_TYPE_BANNED && trigger!=null){
            Calendar calendar = Calendar.getInstance();
            IscriptTaskJob job = new IscriptTaskJob(task.getId(),!cn.myapps.common.util.StringUtil.isBlank(task.getApplicationid())?task.getApplicationid():trigger.getApplicationId());
            if(task.getPeriod()== TaskConstants.REPEAT_TYPE_NONE || task.getPeriod()==TaskConstants.REAPET_TYPE_NOTREAPET || task.getPeriod()==TaskConstants.REPEAT_TYPE_IMMEDIATE){
                job.setLoop(false);
            }else{
                job.setLoop(true);
            }
            TriggerVO newTrigger = new TriggerVO(job, calendar.getTimeInMillis());
            trigger.setJobData(newTrigger.getJobData());
            trigger.setDeadline(trigger.getDeadline());
            trigger.setLastModifyDate(task.getModifyTime());
            trigger.setState(TriggerVO.STATE_WAITING);
            triggerProcess.doUpdate(trigger);
        }else{
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(task.getRunningTime());
            IscriptTaskJob job = new IscriptTaskJob(task.getId(),task.getApplicationid());
            if(task.getPeriod()==TaskConstants.REPEAT_TYPE_NONE || task.getPeriod()==TaskConstants.REAPET_TYPE_NOTREAPET || task.getPeriod()==TaskConstants.REPEAT_TYPE_IMMEDIATE){
                job.setLoop(false);
            }else{
                job.setLoop(true);
            }
            trigger = new TriggerVO(job, calendar.getTimeInMillis());
            trigger.setId(Sequence.getSequence());
            trigger.setApplicationId(task.getApplicationid());
            trigger.setLastModifyDate(task.getModifyTime());
            trigger.setRunTimes(0);
            trigger.setTaskId(task.getId());
            if(task.getStartupType()!=TaskConstants.STARTUP_TYPE_AUTO){
                trigger.setState(TriggerVO.STATE_STOP);
            }
            triggerProcess.doCreate(trigger);
        }

    }

}
