package cn.myapps.designtime.statelabel.controller;

import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseStatus;

import cn.myapps.common.data.DataPackage;
import cn.myapps.common.exception.OBPMValidateException;
import cn.myapps.common.model.statelabel.StateLabel;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.statelabel.service.StateLabelService;
import cn.myapps.util.StringUtil;
import cn.myapps.util.sequence.Sequence;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;

@Api(tags = "标签状态模块")
@Component
@RequestMapping(path = "/api/designtime/applications", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class StateLabelController extends AbstractDesignTimeController {

    private StateLabelService stateLabelService;

    public StateLabelController() throws Exception{
        stateLabelService = DesignTimeServiceManager.stateLabelService();
    }

    /**
     * 获取标签状态列表
     *
     * @param applicationId 软件id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/statelabels")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取标签状态列表", notes = "获取标签状态列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "pageNo", value = "页码", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "linesPerPage", value = "页条数", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "searchword", value = "查询名称或者描述", required = true, paramType = "query", dataType = "string")
    })
    public Resource getStateLabelsList(@PathVariable String applicationId, String searchword, String pageNo, String linesPerPage){

        int page = (pageNo != null && pageNo.length() > 0) ? Integer.parseInt(pageNo) : 1;
        int line = (linesPerPage != null && linesPerPage.length() > 0) ? Integer.parseInt(linesPerPage) : 10;
        try {
            DataPackage<StateLabel> data = stateLabelService.queryByNameOrDescript(applicationId, searchword, page, line);
            JSONObject result = new JSONObject();
            JSONArray array = new JSONArray();
            for(StateLabel stateLabel:data.getDatas()){
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("id", stateLabel.getId());
                jsonObject.put("name", stateLabel.getName());
                jsonObject.put("description", stateLabel.getDescription());
                jsonObject.put("value", stateLabel.getValue());
                jsonObject.put("orderNo", stateLabel.getOrderNo());
                array.add(jsonObject);
            }
            result.put("linesPerPage",data.getLinesPerPage());
            result.put("pageCount",data.getPageCount());
            result.put("pageNo",data.getPageNo());
            result.put("rowCount",data.getRowCount());
            result.put("data",array);
            return success("ok", result);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取标签状态详情
     *
     * @param stateLabelId 标签状态id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/statelabels/{stateLabelId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取标签状态详情", notes = "获取标签状态详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "stateLabelId", value = "标签状态id", required = true, paramType = "path", dataType = "string")
    })
    public Resource doGetStateLabel(@PathVariable String stateLabelId) {
        try {
            StateLabel stateLabel = stateLabelService.findById(stateLabelId);
            return success("ok", stateLabel);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建标签状态
     *
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/statelabels")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "新建标签状态", notes = "新建标签状态")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCreateStateLabels(@PathVariable String applicationId, @RequestBody String content){
        try {

            JSONObject json = JSONObject.fromObject(content);
            StateLabel stateLabel = (StateLabel)json2obj(json,StateLabel.class);

            stateLabel.setApplicationid(applicationId);
            stateLabel.setParentId(applicationId);

            if(StringUtil.isBlank(stateLabel.getId())){
                stateLabel.setId(Sequence.getDesignTimeSequence());
            }
            if (stateLabel.getName().equals("")) {
                throw new OBPMValidateException(
                        "{*[page.name.notexist]*}");
            }
            //判断名称是否含有特殊字符
            String regEx = "[ \\~\\!\\/\\@\\#\\$\\%\\^\\&\\*\\(\\)\\-\\=\\+\\\\\\|\\[\\{\\}\\]\\;\\:\\\'\\\"\\,\\<\\.\\>\\/\\?]";
            Pattern p = Pattern.compile(regEx);
            Matcher m = p.matcher(stateLabel.getName());

            if (m.find()){
                throw new OBPMValidateException("{*[Name]*}{*[can.not.exist.invalidchar]*}");
            }
            doSaveValidate(stateLabel);
            stateLabelService.save(stateLabel);

            JSONObject jsonObject = new JSONObject();
            jsonObject.put("id", stateLabel.getId());
            return success("ok", jsonObject);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新标签状态
     *
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/statelabels")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "更新标签状态", notes = "更新标签状态")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource doUpdateStateLabels(@RequestBody String content) {
        try {
            JSONObject json = JSONObject.fromObject(content);
            StateLabel stateLabel = (StateLabel)json2obj(json,StateLabel.class);
            doSaveValidate(stateLabel);
            stateLabelService.saveOrUpdate(stateLabel);
            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除标签状态（可批量）
     *
     * @param ids 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/statelabels")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "删除标签状态", notes = "删除标签状态（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doDeleteStateLabels (@RequestBody String[] ids) {
        try {
            stateLabelService.delete(ids);
            return success("ok", "删除成功");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 保存前的校验
     *
     * @param stateLabel
     * @throws Exception
     */
    private void doSaveValidate(StateLabel stateLabel) throws Exception {
        List<StateLabel> list = stateLabelService.list(stateLabel.getApplicationid(), stateLabel.getName());
        for(StateLabel label:list){
            if(stateLabel.getId().equals(label.getId()) && stateLabel.getName().equals(label.getName())){
                break;
            }
            if(label.getName().equals(stateLabel.getName())){
                throw new OBPMValidateException(" 该名称已存在,请重新命名再保存！ ");
            }
        }
    }
}
