package cn.myapps.designtime.macro.controller;

import cn.myapps.common.data.DataPackage;
import cn.myapps.common.exception.OBPMValidateException;
import cn.myapps.common.model.macrorepository.RepositoryVO;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.macro.repository.service.RepositoryDesignTimeService;
import cn.myapps.util.StringUtil;
import cn.myapps.util.sequence.Sequence;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONObject;
import org.apache.commons.beanutils.BeanUtils;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.*;


@Api(tags = "函数库模块")
@Component
@RequestMapping(path = "/api/designtime/applications", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class MacrosController extends AbstractDesignTimeController {

    private RepositoryDesignTimeService repositoryDesignTimeService;

    public MacrosController() throws Exception {
        repositoryDesignTimeService = DesignTimeServiceManager.repositoryDesignTimeService();
    }

    /**
     * 获取函数库列表（可根据名字查询）
     *
     * @param applicationId 软件id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/macros")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取函数库列表", notes = "获取函数库列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "pageNo", value = "页码", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "linesPerPage", value = "页条数", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "name", value = "函数库名称", required = true, paramType = "path", dataType = "string")})
    public Resource getMacrosList(@PathVariable String applicationId, String name,String pageNo,String linesPerPage){
        int page = (pageNo != null && pageNo.length() > 0) ? Integer.parseInt(pageNo) : 1;
        int line = (linesPerPage != null && linesPerPage.length() > 0) ? Integer.parseInt(linesPerPage) : 10;
        try {
            DataPackage<RepositoryVO>  data = repositoryDesignTimeService.query(applicationId, name, page, line);

            return success("ok", data);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取函数库详情
     *
     * @param macroId 函数库id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/macros/{macroId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取函数库详情", notes = "获取函数库详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "macroId", value = "函数库id", required = true, paramType = "path", dataType = "string")
    })
    public Resource doGetMacro(@PathVariable String applicationId,@PathVariable String macroId) throws Exception {
        try {
            RepositoryVO macro = repositoryDesignTimeService.findById(macroId);
            return success("ok", macro);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建函数库
     *
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/macros")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "新建函数库", notes = "新建函数库")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCreateMacros(@PathVariable String applicationId, @RequestBody String content) throws Exception {
        try {

            JSONObject json = JSONObject.fromObject(content);
            RepositoryVO rv = (RepositoryVO)json2obj(json,RepositoryVO.class);
            rv.setApplicationid(applicationId);
            rv.setParentId(applicationId);

            if(StringUtil.isBlank(rv.getId())){
                rv.setId(Sequence.getDesignTimeSequence());
            }
            doSaveValidate(rv);
            repositoryDesignTimeService.doCreate(rv);

            JSONObject formJson = new JSONObject();
            formJson.put("id", rv.getId());

            return success("ok", formJson);
        } catch (OBPMValidateException ex){
            ex.printStackTrace();
            return error(40001, ex.getValidateMessage(), null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新函数库
     *
     * @param applicationId 软件id
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/{applicationId}/macros/{macroId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "更新函数库", notes = "更新函数库")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "macroId", value = "函数库id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource doUpdateMacros(@PathVariable String applicationId, @PathVariable String macroId, @RequestBody String content) throws Exception {
        try {
            JSONObject json = JSONObject.fromObject(content);
            RepositoryVO rv = (RepositoryVO)json2obj(json,RepositoryVO.class);

            RepositoryVO oldMacro = repositoryDesignTimeService.findById(macroId);
            RepositoryVO macro = (RepositoryVO)BeanUtils.cloneBean(oldMacro);

            macro.setApplicationid(applicationId);
            macro.setContent(rv.getContent());
            macro.setName(rv.getName());
            macro.setDescription(rv.getDescription());
            doSaveValidate(macro);
            repositoryDesignTimeService.doUpdate(macro);

            return success("ok", null);
        } catch (OBPMValidateException ex){
            ex.printStackTrace();
            return error(40001, ex.getValidateMessage(), null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }
    /**
     * 删除函数库（可批量）
     *
     * @param ids 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/{applicationId}/macros")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "删除函数库", notes = "删除函数库（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doDeleteMacros(@RequestBody String[] ids) throws Exception {
        try {
            repositoryDesignTimeService.delete(ids);
            return success("ok", "删除成功");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 保存前的校验
     *
     * @param repositoryVO
     * @throws Exception
     */
    private void doSaveValidate(RepositoryVO repositoryVO) throws Exception {
        RepositoryVO po =  repositoryDesignTimeService.getRepositoryByName(repositoryVO.getName(), repositoryVO.getApplicationid());
        if (po != null && StringUtil.isBlank(repositoryVO.getId())) {
            //新建
            throw new OBPMValidateException(" 库名称重复 ,请重新命名再保存！ ");
        }
        else {
            //更新
            if (po != null && !StringUtil.isBlank(repositoryVO.getId()) && !repositoryVO.getId().equals(po.getId())) {
                throw new OBPMValidateException(" 库名称重复 ,请重新命名再保存！ ");
            }
        }
    }
}
