package cn.myapps.designtime.link.controller;

import cn.myapps.common.data.DataPackage;
import cn.myapps.common.model.link.LinkVO;
import cn.myapps.common.util.StringUtil;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.link.service.LinkDesignTimeService;
import cn.myapps.util.sequence.Sequence;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONObject;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.*;

@Api(tags = "视图链接模块")
@Component
@RequestMapping(path = "/api/designtime/applications", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class LinkController extends AbstractDesignTimeController {

    private LinkDesignTimeService linkDesignTimeService;

    public LinkController() {
        linkDesignTimeService = DesignTimeServiceManager.linkDesignTimeService();
    }

    /**
     * 获取链接列表
     *
     * @param applicationId 软件id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/links")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取标签状态列表", notes = "获取标签状态列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "pageNo", value = "页码", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "linesPerPage", value = "页条数", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "name", value = "查询名称", required = true, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "type", value = "类型", required = true, paramType = "query", dataType = "string")
    })
    public Resource getLinkList(@PathVariable String applicationId, String name, String type, String pageNo, String linesPerPage){

        int page = (pageNo != null && pageNo.length() > 0) ? Integer.parseInt(pageNo) : 1;
        int line = (linesPerPage != null && linesPerPage.length() > 0) ? Integer.parseInt(linesPerPage) : 10;
        try {
            DataPackage<LinkVO> data = linkDesignTimeService.queryByNameAndType(applicationId, name, type, page, line);
            return success("ok", data);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取链接详情
     *
     * @param linkId 链接id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/links/{linkId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取链接详情", notes = "获取链接详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "linkId", value = "链接id", required = true, paramType = "path", dataType = "string")
    })
    public Resource doGetLink(@PathVariable String linkId) {
        try {
            LinkVO linkVO = linkDesignTimeService.findById(linkId);
            return success("ok", linkVO);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建链接
     *
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/links")
    @ApiOperation(value = "新建链接", notes = "新建链接")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCreateLink(@PathVariable String applicationId, @RequestBody String content){
        try {

            JSONObject json = JSONObject.fromObject(content);
            LinkVO linkVO = (LinkVO)json2obj(json,LinkVO.class);

            linkVO.setApplicationid(applicationId);
            linkVO.setParentId(applicationId);

            if(StringUtil.isBlank(linkVO.getId())){
                linkVO.setId(Sequence.getDesignTimeSequence());
            }

            // 链接名称非空校验
            if (StringUtil.isBlank(linkVO.getName())) {
                return error(500, "{*[cn.myapps.km.category.name_not_null]*}",
                        null);
            }

            // 链接名称重名校验
            DataPackage<LinkVO> data = linkDesignTimeService.queryByNameAndType(applicationId, "", "", 1, 10);
            for (LinkVO vo : data.getDatas()) {
                if (linkVO.getName().equals(vo.getName()))
                    return error(500, "{*[core.form.exist]*}",
                            null);
            }

            linkDesignTimeService.save(linkVO);

            JSONObject jsonObject = new JSONObject();
            jsonObject.put("id", linkVO.getId());
            return success("ok", jsonObject);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新链接
     *
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/links")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "更新链接", notes = "更新链接")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource doUpdateLink(@RequestBody String content) {
        try {
            JSONObject json = JSONObject.fromObject(content);
            LinkVO linkVO = (LinkVO)json2obj(json,LinkVO.class);
            linkDesignTimeService.saveOrUpdate(linkVO);
            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除链接（可批量）
     *
     * @param ids 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/links")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "删除链接", notes = "删除链接（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doDeleteLinks (@RequestBody String[] ids) {
        try {
            linkDesignTimeService.delete(ids);
            return success("ok", "删除成功");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }


}
