package cn.myapps.designtime.fckeditor.connector;

import java.io.IOException;
import java.util.List;

import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.fckeditor.connector.exception.FolderAlreadyExistsException;
import net.fckeditor.connector.exception.InvalidCurrentFolderException;
import net.fckeditor.connector.exception.InvalidNewFolderNameException;
import net.fckeditor.connector.exception.ReadException;
import net.fckeditor.connector.exception.WriteException;
import net.fckeditor.handlers.Command;
import net.fckeditor.handlers.PropertiesLoader;
import net.fckeditor.handlers.RequestCycleHandler;
import net.fckeditor.handlers.ResourceType;
import net.fckeditor.requestcycle.Context;
import net.fckeditor.requestcycle.ThreadLocalData;
import net.fckeditor.response.GetResponse;
import net.fckeditor.response.UploadResponse;
import net.fckeditor.tool.Utils;
import net.fckeditor.tool.UtilsFile;
import net.fckeditor.tool.UtilsResponse;

public class Dispatcher {
    private static final Logger logger;
    private Connector connector;

    Dispatcher(ServletContext servletContext) throws Exception {
        String className = "cn.myapps.designtime.fckeditor.connector.ContextConnector";
        if (Utils.isEmpty(className)) {
            logger.error("Empty Connector implementation class name provided");
        } else {
            try {
                Class<?> clazz = Class.forName(className);
                this.connector = (Connector)clazz.newInstance();
            } catch (Throwable var4) {
                logger.error("Connector implementation {} could not be instantiated", className);
                throw new RuntimeException("Connector implementation " + className + " could not be instantiated", var4);
            }
        }

        this.connector.init(servletContext);
    }

    GetResponse doGet(HttpServletRequest request) {
        Context context = ThreadLocalData.getContext();
        context.logBaseParameters();
        GetResponse getResponse = null;
        if (!Command.isValidForGet(context.getCommandStr())) {
            getResponse = GetResponse.getInvalidCommandError();
        } else if (!ResourceType.isValidType(context.getTypeStr())) {
            getResponse = GetResponse.getInvalidResourceTypeError();
        } else if (!UtilsFile.isValidPath(context.getCurrentFolderStr())) {
            getResponse = GetResponse.getInvalidCurrentFolderError();
        } else {
            ResourceType type = context.getResourceType();
            Command command = context.getCommand();
            if ((command.equals(Command.GET_FOLDERS) || command.equals(Command.GET_FOLDERS_AND_FILES)) && !RequestCycleHandler.isGetResourcesEnabled(request)) {
                getResponse = GetResponse.getGetResourcesDisabledError();
            } else if (command.equals(Command.CREATE_FOLDER) && !RequestCycleHandler.isCreateFolderEnabled(request)) {
                getResponse = GetResponse.getCreateFolderDisabledError();
            } else {
                try {
                    String url;
                    if (command.equals(Command.CREATE_FOLDER)) {
                        url = request.getParameter("NewFolderName");
                        String sanitizedNewFolderNameStr = UtilsFile.sanitizeFolderName(url);
                        if (Utils.isEmpty(sanitizedNewFolderNameStr)) {
                            getResponse = GetResponse.getInvalidNewFolderNameError();
                        } else {
                            this.connector.createFolder(type, context.getCurrentFolderStr(), sanitizedNewFolderNameStr);
                            getResponse = GetResponse.getOK();
                        }
                    } else if (command.equals(Command.GET_FOLDERS) || command.equals(Command.GET_FOLDERS_AND_FILES)) {
                        url = UtilsResponse.getUrl(request.getContextPath()+"/uploads/lib", type, context.getCurrentFolderStr());
                        getResponse = this.getFoldersAndOrFiles(command, type, context.getCurrentFolderStr(), url);
                    }
                } catch (InvalidCurrentFolderException var8) {
                    getResponse = GetResponse.getInvalidCurrentFolderError();
                } catch (InvalidNewFolderNameException var9) {
                    getResponse = GetResponse.getInvalidNewFolderNameError();
                } catch (FolderAlreadyExistsException var10) {
                    getResponse = GetResponse.getFolderAlreadyExistsError();
                } catch (WriteException var11) {
                    getResponse = GetResponse.getCreateFolderWriteError();
                } catch (ReadException var12) {
                    getResponse = GetResponse.getGetResourcesReadError();
                }
            }
        }

        return getResponse;
    }

    private GetResponse getFoldersAndOrFiles(Command command, ResourceType type, String currentFolder, String constructedUrl) throws InvalidCurrentFolderException, ReadException {
        GetResponse getResponse = new GetResponse(command, type, currentFolder, constructedUrl);
        getResponse.setFolders(this.connector.getFolders(type, currentFolder));
        if (command.equals(Command.GET_FOLDERS_AND_FILES)) {
            getResponse.setFiles(this.connector.getFiles(type, currentFolder));
        }

        return getResponse;
    }

    UploadResponse doPost(HttpServletRequest request) {
        Context context = ThreadLocalData.getContext();
        context.logBaseParameters();
        UploadResponse uploadResponse = null;
        if (!RequestCycleHandler.isFileUploadEnabled(request)) {
            uploadResponse = UploadResponse.getFileUploadDisabledError();
        } else if (!Command.isValidForPost(context.getCommandStr())) {
            uploadResponse = UploadResponse.getInvalidCommandError();
        } else if (!ResourceType.isValidType(context.getTypeStr())) {
            uploadResponse = UploadResponse.getInvalidResourceTypeError();
        } else if (!UtilsFile.isValidPath(context.getCurrentFolderStr())) {
            uploadResponse = UploadResponse.getInvalidCurrentFolderError();
        } else {
            ResourceType type = context.getDefaultResourceType();
            FileItemFactory factory = new DiskFileItemFactory();
            ServletFileUpload upload = new ServletFileUpload(factory);

            try {
                List<FileItem> items = upload.parseRequest(request);
                FileItem uplFile = (FileItem)items.get(0);
                String fileName = FilenameUtils.getName(uplFile.getName());
                logger.debug("Parameter NewFile: {}", fileName);
                if (type.isDeniedExtension(FilenameUtils.getExtension(fileName))) {
                    uploadResponse = UploadResponse.getInvalidFileTypeError();
                } else if (type.equals(ResourceType.IMAGE) && PropertiesLoader.isSecureImageUploads() && !UtilsFile.isImage(uplFile.getInputStream())) {
                    uploadResponse = UploadResponse.getInvalidFileTypeError();
                } else {
                    String sanitizedFileName = UtilsFile.sanitizeFileName(fileName);
                    logger.debug("Parameter NewFile (sanitized): {}", sanitizedFileName);
                    String newFileName = this.connector.fileUpload(type, context.getCurrentFolderStr(), sanitizedFileName, uplFile.getInputStream());
                    String fileUrl = UtilsResponse.fileUrl(RequestCycleHandler.getUserFilesPath(request), type, context.getCurrentFolderStr(), newFileName);
                    if (sanitizedFileName.equals(newFileName)) {
                        uploadResponse = UploadResponse.getOK(fileUrl);
                    } else {
                        uploadResponse = UploadResponse.getFileRenamedWarning(fileUrl, newFileName);
                        logger.debug("Parameter NewFile (renamed): {}", newFileName);
                    }
                }

                uplFile.delete();
            } catch (InvalidCurrentFolderException var13) {
                uploadResponse = UploadResponse.getInvalidCurrentFolderError();
            } catch (WriteException var14) {
                uploadResponse = UploadResponse.getFileUploadWriteError();
            } catch (IOException var15) {
                uploadResponse = UploadResponse.getFileUploadWriteError();
            } catch (FileUploadException var16) {
                uploadResponse = UploadResponse.getFileUploadWriteError();
            }
        }

        return uploadResponse;
    }

    static {
        logger = LoggerFactory.getLogger(net.fckeditor.connector.Dispatcher.class);
    }
}
