package cn.myapps.designtime.datasource.controller;

import java.sql.Connection;
import java.sql.DriverManager;
import java.util.*;

import cn.myapps.common.exception.OBPMValidateException;
import cn.myapps.common.model.application.Application;
import cn.myapps.common.model.dynaform.dts.metadata.model.ITable;
import cn.myapps.common.model.table.Column;
import cn.myapps.common.model.table.Table;
import cn.myapps.designtime.application.service.ApplicationDesignTimeService;
import cn.myapps.designtime.common.service.DesignTimeServiceFactory;
import cn.myapps.designtime.datasource.model.TableMetadata;
import cn.myapps.designtime.dynaform.dts.metadata.ejb.MetadataProcess;
import cn.myapps.designtime.dynaform.dts.metadata.ejb.MetadataProcessBean;
import cn.myapps.designtime.form.service.FormDesignTimeService;
import cn.myapps.designtime.form.service.FormDesignTimeServiceImpl;
import cn.myapps.runtime.dynaform.form.ejb.Form;
import cn.myapps.runtime.dynaform.form.ejb.FormField;
import cn.myapps.runtime.dynaform.form.ejb.FormTableProcessBean;
import cn.myapps.runtime.dynaform.form.ejb.mapping.TableMapping;
import cn.myapps.runtime.dynaform.view.tree.Node;
import cn.myapps.util.DbTypeUtil;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseStatus;

import cn.myapps.common.data.DataPackage;
import cn.myapps.common.model.datasource.DataSource;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.datasource.service.DataSourceDesignTimeService;
import cn.myapps.util.StringUtil;
import cn.myapps.util.sequence.Sequence;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONObject;

import javax.servlet.http.Cookie;

@Api(tags = "数据源模块")
@Component
@RequestMapping(path = "/api/designtime/applications", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class DataSourceController extends AbstractDesignTimeController {

    private DataSourceDesignTimeService dataSourceDesignTimeService;

    public DataSourceController() throws Exception{
        dataSourceDesignTimeService = DesignTimeServiceManager.dataSourceDesignTimeService();
    }
    /**
     * 获取数据源列表（可根据名字查询）
     *
     * @param applicationId 软件id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/datasources")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取数据源列表", notes = "获取数据源列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "pageNo", value = "页码", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "linesPerPage", value = "页条数", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "name", value = "数据源名称", required = true, paramType = "path", dataType = "string")})
    public Resource getDataSourcesList(@PathVariable String applicationId, String name, String pageNo, String linesPerPage){
        int page = (pageNo != null && pageNo.length() > 0) ? Integer.parseInt(pageNo) : 1;
        int line = (linesPerPage != null && linesPerPage.length() > 0) ? Integer.parseInt(linesPerPage) : 10;
        try {
            DataPackage<DataSource> data = dataSourceDesignTimeService.query(applicationId, name, page, line);

            return success("ok", data);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取数据源详情
     *
     * @param dataSourceId 数据源id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/datasources/{dataSourceId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取数据源详情", notes = "获取数据源详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "dataSourceId", value = "数据源id", required = true, paramType = "path", dataType = "string")
    })
    public Resource doGetDataSource(@PathVariable String applicationId,@PathVariable String dataSourceId) throws Exception {
        try {
            DataSource dataSource = dataSourceDesignTimeService.findById(dataSourceId);
            return success("ok", dataSource);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建数据源
     *
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/datasources")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "新建数据源", notes = "新建数据源")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCreateDataSources(@PathVariable String applicationId, @RequestBody String content) throws Exception {
        try {

            JSONObject jsonObject = JSONObject.fromObject(content);
            DataSource ds = (DataSource)json2obj(jsonObject,DataSource.class);
            if(StringUtil.isBlank(ds.getId())){
                ds.setId(Sequence.getDesignTimeSequence());
            }
            ds.setParentId(applicationId);

            doSaveValidate(ds);
            dataSourceDesignTimeService.save(ds);

            return success("ok", ds);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新数据源
     *
     * @param applicationId 软件id
     * @param dataSourceId 数据源id
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/{applicationId}/datasources/{dataSourceId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "更新数据源", notes = "更新数据源")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "dataSourceId", value = "数据源id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource doUpdateDataSources(@PathVariable String applicationId, @PathVariable String dataSourceId, @RequestBody String content) throws Exception {
        try {
            JSONObject json = JSONObject.fromObject(content);
            DataSource ds = (DataSource)json2obj(json,DataSource.class);
            ds.setId(dataSourceId);
            ds.setParentId(applicationId);

            doSaveValidate(ds);
            dataSourceDesignTimeService.saveOrUpdate(ds);

            return success("ok", ds);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }
    /**
     * 删除数据源（可批量）
     *
     * @param ids 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/{applicationId}/datasources")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "删除数据源", notes = "删除数据源（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doDeleteDataSources(@RequestBody String[] ids) throws Exception {
        try {
            dataSourceDesignTimeService.delete(ids);
            return success("ok", "删除成功");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 测试数据源
     *
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/testdatasources")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "测试数据源", notes = "测试数据源")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource testDataSources(@RequestBody String content) throws Exception {
        JSONObject jsonObject = JSONObject.fromObject(content);

        String username = (String)jsonObject.get("username");
        String password = (String)jsonObject.get("password");
        String driverClass = (String)jsonObject.get("driverClass");
        String url = (String)jsonObject.get("url");
        String msg = "{*[cn.myapps.core.deploy.application.connect_success]*}!";
        try {
            Class.forName(driverClass).newInstance();
            Connection conn = DriverManager.getConnection(url, username,password);
            conn.close();
        } catch (ClassNotFoundException e ) {
            msg = "数据驱动填写错误! \n";
            return error(500, msg, null);
        } catch (Exception e) {
            msg = "{*[cn.myapps.core.deploy.application.connect_error]*}! \n原因:地址或用户名称,密码填写错误!" ;
            return error(500, msg, null);
        }
        return success("ok", msg);
    }

    /**
     * 获取源数据
     *
     * @param applicationId 软件id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/datasources/getMetadata")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取源数据", notes = "获取源数据")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "datasourceId", value = "数据源id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "subNodes", value = "子节点", required = true, paramType = "path", dataType = "string")})
    public Resource doGetMetadata(@PathVariable String applicationId,String datasourceId,String subNodes) throws Exception{

        Collection<Node> childrenNodes = new ArrayList<Node>();

        if (subNodes == null || "".equals(subNodes)) {
            if (!cn.myapps.common.util.StringUtil.isBlank(applicationId)) {
                Collection<DataSource> dts = dataSourceDesignTimeService.list(applicationId, null);
                Iterator<DataSource> it = dts.iterator();
                while (it.hasNext()) {
                    DataSource ds = (DataSource) it.next();
                    ApplicationDesignTimeService appService = (ApplicationDesignTimeService) DesignTimeServiceFactory.resolve(ApplicationDesignTimeService.class);
                    Application app = (Application) appService.findById(applicationId);
                    if (ds != null) {
                        Node node = new Node();
                        node.setId(ds.getId());
                        node.setData(ds.getName());//
                        node.addAttr("name", ds.getName());
                        node.addAttr("appication", ds.getApplicationid());
                        node.addAttr("datasourceId", ds.getId());
                        node.addAttr("curNode", "isDts");
                        if (app != null
                                && app.getDatasourceId().equals(ds.getId())) {
                            node.addAttr("rel", "db_selected");
                        } else {
                            node.addAttr("rel", "db");
                        }
                        node.addAttr("subNodes", "isTables");
                        node.setState(Node.STATE_CLOSED);
                        childrenNodes.add(node);
                    }
                }
            }
        } else if ("isTables".equals(subNodes)) {
            FormDesignTimeService fp = (FormDesignTimeService) DesignTimeServiceFactory.resolve(FormDesignTimeService.class);
            Collection<Form> forms = fp.getFormsByApplication(applicationId);
            // long e1 = System.currentTimeMillis();
            DataSource ds = dataSourceDesignTimeService.findById(datasourceId);
            Collection<Table> tables = DbTypeUtil.getTables(null, ds
                    .getDbTypeName(), ds.getConnection());
            // 指定数据源的所有的表集合
            Collection<String> table_names = new ArrayList<String>();
            Iterator<Table> it_table = tables.iterator();
            // 初始化指定数据源的所有的表
            while (it_table.hasNext()) {
                Table table = it_table.next();
                if (table != null) {
                    table_names.add(table.getName());
                }
            }
            Iterator<Form> it = forms.iterator();
            while (it.hasNext()) {
                Form form = it.next();
                if (form != null) {
                    TableMapping tableMapping = new TableMapping(form);
                    // 过滤掉数据源(指定数据库)不存在的表<表单>
                    if (table_names.contains(tableMapping.getTableName())) {
                        Node node = new Node();
                        node.setId(form.getId());
                        node.setData(tableMapping.getTableName());//
                        node.addAttr("name", tableMapping.getTableName());
                        node.addAttr("formId", form.getId());
                        node.addAttr("datasourceId", datasourceId);
                        node.addAttr("curNode", "isTable");
                        node.addAttr("rel", "table");
                        node.addAttr("subNodes", "none");
                        node.addAttr("appication", form.getApplicationid());
                        node.setState(Node.STATE_CLOSED);
                        childrenNodes.add(node);
                    }
                }
            }
        }
        return success("ok", childrenNodes);
    }


    /**
     * 获取索引信息
     *
     * @param applicationId 软件id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/datasources/getIndex")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取索引信息", notes = "获取索引信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "datasourceId", value = "数据源id", required = true, paramType = "path", dataType = "string")})
    public Resource doGetIndex(@PathVariable String applicationId,String datasourceId) throws Exception{
        DataSource datasource = dataSourceDesignTimeService.findById(datasourceId);
        MetadataProcess process=new MetadataProcessBean();
        Collection<ITable> datas = process.getAllTables(datasource);
        return success("ok", datas);
    }

    @GetMapping("/{applicationid}/datasources/optimization")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "优化索引", notes = "优化索引")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationid",value = "应用id",required = true,paramType = "path",dataType = "string")
    })
    public Resource doIndexOptimization(@PathVariable String applicationid){
        try {
            ApplicationDesignTimeService appservice = (ApplicationDesignTimeService) DesignTimeServiceFactory
                    .resolve(ApplicationDesignTimeService.class);
            Application application = appservice.findById(applicationid);
            DataSource datasource =dataSourceDesignTimeService.findById(application.getDatasourceId());
            MetadataProcess process=new MetadataProcessBean();
            process.doIndexOptimization(datasource);
            return  success("ok","优化完成");
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 获取表字段信息
     *
     * @param applicationId 软件id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/datasources/getTable")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取表字段信息", notes = "获取表字段信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "datasourceId", value = "数据源id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "formId", value = "表单id", required = true, paramType = "path", dataType = "string")})
    public Resource doGetTable(@PathVariable String applicationId,String datasourceId,String formId) throws Exception{
        TableMetadata tableMetadata = new TableMetadata();
        DataSource datasource = dataSourceDesignTimeService.findById(datasourceId);
        MetadataProcess process=new MetadataProcessBean();
        tableMetadata.setDatasource(datasource);
        FormDesignTimeService fp = (FormDesignTimeService) DesignTimeServiceFactory.resolve(FormDesignTimeService.class);
        Form form = fp.findById(formId);
        tableMetadata.setForm(form);
        return success("ok", tableMetadata);
    }


    /**
     *同步表单字段
     *
     * @param applicationId 软件id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/datasources/synFormTable")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "同步表单字段", notes = "同步表单字段")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "datasourceId", value = "数据源id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "formId", value = "表单id", required = true, paramType = "path", dataType = "string")})
    public Resource doSynFormTable(@PathVariable String applicationId,String datasourceId,String formId) throws Exception{
        FormDesignTimeServiceImpl fp = new FormDesignTimeServiceImpl();
        Form form =  fp.findById(formId);
        Collection<Form> forms = new ArrayList<Form>();
        Collection<Form> superior_form = fp.getSuperiors(form);
        forms.add(form);
        forms.addAll(superior_form);
        DataSource ds = dataSourceDesignTimeService.findById(datasourceId);
        Iterator<Form> it_f = forms.iterator();
        while (it_f.hasNext()) {
            Form f = it_f.next();
            if (f != null && ds != null) {
                TableMapping tableMapping = new TableMapping(f);
                Table table = DbTypeUtil.getTable(tableMapping
                        .getTableName(), ds.getDbTypeName(), ds
                        .getConnection());
                Collection<Column> columns = table.getColumns();
                Collection<String> columnNames = new ArrayList<String>();
                Iterator<Column> it = columns.iterator();
                while (it.hasNext()) {
                    Column column = it.next();
                    if (column != null) {
                        columnNames.add(column.getName());
                    }
                }
                Collection<FormField> fields = f.getFields();
                Collection<FormField> fields_old = new ArrayList<FormField>();
                Iterator<FormField> it_field = fields.iterator();
                while (it_field.hasNext()) {
                    FormField field = it_field.next();
                    if (field != null
                            && columnNames.contains(tableMapping
                            .getColumnName(field.getName()))) {
                        fields_old.add(field);
                    }
                }
                Form form_old = (Form) f.clone();
                form_old.removeAllField(form_old.getAllFields());
                form_old.addAllField(fields_old);
                FormTableProcessBean ftp = new FormTableProcessBean(f
                        .getApplicationid());
                ftp.setDatasourceId(datasourceId);
                ftp.synDynaTable(f, form_old);

            }
        }
        return success("ok", "同步完成");
    }

    /**
     * 保存前的校验
     *
     * @param dataSource
     * @throws Exception
     */
    private void doSaveValidate(DataSource dataSource) throws Exception {
        //检验重名
        DataSource dataSource_Exsited = dataSourceDesignTimeService.getDataSourceByName(dataSource.getName(), dataSource.getParentId());

        if (dataSource_Exsited != null && StringUtil.isBlank(dataSource.getId())) {// 判断新建不能重名
            throw new OBPMValidateException("[" + dataSource.getName() + "]{*[core.form.exist]*}");
        } else if (dataSource_Exsited != null
                && !dataSource_Exsited.getId().trim().equalsIgnoreCase(
                dataSource.getId())) {// 修改不能重名
            throw new OBPMValidateException("[" + dataSource.getName() + "]{*[core.form.exist]*}");
        }

    }
}
