package cn.myapps.designtime.application.controller;

import cn.myapps.base.web.WebUser;
import cn.myapps.common.Environment;
import cn.myapps.common.data.DataPackage;
import cn.myapps.common.model.application.Application;
import cn.myapps.common.model.datasource.DataSource;
import cn.myapps.common.model.superuser.SuperUserVO;
import cn.myapps.common.util.Security;
import cn.myapps.common.util.StringUtil;
import cn.myapps.designtime.application.service.ApplicationDesignTimeService;
import cn.myapps.designtime.callback.util.CallBackLog;
import cn.myapps.designtime.callback.util.CallBackLogStorage;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.datasource.service.DataSourceDesignTimeService;
import cn.myapps.designtime.superuser.service.SuperUserDesignTimeService;
import cn.myapps.util.DbTypeUtil;
import cn.myapps.util.sequence.Sequence;
import com.bcxin.saas.core.exceptions.SaasBadException;
import com.bcxin.saas.core.exceptions.SaasNofoundException;
import com.bcxin.saas.core.utils.ExceptionUtils;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.commons.beanutils.BeanUtils;
import org.springframework.context.annotation.Scope;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.*;

import java.io.*;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

/**
 * 软件模块
 */
@Component
@RequestMapping(value = "/api/designtime")
@Scope("prototype")
public class ApplicationDesignTimeController extends AbstractDesignTimeController {

    private ApplicationDesignTimeService appService;

    public ApplicationDesignTimeController() throws Exception{
        appService = DesignTimeServiceManager.applicationDesignTimeService();
    }

    /**
     * 获取软件列表
     * @return
     */
    @GetMapping("/applications")
    @ApiOperation(value = "获取软件列表", notes = "获取软件列表（可根据名字或者描述查询）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "searchword", value = "查询名称", required = false, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "type", value = "类型", required = false, paramType = "query", dataType = "Integer"),
            @ApiImplicitParam(name = "pageNo", value = "第几页", required = false, paramType = "query", dataType = "Integer"),
            @ApiImplicitParam(name = "linesPerPage", value = "每页显示条数", required = false, paramType = "query", dataType = "Integer")
    })
    public Resource getApplications(@RequestParam(required = false) String searchword, @RequestParam(required = false, defaultValue = "0") Integer type, @RequestParam(required = false, defaultValue = "1") Integer pageNo, @RequestParam(required = false, defaultValue = "10") Integer linesPerPage){
        try{
            List<Application> list = new ArrayList<>();
            String userId = Security.getDesignerIdFromToken(request);
            SuperUserDesignTimeService process = DesignTimeServiceManager.superUserDesignTimeService();
            SuperUserVO superUserVO = process.doView(userId);
            //判断当前登录者是否为超级管理员
            if (superUserVO.isSuperAdmin()) {
                list = appService.list("", searchword);
            } else {
                Collection<Application> applications = superUserVO.getApplicationList();
                list = new ArrayList<>(applications);
            }

            for(Iterator<Application> iterator =list.iterator();iterator.hasNext();){
                Application application = iterator.next();
                if(Application.SYSTEM_TYPE == application.getType()){
                    iterator.remove();
                }
            }

            //重新分页
            DataPackage<Application> datas = new DataPackage<Application>();
            datas.setRowCount(list.size());
            datas.setLinesPerPage(linesPerPage);
            datas.setPageNo(pageNo);
            //分页后的数据
            int beginIndex=(pageNo-1)*linesPerPage;
            int endIndex=pageNo*linesPerPage;
            if(beginIndex>=list.size()){
                datas.setDatas(null);
                datas.setRowCount(0);
            }else if(endIndex>=list.size()){
                endIndex=list.size();
            }
            if(list.size()>0){
                datas.setDatas(list.subList(beginIndex, endIndex));
            }
            return success("ok", datas);
        } catch(Exception e) {
            e.printStackTrace();
            throw new SaasBadException(String.format("获取应用列表发生异常:%s", e.getMessage()), e);
        }
    }

    /**
     * 获取软件详情
     * @return
     */
    @GetMapping("/applications/{applicationId}")
    @ApiOperation(value = "获取软件详情", notes = "获取软件详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件Id", required = true, paramType = "path", dataType = "string")
    })
    public Resource getApplication(@PathVariable String applicationId){
        try{
            Application application = appService.findById(applicationId);
            if(application==null) {
                throw new SaasNofoundException(String.format("找不到应用(%s)", applicationId));
            }
            JSONObject data = new JSONObject();
            data.put("id",application.getId());
            data.put("name",application.getName());
            data.put("description",application.getDescription());
            data.put("type",application.getType());
            data.put("activated",application.isActivated());
            data.put("orderNo",application.getOrderNo());
            data.put("uri",application.getUri());
            data.put("datasourceId",application.getDatasourceId());
            data.put("url",application.getUrl());
            data.put("token",application.getToken());
            data.put("encodingAESkey",application.getEncodingAESkey());
            if(application.getDataSourceDefine()== null){
                data.put("datasourceName","");
            }else{
                data.put("datasourceName",application.getDataSourceDefine().getName());
            }
            return success("ok", data);
        } catch(Exception e) {
            e.printStackTrace();
            throw new SaasBadException(String.format("获取应用明细发生异常:%s", e.getMessage()), e);
        }
    }

    /**
     * 新建软件
     * @return
     */
    @PostMapping("/applications")
    @ApiOperation(value = "新建软件", notes = "新建软件")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "软件数据包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource createApplication(@RequestBody String content){
        try{
            JSONObject json = JSONObject.fromObject(content);
            Application application = (Application)json2obj(json, Application.class);

            String validateMessage = validate(application);
            if(!StringUtil.isBlank(validateMessage)){
                return error(40001, validateMessage, null);
            }

            application.setId(Sequence.getDesignTimeSequence());
            appService.save(application);

            WebUser user = getUser();
            if (user.isDeveloper()) {
                setApplications(application, user);
            }

            JSONObject result = new JSONObject();
            result.put("id", application.getId());

            return success("ok", result);
        } catch(Exception e){
            e.printStackTrace();
            throw new SaasBadException(String.format("新增应用发生异常:%s", e.getMessage()), e);
            //return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新软件
     * @return
     */
    @PutMapping("/applications/{applicationId}")
    @ApiOperation(value = "更新软件", notes = "更新软件")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件Id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "软件数据包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource updateApplication(@PathVariable String applicationId, @RequestBody String content){
        try{
            JSONObject json = JSONObject.fromObject(content);
            String name = json.getString("name");
            String description = json.getString("description");
            int type = json.getInt("type");
            boolean activated = json.getBoolean("activated");
            int orderNo = json.getInt("orderNo");
            String datasourceId = json.getString("datasourceId");
            Application app = appService.findById(applicationId);
            Application application = (Application) BeanUtils.cloneBean(app);
            application.setName(name);
            application.setDescription(description);
            application.setType(type);
            application.setActivated(activated);
            application.setOrderNo(orderNo);
            application.setDatasourceId(datasourceId);

            String validateMessage = validate(application);
            if(!StringUtil.isBlank(validateMessage)){
                return error(40001, validateMessage, null);
            }

            appService.update(application);

            WebUser user = getUser();
            if (user.isDeveloper()) {
                setApplications(application, user);
            }

            //更改软件的数据库类型
            DataSourceDesignTimeService dataSourceService = DesignTimeServiceManager.dataSourceDesignTimeService();
            DataSource dataSource = dataSourceService.findById(application.getDatasourceId());
            if(dataSource!=null) {
                DbTypeUtil._dbTypes.put(application.getId(), dataSource.getDbTypeName());
            }
            return success("ok", null);
        } catch(Exception e){
            e.printStackTrace();
            throw new SaasBadException(String.format("更新应用发生异常:%s", e.getMessage()), e);
            //return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除软件(可批量)
     * @return
     */
    @DeleteMapping("/applications")
    @ApiOperation(value = "删除软件(可批量)", notes = "删除软件(可批量)")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource removeApplication(@RequestBody String content){
        try{
            List<String> appIds = cn.myapps.util.json.JsonTmpUtil.fromObject(content);

            for(String applicationId : appIds){
                appService.delete(applicationId);

                SuperUserDesignTimeService sup = DesignTimeServiceManager.superUserDesignTimeService();
                Collection<SuperUserVO> superUsers = sup.getAllSuperUser();

                //删除超级用户绑定的软件
                for (SuperUserVO user : superUsers) {
                    Collection<String> apps = user.getApplications();

                    Iterator<String> it = apps.iterator();
                    while (it.hasNext()) {
                        String id = it.next();
                        if (applicationId.equals(id)) {
                            it.remove();
                            break;
                        }
                    }
                    user.setApplications(apps);
                    sup.doUpdate(user);
                }
            }

            return success("ok", null);
        } catch(Exception e){
            e.printStackTrace();
            throw new SaasBadException(String.format("删除应用发生异常:%s", e.getMessage()), e);
            // return error(500, e.getMessage(), null);
        }
    }

    /**
     * 生成概览
     * @return
     */
    @GetMapping("/applications/{applicationId}/overview")
    @ApiOperation(value = "生成概览", notes = "生成概览")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件Id", required = true, paramType = "path", dataType = "string")
    })
    public ResponseEntity<byte[]> createOverview(@PathVariable String applicationId) {
        HttpHeaders headers = new HttpHeaders();
        try {
            String overViewFile = "appOverview" + System.currentTimeMillis() + ".pdf";
            ApplicationUtil.createOverview(applicationId, overViewFile);

            headers.setContentType(MediaType.APPLICATION_OCTET_STREAM);
            headers.setContentDispositionFormData("attachment", overViewFile);
            return new ResponseEntity<byte[]>(toByteArray(getOverview(overViewFile)),headers, HttpStatus.CREATED);

        } catch (Exception e) {
            e.printStackTrace();
            return new ResponseEntity<byte[]>(null,headers,HttpStatus.CREATED);
        }
    }

    /**
     * 获取平台模式回调日志
     * @return
     */
    @GetMapping("/applications/{applicationId}/getCallBackLogs")
    @ApiOperation(value = "获取平台模式回调日志", notes = "获取平台模式回调日志")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件Id", required = true, paramType = "path", dataType = "string")
    })
    public Resource getCallBackLogs(@PathVariable String applicationId){
        try {
            Collection<CallBackLog> callBackLogs = CallBackLogStorage.getAll(applicationId);
            return success("ok", callBackLogs);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 设置当前登陆用户的软件
     * @param application
     * @param user
     */
    private void setApplications(Application application, WebUser user) throws Exception{
    	SuperUserDesignTimeService process = DesignTimeServiceManager.superUserDesignTimeService();
        SuperUserVO spu = process.doView(user.getId());

        Collection<String> apps = spu.getApplications();

        if(apps == null) {
            apps = new ArrayList<String>();
            spu.setApplications(apps);
        }
        apps.add(application.getId());
        process.doUpdate(spu);
    }

    /**
     * 获取OverWrite预览文件InputStream
     * @return
     */
    private InputStream getOverview(String overViewFile) throws Exception{
        // 获得文件保存的真实路径
        final String filePath = Environment.getInstance().getRealPath("/uploads")+"/overview/";
        //获取文件输入流
        InputStream is = new FileInputStream(filePath+overViewFile);
        if (filePath != null) {
            try {
                return is;
            } catch (Exception e) {
                e.printStackTrace();
                throw e;
            } finally {
                new Thread(new Runnable() {
                    public void run() {
                        File file = new File(filePath+overViewFile);
                        while (file.exists()) {
                            if (file.delete()) {
                                break;
                            }
                        }
                    }
                }).start();
            }
        }
        return null;
    }

    /**
     * byte数组转InputStream
     * @param input
     * @return
     * @throws IOException
     */
    private byte[] toByteArray(InputStream input) throws IOException {
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        byte[] buffer = new byte[4096];
        int n = 0;
        while (-1 != (n = input.read(buffer))) {
            output.write(buffer, 0, n);
        }
        return output.toByteArray();
    }

    /**
     * 校验方法
     * @param applicaiton
     * @return
     * @throws Exception
     */
    private String validate(Application applicaiton) throws Exception {
        String name = applicaiton.getName();
        String id = applicaiton.getId();
        if(StringUtil.isBlank(name)){
            return "软件名称不能为空!";
        }
        List<Application> list = appService.list(null, name);

        if(list != null && !list.isEmpty()){
            if(id == null || "".equals(id)) {
                //新建
                for(Application app : list){
                    if(app.getName().equals(name)){
                        return "该软件名称已存在!";
                    }
                }
            }else{
                //更新
                for(Application app : list){
                    if(app.getId().equals(id)) {
                        continue;
                    }
                    if(app.getName().equals(name)){
                        return "该软件名称已存在!";
                    }
                }
            }
        }
        return null;
    }
}
