package cn.myapps.designtime.multilanguage.controller;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.*;

import cn.myapps.common.model.multilanguage.MultiLanguage;
import cn.myapps.common.util.StringUtil;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.multilanguage.service.MultiLanguageService;
import cn.myapps.util.property.MultiLanguageProperty;
import cn.myapps.util.sequence.Sequence;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONObject;

@Api(tags = "多语言模块")
@Component
@RequestMapping(path = "/api/designtime/applications", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class MulitLanguageController extends AbstractDesignTimeController {

    private MultiLanguageService multiLanguageService;

    public MulitLanguageController() throws Exception {
        multiLanguageService = DesignTimeServiceManager.multiLanguageService();
    }

    /**
     * 获取多语言列表（可根据标签查询）
     *
     * @param applicationId 软件id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/mulitlangs")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取多语言列表", notes = "获取多语言列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "pageNo", value = "页码", required = false, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "linesPerPage", value = "页条数", required = false, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "label", value = "标签", required = false, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "text", value = "文本", required = false, paramType = "query", dataType = "string"),
            @ApiImplicitParam(name = "type", value = "类型", required = false, paramType = "query", dataType = "int")
    })
    public Resource getMulitlangsList(@PathVariable String applicationId,String pageNo,String linesPerPage, String label, String text, @RequestParam(required = false,defaultValue = "0") int type){
        int page = (pageNo != null && pageNo.length() > 0) ? Integer.parseInt(pageNo) : 1;
        int line = (linesPerPage != null && linesPerPage.length() > 0) ? Integer.parseInt(linesPerPage) : 10;
        List<MultiLanguage> multilanguageList=null;
        try {
            //查询所有的多语言列表
            multilanguageList = multiLanguageService.list(applicationId, "");
            Iterator multilanguageListTemp = multilanguageList.iterator();
            while(multilanguageListTemp.hasNext()){
                MultiLanguage multiLanguage=(MultiLanguage) multilanguageListTemp.next();
                //根据text查询
                if(!StringUtil.isBlank(text)){
                    if(!multiLanguage.getText().contains(text)){
                        multilanguageListTemp.remove();
                        continue;
                    }
                }
                if(!StringUtil.isBlank(label)) {
                    //根据label查询
                    if(!multiLanguage.getLabel().contains(label)) {
                        multilanguageListTemp.remove();
                        continue;
                    }
                }
                //根据type查询
                if(type!=0){
                    if(multiLanguage.getType()!=type){
                        multilanguageListTemp.remove();
                    }
                }
            }
            //分页后的数据
            int beginIndex=(page-1)*line;
            int endIndex=page*line;
            int rowCount = multilanguageList.size();
            if(beginIndex>=multilanguageList.size()){
                multilanguageList = null;
                rowCount = 0;
            }else if(endIndex>=multilanguageList.size()){
                endIndex=multilanguageList.size();
            }

            //获取分页后数据
            if(multilanguageList != null && multilanguageList.size()>0){
                multilanguageList = multilanguageList.subList(beginIndex, endIndex);
            }

            Map result = new HashMap();

            result.put("linesPerPage",line);
            result.put("pageCount",(int) Math.ceil((double) rowCount / (double) line));
            result.put("pageNo",page);
            result.put("rowCount",rowCount);
            result.put("data",multilanguageList);
            return success("ok", result);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取多语言详情
     *
     * @param mulitlangId 多语言id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/mulitlangs/{mulitlangId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取多语言详情", notes = "获取多语言详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "mulitlangId", value = "多语言id", required = true, paramType = "path", dataType = "string")
    })
    public Resource doGetMulitlang(@PathVariable String mulitlangId) {
        try {
            MultiLanguage multiLanguage = multiLanguageService.findById(mulitlangId);
            return success("ok", multiLanguage);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建多语言
     * @return
     */
    @PostMapping("/{applicationId}/mulitlangs")
    @ApiOperation(value = "新建多语言", notes = "新建多语言")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件Id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "数据包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource createMulitlang(@PathVariable String applicationId, @RequestBody String content){
        try{
            JSONObject json = JSONObject.fromObject(content);
            MultiLanguage multiLanguage = (MultiLanguage)json2obj(json, MultiLanguage.class);

            multiLanguage.setParentId(applicationId);
            multiLanguage.setApplicationid(applicationId);
            multiLanguage.setName(multiLanguage.getText());

            if(cn.myapps.util.StringUtil.isBlank(multiLanguage.getId())){
                multiLanguage.setId(Sequence.getDesignTimeSequence());
            }

            multiLanguageService.save(multiLanguage);
            MultiLanguageProperty.putLanguage(applicationId, multiLanguage);
            JSONObject result = new JSONObject();
            result.put("id", multiLanguage.getId());

            return success("ok", result);
        } catch(Exception e){
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新多语言
     *
     * @param content 请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/mulitlangs")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "更新函数库", notes = "更新函数库")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")
    })
    public Resource doUpdateMulitlangs(@RequestBody String content) {
        try {
            JSONObject json = JSONObject.fromObject(content);
            MultiLanguage multiLanguage = (MultiLanguage)json2obj(json,MultiLanguage.class);
            multiLanguage.setName(multiLanguage.getText());

            multiLanguageService.saveOrUpdate(multiLanguage);
            MultiLanguageProperty.putLanguage(multiLanguage.getApplicationid(), multiLanguage);
            return success("ok", null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除多语言（可批量）
     *
     * @param ids 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/mulitlangs")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "删除多语言", notes = "删除多语言（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doDeleteMulitlangs (@RequestBody String[] ids) {
        try {
            multiLanguageService.delete(ids);
            return success("ok", "删除成功");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 重新加载语言属性
     *
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/reloadLanguage")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "重新加载语言属性", notes = "重新加载语言属性")
    public Resource reloadLanguage() {
        try {
            MultiLanguageProperty.init();
            return success("ok", "加载成功");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }
}
