package cn.myapps.designtime.eip.controller;

import cn.myapps.common.data.DataPackage;
import cn.myapps.common.exception.OBPMValidateException;

import cn.myapps.common.model.eip.EipVO;
import cn.myapps.designtime.common.controller.AbstractDesignTimeController;
import cn.myapps.designtime.common.controller.Resource;
import cn.myapps.designtime.common.service.DesignTimeServiceManager;
import cn.myapps.designtime.eip.service.EipDesignTimeService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import net.sf.json.JSONObject;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.*;

import java.util.Iterator;
import java.util.List;

@Api(tags = "eip模块")
@Component
@RequestMapping(path = "/api/designtime/applications", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
public class EipController extends AbstractDesignTimeController {

    private EipDesignTimeService eipDesignTimeService;

    public EipController() throws Exception {
        eipDesignTimeService = DesignTimeServiceManager.eipDesignTimeService();
    }

    /**
     * 获取eip列表（可根据名字查询）
     *
     * @param applicationId 软件id
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/eips")
    @ApiOperation(value = "获取函数库列表", notes = "获取函数库列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "pageNo", value = "页码", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "linesPerPage", value = "页条数", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "name", value = "名称", required = true, paramType = "path", dataType = "string")})
    public Resource getEipsList(@PathVariable String applicationId, String name, String pageNo, String linesPerPage){
        int page = (pageNo != null && pageNo.length() > 0) ? Integer.parseInt(pageNo) : 1;
        int line = (linesPerPage != null && linesPerPage.length() > 0) ? Integer.parseInt(linesPerPage) : 10;
        try {
            DataPackage<EipVO> data = eipDesignTimeService.query(applicationId, name, page, line);
            return success("ok", data);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 获取eip详情
     *
     * @param eipId eipid
     * @return
     * @throws Exception
     */
    @GetMapping(path = "/{applicationId}/eips/{eipId}")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "获取eip详情", notes = "获取eip详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "eipId", value = "eipid", required = true, paramType = "path", dataType = "string")
    })
    public Resource doGetEip(@PathVariable String applicationId,@PathVariable String eipId) throws Exception {
        try {
            EipVO eipVO = eipDesignTimeService.findById(eipId);
            return success("ok", eipVO);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 新建eip
     *
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PostMapping(path = "/{applicationId}/eips")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "新建eip", notes = "新建eip")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doCreateEips(@PathVariable String applicationId, @RequestBody String content) throws Exception {
        try {

            JSONObject json = JSONObject.fromObject(content);
            EipVO eipVO = (EipVO)json2obj(json,EipVO.class);
            eipVO.setApplicationid(applicationId);
            eipVO.setParentId(applicationId);

            doSaveValidate(eipVO);
            eipDesignTimeService.saveOrUpdate(eipVO);

            JSONObject eipJson = new JSONObject();
            eipJson.put("id", eipVO.getId());

            return success("ok", eipJson);
        } catch (OBPMValidateException ex){
            ex.printStackTrace();
            return error(40001, ex.getValidateMessage(), null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 更新eip
     *
     * @param applicationId 软件id
     * @param content  请求包体
     * @return
     * @throws Exception
     */
    @PutMapping(path = "/{applicationId}/eips/{eipId}")
    @ResponseStatus(HttpStatus.CREATED)
    @ApiOperation(value = "更新eip", notes = "更新eip")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "applicationId", value = "软件id", required = true, paramType = "path", dataType = "string"),
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doUpdateEips(@PathVariable String applicationId, @PathVariable String eipId, @RequestBody String content) throws Exception {
        try {

            JSONObject json = JSONObject.fromObject(content);
            EipVO rv = (EipVO)json2obj(json,EipVO.class);

            EipVO eipVO = eipDesignTimeService.findById(eipId);

            EipVO eip =(EipVO) BeanUtils.cloneBean(eipVO);
            eip.setApplicationid(applicationId);
            eip.setXml(rv.getXml());
            eip.setName(rv.getName());
            eip.setDescription(rv.getDescription());
            doSaveValidate(eip);
            eipDesignTimeService.saveOrUpdate(eip);

            return success("ok", null);
        } catch (OBPMValidateException ex){
            ex.printStackTrace();
            return error(40001, ex.getValidateMessage(), null);
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 删除eip（可批量）
     *
     * @param ids 请求包体
     * @return
     * @throws Exception
     */
    @DeleteMapping(path = "/eips")
    @ResponseStatus(HttpStatus.OK)
    @ApiOperation(value = "删除eip", notes = "删除eip（可批量）")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "请求包体", required = true, paramType = "body", dataType = "string")})
    public Resource doDeleteEips(@RequestBody String[] ids) throws Exception {
        try {
            eipDesignTimeService.delete(ids);
            return success("ok", "删除成功");
        } catch (Exception e) {
            e.printStackTrace();
            return error(500, e.getMessage(), null);
        }
    }

    /**
     * 保存前的校验
     *
     * @param eipVO
     * @throws Exception
     */
    private void doSaveValidate(EipVO eipVO) throws Exception {
        List<EipVO> list = eipDesignTimeService.list(eipVO.getParentId(),eipVO.getName());
        if(StringUtils.isBlank(eipVO.getId())) {
            //新建
            for (Iterator<EipVO> iterator = list.iterator(); iterator.hasNext();) {
                EipVO r = iterator.next();
                if(r.getName().equals(eipVO.getName())){
                    throw new OBPMValidateException("名称已经存在！");
                }
            }
        }else {
            //更新
            for (Iterator<EipVO> iterator = list.iterator(); iterator.hasNext();) {
                EipVO r = iterator.next();
                if(r.getId().equals(eipVO.getId())) {
                    continue;
                }
                if(r.getName().equals(eipVO.getName())){
                    throw new OBPMValidateException("名称已经存在！");
                }
            }
        }
        for (Iterator<EipVO> iterator = list.iterator(); iterator.hasNext();) {
            EipVO r = iterator.next();
            if(r.getName().equals(eipVO.getName()) && !r.getId().equals(eipVO.getId())){
                throw new OBPMValidateException( "名称已经存在！");
            }
        }
    }
}
